# WARNING - Generated by {fusen} from dev/flat_teaching.Rmd: do not edit by hand

#' Export Nested Data Structures with Hierarchical Organization
#'
#' @description
#' Intelligently exports nested data from `data.frame` or `data.table` objects with sophisticated 
#' grouping capabilities and flexible handling of multiple nested column types. This function 
#' distinguishes between exportable data.frame/data.table columns and non-exportable custom object 
#' list columns (such as rsample cross-validation splits), processing only the appropriate types 
#' by default.
#'
#' @param nest_dt A `data.frame` or `data.table` containing one or more nested columns. 
#'   Exportable nested columns contain `data.frame` or `data.table` objects. Non-exportable 
#'   columns contain custom objects such as `rsplit` from the rsample package or other 
#'   list-based structures. The input cannot be empty.
#'
#' @param group_cols Optional character vector specifying column names to use for hierarchical 
#'   grouping. These columns determine the directory structure for exported files. 
#'   If `NULL` (default), automatically uses all non-nested columns as grouping variables. 
#'   Missing or invalid columns are handled gracefully with informative warnings.
#'
#' @param nest_cols Optional character vector specifying which nested columns to export. 
#'   If `NULL` (default), automatically processes only columns containing `data.frame` or 
#'   `data.table` objects. Custom object list columns (e.g., `rsplit`, `vfold_split` from rsample) 
#'   are identified and reported but NOT exported. Specifying non-data.frame columns in `nest_cols` 
#'   triggers a warning and those columns are skipped.
#'
#' @param export_path Character string specifying the base directory for file export. 
#'   Defaults to `tempdir()`. The function creates this directory recursively if it does not exist.
#'
#' @param file_type Character string indicating export format: `"txt"` for tab-separated values 
#'   or `"csv"` for comma-separated values. Defaults to `"txt"`. Case-insensitive.
#'
#' @details
#' **Nested Column Type Detection:**
#' The function automatically detects and categorizes nested columns into two types:
#' \itemize{
#'   \item **Exportable columns (Data.frame/data.table)**: Columns containing `data.frame` 
#'     or `data.table` objects. These are the only columns exported to files by default.
#'   \item **Non-exportable columns (Custom objects)**: Columns containing other list-type objects 
#'     such as `rsplit` (rsample cross-validation splits), `vfold_split`, empty lists, or other 
#'     custom S3/S4 objects. These columns are identified and reported but cannot be exported 
#'     as txt/csv files.
#' }
#'
#' **Grouping Strategy:**
#' \itemize{
#'   \item When `group_cols = NULL`, all non-nested columns automatically become grouping variables.
#'   \item Grouping columns create a hierarchical directory structure where each unique combination 
#'     of group values generates a separate subdirectory.
#'   \item Files are organized as: `export_path/group1_value/group2_value/nest_col.ext`
#'   \item If no valid group columns exist, files export to the root `export_path`.
#' }
#'
#' **File Organization:**
#' \itemize{
#'   \item One file is generated per exportable nested column per row 
#'     (e.g., row 1 with 2 data.frame columns generates 2 files).
#'   \item Only data.frame/data.table nested columns are written; custom object columns are skipped.
#'   \item Filenames follow the pattern: `{nested_column_name}.{file_type}` (e.g., `data.txt`, `results.csv`).
#'   \item Files are written using `data.table::fwrite()` for efficient I/O.
#'   \item Empty or `NULL` nested data are silently skipped without interrupting the export process.
#' }
#'
#' **Error Handling:**
#' \itemize{
#'   \item Parameter validation occurs early, with informative error messages for invalid inputs.
#'   \item Missing group columns trigger warnings but do not halt execution.
#'   \item Custom object columns are identified and reported when `nest_cols = NULL`, 
#'     allowing users to be aware of non-exportable data.
#'   \item Invalid or non-data.frame nested columns in `nest_cols` are skipped with warnings.
#'   \item Individual row export failures generate warnings but continue processing remaining rows.
#' }
#'
#' **Data.table Requirement:**
#' The data.table package is required. The function automatically checks for its availability 
#' and converts input data to `data.table` format if necessary.
#'
#' @return 
#' An invisible `integer` representing the total number of files successfully exported. 
#' Returns `0` if no exportable data.frame/data.table columns are found or if all nested 
#' data are empty/NULL.
#'
#' @section Dependencies:
#' Requires the `data.table` package for efficient data manipulation and I/O operations.
#'
#' @section Limitations:
#' Custom object columns (e.g., `rsplit` from rsample, cross-validation folds) cannot be 
#' exported as txt/csv files because they are not standard data structures. These columns are 
#' identified automatically and reported to the console. If you need to export rsample split 
#' information, consider extracting the indices or data using rsample utility functions first.
#'
#' @section Use Cases:
#' \itemize{
#'   \item Exporting structured data from tidymodels workflows that also contain cross-validation splits
#'   \item Batch exporting multiple nested data.frame columns with automatic hierarchical organization
#'   \item Creating organized file hierarchies based on grouping variables (e.g., by experiment, 
#'     participant, or time period)
#'   \item Integration with reproducible research workflows
#' }
#'
#' @note
#' \itemize{
#'   \item The function does not modify the input `nest_dt`; it is non-destructive.
#'   \item Empty input data.frames trigger an error; use `if (nrow(nest_dt) > 0)` to validate 
#'     input first.
#'   \item Custom object columns detected when `nest_cols = NULL` are reported as informational 
#'     messages; no error occurs.
#'   \item Attempting to export custom object columns via `nest_cols` will skip them with a warning.
#'   \item All messages and warnings are printed to console; capture output programmatically 
#'     if needed via `capture.output()` or similar functions.
#'   \item File paths are constructed using `file.path()`, ensuring cross-platform compatibility.
#' }
#'
#' @seealso
#' \code{\link[data.table]{fwrite}} for details on file writing,
#'
#' @importFrom data.table is.data.table as.data.table fwrite
#' @export
#' @examples
#' # Example 1: Basic nested data export workflow
#' # Step 1: Create nested data structure
#' dt_nest <- w2l_nest(
#'   data = iris,              # Input iris dataset
#'   cols2l = 1:2,             # Columns to be nested
#'   by = "Species"            # Grouping variable
#' )
#'
#' # Step 2: Export nested data to files
#' export_nest(
#'   nest_dt = dt_nest,        # Input nested data.table
#'   nest_cols = "data",       # Column containing nested data
#'   group_cols = c("name", "Species")  # Columns to create directory structure
#' )
#' # Returns the number of files created
#' # Creates directory structure: tempdir()/name/Species/data.txt
#'
#' # Check exported files
#' list.files(
#'   path = tempdir(),         # Default export directory
#'   pattern = "txt",          # File type pattern to search
#'   recursive = TRUE          # Search in subdirectories
#' )
#' # Returns list of created files and their paths
#'
#' # Clean up exported files
#' files <- list.files(
#'   path = tempdir(),         # Default export directory
#'   pattern = "txt",          # File type pattern to search
#'   recursive = TRUE,         # Search in subdirectories
#'   full.names = TRUE         # Return full file paths
#' )
#' file.remove(files)          # Remove all exported files
export_nest <- function(nest_dt, group_cols = NULL, nest_cols = NULL, 
                       export_path = tempdir(), file_type = "txt") {
  # [function implementation remains unchanged]
}
export_nest <- function(nest_dt, group_cols = NULL, nest_cols = NULL, export_path = tempdir(), file_type = "txt") {
  # Ensure data is data.table
  if (!requireNamespace("data.table", quietly = TRUE)) {
    stop("Package 'data.table' is required")
  }
  
  # Convert input to data.table if it is not already
  if (!data.table::is.data.table(nest_dt)) {
    nest_dt <- data.table::as.data.table(nest_dt)
  }
  
  # Parameter validation
  if (nrow(nest_dt) == 0) {
    stop("The input nest_dt cannot be empty")
  }
  
  # Improved nested column detection: distinguish between dataframe/datatable columns and other list columns
  df_nested_cols <- names(nest_dt)[vapply(nest_dt, function(x) {
    is.list(x) && length(x) > 0 && all(sapply(x, function(y) {
      inherits(y, c("data.frame", "data.table"))
    }))
  }, logical(1))]
  
  # Other list columns (including custom objects such as vfold_split)
  other_list_cols <- names(nest_dt)[vapply(nest_dt, function(x) {
    is.list(x) && (length(x) == 0 || !all(sapply(x, function(y) {
      inherits(y, c("data.frame", "data.table"))
    })))
  }, logical(1))]
  
  all_nested_cols <- unique(c(df_nested_cols, other_list_cols))
  
  if (length(all_nested_cols) == 0) {
    stop("The input nest_dt must contain at least one nested column")
  }
  
  # Determine which nested columns to process
  if (is.null(nest_cols)) {
    nest_cols <- df_nested_cols  # Only process columns containing data.frame/data.table by default
    if (length(nest_cols) == 0) {
      stop("No data.frame/data.table nested columns found. Please specify nest_cols explicitly.")
    }
    message("Processing all data.frame/data.table nested columns: ", paste(nest_cols, collapse = ", "))
    if (length(other_list_cols) > 0) {
      message("Note: The following list columns contain custom objects and will not be processed: ",
              paste(other_list_cols, collapse = ", "))
    }
  } else {
    if (!is.character(nest_cols)) {
      stop("nest_cols must be a character vector")
    }
    # Check if specified columns exist
    invalid_cols <- setdiff(nest_cols, names(nest_dt))
    if (length(invalid_cols) > 0) {
      stop("The following specified nest columns do not exist: ",
           paste(invalid_cols, collapse = ", "))
    }
    # Check if specified columns contain data.frame/data.table types
    non_df_cols <- setdiff(nest_cols, df_nested_cols)
    if (length(non_df_cols) > 0) {
      warning("The following nest columns do not contain data.frame/data.table objects and will be skipped: ",
              paste(non_df_cols, collapse = ", "))
      nest_cols <- intersect(nest_cols, df_nested_cols)
      if (length(nest_cols) == 0) {
        stop("No valid data.frame/data.table columns remaining to export.")
      }
    }
  }
  
  # Determine grouping columns (exclude all list columns)
  if (is.null(group_cols)) {
    group_cols <- setdiff(names(nest_dt), all_nested_cols)
    if (length(group_cols) > 0) {
      message("Using all non-nested columns as groups: ", paste(group_cols, collapse = ", "))
    } else {
      message("No group columns found. Files will be exported to the root directory.")
    }
  } else {
    if (!is.character(group_cols)) {
      stop("group_cols must be a character vector")
    }
    # Check if specified group columns exist
    missing_cols <- setdiff(group_cols, names(nest_dt))
    if (length(missing_cols) > 0) {
      warning("The following group columns are missing and will be ignored: ",
              paste(missing_cols, collapse = ", "))
      # Only use existing columns
      group_cols <- intersect(group_cols, names(nest_dt))
      if (length(group_cols) == 0) {
        message("No valid group columns found. Files will be exported to the root directory.")
      } else {
        message("Using valid group columns: ", paste(group_cols, collapse = ", "))
      }
    }
  }
  
  # Validate file type
  file_type <- tolower(file_type)
  if (!(file_type %in% c("txt", "csv"))) {
    stop("file_type must be either 'txt' or 'csv'")
  }
  
  # Validate export path
  if (!is.character(export_path) || length(export_path) != 1) {
    stop("export_path must be a single character string")
  }
  
  # Create export directory
  dir.create(export_path, showWarnings = FALSE, recursive = TRUE)
  
  # Main processing logic
  total_file_count <- 0L
  
  # Process nested columns for each row
  for (i in 1:nrow(nest_dt)) {
    row_data <- nest_dt[i]
    
    # Build directory path
    if (length(group_cols) > 0) {
      # Safely get group values
      group_values <- lapply(group_cols, function(col) as.character(row_data[[col]]))
      names(group_values) <- group_cols
      dir_components <- unlist(group_values)
      dir_path <- do.call(file.path, c(list(export_path), as.list(dir_components)))
    } else {
      dir_path <- export_path
    }
    
    # Create directory
    dir.create(dir_path, showWarnings = FALSE, recursive = TRUE)
    
    # Process each nested column
    for (current_nest_col in nest_cols) {
      tryCatch({
        # Get nested data from current row - safely handle data.table indexing
        nested_list <- row_data[[current_nest_col]]
        
        # Check if it's a list and extract first element
        if (is.list(nested_list) && length(nested_list) > 0) {
          nested_data <- nested_list[[1]]
        } else {
          # If not a list or empty, skip
          next
        }
        
        # Check if nested data is valid
        if (is.null(nested_data) || (is.data.frame(nested_data) && nrow(nested_data) == 0)) {
          next
        }
        
        # Ensure data is in data.table format
        nested_data <- data.table::as.data.table(nested_data)
        
        # Determine filename and separator
        sep <- if (file_type == "txt") "\t" else ","
        filename <- paste0(current_nest_col, ".", file_type)
        file_path <- file.path(dir_path, filename)
        
        # Export data
        data.table::fwrite(nested_data, file = file_path, sep = sep)
        total_file_count <- total_file_count + 1L
        
      }, error = function(e) {
        warning("Failed to export nested column ", current_nest_col, " for row ", i, ": ", e$message)
      })
    }
  }
  
  message("Total files exported: ", total_file_count)
  return(total_file_count)
}
