/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h" 
#include "H5Tconv.h"   
#include "H5Tconv_macros.h"
#include "H5Tconv_complex.h"
#include "H5Tconv_integer.h"
#include "H5Tconv_float.h"

herr_t
H5T__conv_f_f(const H5T_t *src_p, const H5T_t *dst_p, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
              size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
              void H5_ATTR_UNUSED *bkg)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    switch (cdata->command) {
        case H5T_CONV_INIT: {
            H5T_atomic_t src_atomic; 
            H5T_atomic_t dst_atomic; 

            if (NULL == src_p || NULL == dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            src_atomic = src_p->shared->u.atomic;
            dst_atomic = dst_p->shared->u.atomic;
            if (H5T_ORDER_LE != src_atomic.order && H5T_ORDER_BE != src_atomic.order &&
                H5T_ORDER_VAX != src_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unsupported byte order");
            if (H5T_ORDER_LE != dst_atomic.order && H5T_ORDER_BE != dst_atomic.order &&
                H5T_ORDER_VAX != dst_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unsupported byte order");
            if (dst_p->shared->size > TEMP_FLOAT_CONV_BUFFER_SIZE)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "destination size is too large");
            if (8 * sizeof(int64_t) - 1 < src_atomic.u.f.esize ||
                8 * sizeof(int64_t) - 1 < dst_atomic.u.f.esize)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "exponent field is too large");
            cdata->need_bkg = H5T_BKG_NO;

            break;
        }

        case H5T_CONV_FREE:
            break;

        case H5T_CONV_CONV:
            if (NULL == src_p || NULL == dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (NULL == conv_ctx)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid datatype conversion context pointer");
            if (H5T__conv_f_f_loop(src_p, dst_p, conv_ctx, nelmts, buf_stride, buf) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "unable to convert data values");
            break;

        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown conversion command");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__conv_f_f_loop(const H5T_t *src_p, const H5T_t *dst_p, const H5T_conv_ctx_t *conv_ctx, size_t nelmts,
                   size_t buf_stride, void *buf)
{
    H5T_atomic_t src_atomic;                        
    H5T_atomic_t dst_atomic;                        
    hssize_t     expo_max;                          
    ssize_t      src_delta, dst_delta;              
    uint8_t     *s, *sp, *d, *dp;                   
    uint8_t     *src_rev = NULL;                    
    uint8_t      dbuf[TEMP_FLOAT_CONV_BUFFER_SIZE]; 
    size_t       olap;                              
    int          direction;                         
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(src_p);
    assert(src_p->shared->type == H5T_FLOAT || src_p->shared->type == H5T_COMPLEX);
    assert(dst_p);
    assert(dst_p->shared->type == H5T_FLOAT || dst_p->shared->type == H5T_COMPLEX);
    assert(conv_ctx);
    assert(buf);

    if (src_p->shared->type == H5T_COMPLEX)
        src_atomic = src_p->shared->parent->shared->u.atomic;
    else
        src_atomic = src_p->shared->u.atomic;
    if (dst_p->shared->type == H5T_COMPLEX)
        dst_atomic = dst_p->shared->parent->shared->u.atomic;
    else
        dst_atomic = dst_p->shared->u.atomic;

    expo_max = ((hssize_t)1 << dst_atomic.u.f.esize) - 1;

#ifndef NDEBUG
    
    if ((src_p->shared->type == H5T_COMPLEX && dst_p->shared->type == H5T_FLOAT) ||
        (src_p->shared->type == H5T_FLOAT && dst_p->shared->type == H5T_COMPLEX)) {
        const H5T_t *src_base = (src_p->shared->type == H5T_FLOAT) ? src_p : src_p->shared->parent;
        const H5T_t *dst_base = (dst_p->shared->type == H5T_FLOAT) ? dst_p : dst_p->shared->parent;
        assert(0 != (H5T_cmp(src_base, dst_base, false)));
    }
#endif

    
    if (src_p->shared->size == dst_p->shared->size || buf_stride) {
        sp = dp   = (uint8_t *)buf;
        direction = 1;
        olap      = nelmts;
    }
    else if (src_p->shared->size >= dst_p->shared->size) {
        double olap_d =
            ceil((double)(dst_p->shared->size) / (double)(src_p->shared->size - dst_p->shared->size));
        olap = (size_t)olap_d;
        sp = dp   = (uint8_t *)buf;
        direction = 1;
    }
    else {
        double olap_d =
            ceil((double)(src_p->shared->size) / (double)(dst_p->shared->size - src_p->shared->size));
        olap      = (size_t)olap_d;
        sp        = (uint8_t *)buf + (nelmts - 1) * src_p->shared->size;
        dp        = (uint8_t *)buf + (nelmts - 1) * dst_p->shared->size;
        direction = -1;
    }

    
    H5_CHECK_OVERFLOW(buf_stride, size_t, ssize_t);
    H5_CHECK_OVERFLOW(src_p->shared->size, size_t, ssize_t);
    H5_CHECK_OVERFLOW(dst_p->shared->size, size_t, ssize_t);
    src_delta = (ssize_t)direction * (ssize_t)(buf_stride ? buf_stride : src_p->shared->size);
    dst_delta = (ssize_t)direction * (ssize_t)(buf_stride ? buf_stride : dst_p->shared->size);

    
    if (conv_ctx->u.conv.cb_struct.func)
        if (NULL == (src_rev = H5MM_calloc(src_p->shared->size)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "couldn't allocate temporary buffer");

    
    for (size_t elmtno = 0; elmtno < nelmts; elmtno++) {
        H5T_conv_float_specval_t specval_type; 
        H5T_conv_ret_t except_ret = H5T_CONV_UNHANDLED; 
        ssize_t        bitno      = 0;                  
        int64_t        expo;                            
        size_t         implied;                         
        size_t         mpos;                            
        size_t         msize = 0;                       
        size_t         mrsh;                            
        bool           reverse      = true;             
        bool           denormalized = false;            
        bool           carry        = false;            

        
        s = sp;
        if (direction > 0)
            d = elmtno < olap ? dbuf : dp;
        else
            d = elmtno + olap >= nelmts ? dbuf : dp;
        if (d == dbuf)
            memset(dbuf, 0, sizeof(dbuf));

#ifndef NDEBUG
        if (d == dbuf) {
            assert((dp >= sp && dp < sp + src_p->shared->size) ||
                   (sp >= dp && sp < dp + dst_p->shared->size));
        }
        else {
            assert((dp < sp && dp + dst_p->shared->size <= sp) ||
                   (sp < dp && sp + src_p->shared->size <= dp));
        }
#endif

        
        if (H5T_ORDER_BE == src_atomic.order) {
            size_t half_size = src_p->shared->size / 2;

            if (H5T_FLOAT == src_p->shared->type) {
                for (size_t j = 0; j < half_size; j++)
                    H5_SWAP_BYTES(s, j, src_p->shared->size - (j + 1));
            }
            else {
                uint8_t *cur_part = s;
                
                for (size_t j = 0; j < half_size / 2; j++)
                    H5_SWAP_BYTES(cur_part, j, half_size - (j + 1));
                
                cur_part += half_size;
                for (size_t j = 0; j < half_size / 2; j++)
                    H5_SWAP_BYTES(cur_part, j, half_size - (j + 1));
            }
        }
        else if (H5T_ORDER_VAX == src_atomic.order) {
            if (H5T_FLOAT == src_p->shared->type) {
                uint8_t tmp1, tmp2;
                size_t  tsize = src_p->shared->size;
                assert(0 == tsize % 2);

                for (size_t i = 0; i < tsize; i += 4) {
                    tmp1 = s[i];
                    tmp2 = s[i + 1];

                    s[i]     = s[(tsize - 2) - i];
                    s[i + 1] = s[(tsize - 1) - i];

                    s[(tsize - 2) - i] = tmp1;
                    s[(tsize - 1) - i] = tmp2;
                }
            }
            else
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "VAX byte ordering is unsupported for complex number type conversions");
        }

        
        specval_type = H5T__conv_float_find_special(s, &src_atomic, NULL);
        if (specval_type == H5T_CONV_FLOAT_SPECVAL_POSZERO ||
            specval_type == H5T_CONV_FLOAT_SPECVAL_NEGZERO) {
            H5T__bit_copy(d, dst_atomic.u.f.sign, s, src_atomic.u.f.sign, (size_t)1);
            H5T__bit_set(d, dst_atomic.u.f.epos, dst_atomic.u.f.esize, false);
            H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
            goto padding;
        }
        else if (specval_type != H5T_CONV_FLOAT_SPECVAL_REGULAR) {
            
            if (conv_ctx->u.conv.cb_struct.func) {
                H5T_conv_except_t except_type; 

                
                H5T__reverse_order(src_rev, s, src_p);

                if (specval_type == H5T_CONV_FLOAT_SPECVAL_POSINF)
                    except_type = H5T_CONV_EXCEPT_PINF;
                else if (specval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF)
                    except_type = H5T_CONV_EXCEPT_NINF;
                else
                    except_type = H5T_CONV_EXCEPT_NAN;

                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        except_ret = (conv_ctx->u.conv.cb_struct.func)(
                            except_type, conv_ctx->u.conv.src_type_id, conv_ctx->u.conv.dst_type_id, src_rev,
                            d, conv_ctx->u.conv.cb_struct.user_data);
                    }
                H5_AFTER_USER_CB(FAIL)
            }

            if (except_ret == H5T_CONV_UNHANDLED) {
                H5T__bit_copy(d, dst_atomic.u.f.sign, s, src_atomic.u.f.sign, (size_t)1);
                H5T__bit_set(d, dst_atomic.u.f.epos, dst_atomic.u.f.esize, true);
                if (specval_type == H5T_CONV_FLOAT_SPECVAL_NAN)
                    
                    H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, true);
                else {
                    
                    H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
                    
                    if (H5T_NORM_NONE == dst_atomic.u.f.norm)
                        H5T__bit_set(d, dst_atomic.u.f.mpos + dst_atomic.u.f.msize - 1, (size_t)1, true);
                }
            }
            else if (except_ret == H5T_CONV_HANDLED) {
                
                reverse = false;
                goto next;
            }
            else if (except_ret == H5T_CONV_ABORT)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");

            goto padding;
            
        }

        
        expo = (int64_t)H5T__bit_get_d(s, src_atomic.u.f.epos, src_atomic.u.f.esize);

        if (expo == 0)
            denormalized = true;

        
        implied = 1;
        mpos    = src_atomic.u.f.mpos;
        mrsh    = 0;
        if (0 == expo || H5T_NORM_NONE == src_atomic.u.f.norm) {
            if ((bitno = H5T__bit_find(s, src_atomic.u.f.mpos, src_atomic.u.f.msize, H5T_BIT_MSB, true)) >
                0) {
                msize = (size_t)bitno;
            }
            else if (0 == bitno) {
                msize = 1;
                H5T__bit_set(s, src_atomic.u.f.mpos, (size_t)1, false);
            }
        }
        else if (H5T_NORM_IMPLIED == src_atomic.u.f.norm) {
            msize = src_atomic.u.f.msize;
        }
        else {
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "normalization method not implemented yet");
        }

        
        H5T__bit_copy(d, dst_atomic.u.f.sign, s, src_atomic.u.f.sign, (size_t)1);

        
        if (0 == expo || H5T_NORM_NONE == src_atomic.u.f.norm) {
            assert(bitno >= 0);
            expo -= (int64_t)((src_atomic.u.f.ebias - 1) + (src_atomic.u.f.msize - (size_t)bitno));
        }
        else if (H5T_NORM_IMPLIED == src_atomic.u.f.norm) {
            expo -= (int64_t)src_atomic.u.f.ebias;
        }
        else {
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "normalization method not implemented yet");
        }

        
        if (H5T_NORM_NONE == dst_atomic.u.f.norm)
            mrsh++;

        
        expo += (int64_t)dst_atomic.u.f.ebias;

        if (expo < -(hssize_t)(dst_atomic.u.f.msize)) {
            
            expo = 0;
            H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
            msize = 0;
        }
        else if (expo <= 0) {
            
            mrsh += (size_t)(1 - expo);
            expo         = 0;
            denormalized = true;
        }
        else if (expo >= expo_max) {
            
            if (conv_ctx->u.conv.cb_struct.func) { 
                
                H5T__reverse_order(src_rev, s, src_p);

                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        except_ret = (conv_ctx->u.conv.cb_struct.func)(
                            H5T_CONV_EXCEPT_RANGE_HI, conv_ctx->u.conv.src_type_id,
                            conv_ctx->u.conv.dst_type_id, src_rev, d, conv_ctx->u.conv.cb_struct.user_data);
                    }
                H5_AFTER_USER_CB(FAIL)
            }

            if (except_ret == H5T_CONV_UNHANDLED) {
                expo = expo_max;
                H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
                msize = 0;
            }
            else if (except_ret == H5T_CONV_HANDLED) {
                reverse = false;
                goto next;
            }
            else if (except_ret == H5T_CONV_ABORT)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
        }

        
        if (msize > 0 && mrsh <= dst_atomic.u.f.msize && mrsh + msize > dst_atomic.u.f.msize) {
            bitno = (ssize_t)(mrsh + msize - dst_atomic.u.f.msize);
            assert(bitno >= 0 && (size_t)bitno <= msize);
            
            if (H5T__bit_get_d(s, (mpos + (size_t)bitno) - 1, (size_t)1) && !denormalized) {
                
                if ((H5T__bit_find(s, mpos + (size_t)bitno, msize - (size_t)bitno, H5T_BIT_LSB, false) >= 0 ||
                     expo < expo_max - 1)) {
                    carry = H5T__bit_inc(s, mpos + (size_t)bitno - 1, 1 + msize - (size_t)bitno);
                    if (carry)
                        implied = 2;
                }
            }
            else if (H5T__bit_get_d(s, (mpos + (size_t)bitno) - 1, (size_t)1) && denormalized)
                
                H5T__bit_inc(s, mpos + (size_t)bitno - 1, 1 + msize - (size_t)bitno);
        }
        else
            carry = false;

        
        if (mrsh > dst_atomic.u.f.msize + 1) {
            H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
        }
        else if (mrsh == dst_atomic.u.f.msize + 1) {
            H5T__bit_set(d, dst_atomic.u.f.mpos + 1, dst_atomic.u.f.msize - 1, false);
            H5T__bit_set(d, dst_atomic.u.f.mpos, (size_t)1, true);
        }
        else if (mrsh == dst_atomic.u.f.msize) {
            H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
            H5T__bit_set_d(d, dst_atomic.u.f.mpos, MIN(2, dst_atomic.u.f.msize), (hsize_t)implied);
        }
        else {
            if (mrsh > 0) {
                H5T__bit_set(d, dst_atomic.u.f.mpos + dst_atomic.u.f.msize - mrsh, mrsh, false);
                H5T__bit_set_d(d, dst_atomic.u.f.mpos + dst_atomic.u.f.msize - mrsh, (size_t)2,
                               (hsize_t)implied);
            }
            if (mrsh + msize >= dst_atomic.u.f.msize) {
                H5T__bit_copy(d, dst_atomic.u.f.mpos, s, (mpos + msize + mrsh - dst_atomic.u.f.msize),
                              dst_atomic.u.f.msize - mrsh);
            }
            else {
                H5T__bit_copy(d, dst_atomic.u.f.mpos + dst_atomic.u.f.msize - (mrsh + msize), s, mpos, msize);
                H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize - (mrsh + msize), false);
            }
        }

        
        if (carry) {
            expo++;
            if (expo >= expo_max) {
                
                if (conv_ctx->u.conv.cb_struct.func) { 
                    
                    H5T__reverse_order(src_rev, s, src_p);

                    
                    H5_BEFORE_USER_CB(FAIL)
                        {
                            except_ret = (conv_ctx->u.conv.cb_struct.func)(
                                H5T_CONV_EXCEPT_RANGE_HI, conv_ctx->u.conv.src_type_id,
                                conv_ctx->u.conv.dst_type_id, src_rev, d,
                                conv_ctx->u.conv.cb_struct.user_data);
                        }
                    H5_AFTER_USER_CB(FAIL)
                }

                if (except_ret == H5T_CONV_UNHANDLED) {
                    expo = expo_max;
                    H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
                }
                else if (except_ret == H5T_CONV_HANDLED) {
                    reverse = false;
                    goto next;
                }
                else if (except_ret == H5T_CONV_ABORT)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
            }
        }

        carry = false;

        H5_CHECK_OVERFLOW(expo, hssize_t, hsize_t);
        H5T__bit_set_d(d, dst_atomic.u.f.epos, dst_atomic.u.f.esize, (hsize_t)expo);

padding:
        
        if (dst_atomic.offset > 0) {
            assert(H5T_PAD_ZERO == dst_atomic.lsb_pad || H5T_PAD_ONE == dst_atomic.lsb_pad);
            H5T__bit_set(d, (size_t)0, dst_atomic.offset, (bool)(H5T_PAD_ONE == dst_atomic.lsb_pad));
        }
        {
            size_t type_size;

            if (dst_p->shared->type == H5T_FLOAT)
                type_size = dst_p->shared->size;
            else
                type_size = dst_p->shared->parent->shared->size;

            if (dst_atomic.offset + dst_atomic.prec != 8 * type_size) {
                assert(H5T_PAD_ZERO == dst_atomic.msb_pad || H5T_PAD_ONE == dst_atomic.msb_pad);
                H5T__bit_set(d, dst_atomic.offset + dst_atomic.prec,
                             8 * type_size - (dst_atomic.offset + dst_atomic.prec),
                             (bool)(H5T_PAD_ONE == dst_atomic.msb_pad));
            }
        }

        
        if (H5T_ORDER_BE == dst_atomic.order && reverse) {
            size_t half_size = dst_p->shared->size / 2;

            if (H5T_FLOAT == dst_p->shared->type) {
                for (size_t j = 0; j < half_size; j++)
                    H5_SWAP_BYTES(d, j, dst_p->shared->size - (j + 1));
            }
            else {
                for (size_t j = 0; j < half_size / 2; j++)
                    H5_SWAP_BYTES(d, j, half_size - (j + 1));
            }
        }
        else if (H5T_ORDER_VAX == dst_atomic.order && reverse) {
            if (H5T_FLOAT == dst_p->shared->type) {
                uint8_t tmp1, tmp2;
                size_t  tsize = dst_p->shared->size / 2;
                assert(0 == tsize % 2);

                for (size_t i = 0; i < tsize; i += 4) {
                    tmp1 = d[i];
                    tmp2 = d[i + 1];

                    d[i]     = d[(tsize - 2) - i];
                    d[i + 1] = d[(tsize - 1) - i];

                    d[(tsize - 2) - i] = tmp1;
                    d[(tsize - 1) - i] = tmp2;
                }
            }
            else
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "VAX byte ordering is unsupported for complex number type conversions");
        }

next:
        
        if (d == dbuf) {
            if (H5T_FLOAT == dst_p->shared->type)
                H5MM_memcpy(dp, d, dst_p->shared->size);
            else
                H5MM_memcpy(dp, d, dst_p->shared->size / 2);
        }

        
        if (H5T_COMPLEX == dst_p->shared->type)
            memset(dp + (dst_p->shared->size / 2), 0, dst_p->shared->size / 2);

        
        sp += src_delta;
        dp += dst_delta;
    } 

done:
    H5MM_free(src_rev);

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_conv_float_specval_t
H5T__conv_float_find_special(const uint8_t *src_buf, const H5T_atomic_t *src_atomic, uint64_t *sign_out)
{
    uint64_t                 sign; 
    H5T_conv_float_specval_t ret_value = H5T_CONV_FLOAT_SPECVAL_REGULAR;

    FUNC_ENTER_PACKAGE_NOERR

    assert(src_buf);
    assert(src_atomic);

    
    sign = H5T__bit_get_d(src_buf, src_atomic->u.f.sign, (size_t)1);

    
    if (H5T__bit_find(src_buf, src_atomic->u.f.mpos, src_atomic->u.f.msize, H5T_BIT_LSB, true) < 0) {
        
        if (H5T__bit_find(src_buf, src_atomic->u.f.epos, src_atomic->u.f.esize, H5T_BIT_LSB, true) < 0)
            
            ret_value = sign ? H5T_CONV_FLOAT_SPECVAL_NEGZERO : H5T_CONV_FLOAT_SPECVAL_POSZERO;
        
        else if (H5T__bit_find(src_buf, src_atomic->u.f.epos, src_atomic->u.f.esize, H5T_BIT_LSB, false) < 0)
            
            ret_value = sign ? H5T_CONV_FLOAT_SPECVAL_NEGINF : H5T_CONV_FLOAT_SPECVAL_POSINF;
    }
    else {
        bool exp_all_ones =
            (H5T__bit_find(src_buf, src_atomic->u.f.epos, src_atomic->u.f.esize, H5T_BIT_LSB, false) < 0);

        
        if (H5T_NORM_NONE == src_atomic->u.f.norm && exp_all_ones &&
            H5T__bit_find(src_buf, src_atomic->u.f.mpos, src_atomic->u.f.msize - 1, H5T_BIT_LSB, true) < 0)
            ret_value = sign ? H5T_CONV_FLOAT_SPECVAL_NEGINF : H5T_CONV_FLOAT_SPECVAL_POSINF;
        else if (exp_all_ones)
            ret_value = H5T_CONV_FLOAT_SPECVAL_NAN;
    }

    if (sign_out)
        *sign_out = sign;

    FUNC_LEAVE_NOAPI(ret_value);
} 

herr_t
H5T__conv_f_i(const H5T_t *src_p, const H5T_t *dst_p, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
              size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
              void H5_ATTR_UNUSED *bkg)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    switch (cdata->command) {
        case H5T_CONV_INIT: {
            H5T_atomic_t src_atomic; 
            H5T_atomic_t dst_atomic; 

            if (NULL == src_p || NULL == dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            src_atomic = src_p->shared->u.atomic;
            dst_atomic = dst_p->shared->u.atomic;
            if (H5T_ORDER_LE != src_atomic.order && H5T_ORDER_BE != src_atomic.order &&
                H5T_ORDER_VAX != src_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unsupported byte order");
            if (H5T_ORDER_LE != dst_atomic.order && H5T_ORDER_BE != dst_atomic.order &&
                H5T_ORDER_VAX != dst_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unsupported byte order");
            if (dst_p->shared->size > TEMP_INT_CONV_BUFFER_SIZE)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "destination size is too large");
            if (8 * sizeof(hssize_t) - 1 < src_atomic.u.f.esize)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "exponent field is too large");
            cdata->need_bkg = H5T_BKG_NO;

            break;
        }

        case H5T_CONV_FREE:
            break;

        case H5T_CONV_CONV:
            if (NULL == src_p || NULL == dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (NULL == conv_ctx)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid datatype conversion context pointer");

            if (H5T__conv_f_i_loop(src_p, dst_p, conv_ctx, nelmts, buf_stride, buf) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "unable to convert data values");
            break;

        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown conversion command");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__conv_f_i_loop(const H5T_t *src_p, const H5T_t *dst_p, const H5T_conv_ctx_t *conv_ctx, size_t nelmts,
                   size_t buf_stride, void *buf)
{
    H5T_atomic_t src_atomic;                      
    H5T_atomic_t dst_atomic;                      
    ssize_t      src_delta, dst_delta;            
    uint8_t     *s, *sp, *d, *dp;                 
    uint8_t     *int_buf = NULL;                  
    uint8_t     *src_rev = NULL;                  
    uint8_t      dbuf[TEMP_INT_CONV_BUFFER_SIZE]; 
    size_t       int_buf_size;                    
    size_t       src_base_size;                   
    size_t       olap;                            
    int          direction;                       
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(src_p);
    assert(src_p->shared->type == H5T_FLOAT || src_p->shared->type == H5T_COMPLEX);
    assert(dst_p);
    assert(dst_p->shared->type == H5T_INTEGER);
    assert(conv_ctx);
    assert(buf);

    if (src_p->shared->type == H5T_COMPLEX)
        src_atomic = src_p->shared->parent->shared->u.atomic;
    else
        src_atomic = src_p->shared->u.atomic;
    dst_atomic = dst_p->shared->u.atomic;

    
    if (src_p->shared->size == dst_p->shared->size || buf_stride) {
        sp = dp   = (uint8_t *)buf;
        direction = 1;
        olap      = nelmts;
    }
    else if (src_p->shared->size >= dst_p->shared->size) {
        double olap_d =
            ceil((double)(dst_p->shared->size) / (double)(src_p->shared->size - dst_p->shared->size));
        olap = (size_t)olap_d;
        sp = dp   = (uint8_t *)buf;
        direction = 1;
    }
    else {
        double olap_d =
            ceil((double)(src_p->shared->size) / (double)(dst_p->shared->size - src_p->shared->size));
        olap      = (size_t)olap_d;
        sp        = (uint8_t *)buf + (nelmts - 1) * src_p->shared->size;
        dp        = (uint8_t *)buf + (nelmts - 1) * dst_p->shared->size;
        direction = -1;
    }

    
    H5_CHECK_OVERFLOW(buf_stride, size_t, ssize_t);
    H5_CHECK_OVERFLOW(src_p->shared->size, size_t, ssize_t);
    H5_CHECK_OVERFLOW(dst_p->shared->size, size_t, ssize_t);
    src_delta = (ssize_t)direction * (ssize_t)(buf_stride ? buf_stride : src_p->shared->size);
    dst_delta = (ssize_t)direction * (ssize_t)(buf_stride ? buf_stride : dst_p->shared->size);

    
    src_base_size =
        (H5T_FLOAT == src_p->shared->type) ? src_p->shared->size : src_p->shared->parent->shared->size;
    if (dst_atomic.prec / 8 > src_base_size)
        int_buf_size = (dst_atomic.prec + 7) / 8;
    else
        int_buf_size = src_base_size;
    if (NULL == (int_buf = H5MM_calloc(int_buf_size)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "couldn't allocate temporary buffer");

    
    if (conv_ctx->u.conv.cb_struct.func)
        if (NULL == (src_rev = H5MM_calloc(src_p->shared->size)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "couldn't allocate temporary buffer");

    
    for (size_t elmtno = 0; elmtno < nelmts; elmtno++) {
        H5T_conv_float_specval_t specval_type; 
        H5T_conv_ret_t except_ret = H5T_CONV_UNHANDLED; 
        uint64_t       sign;              
        hssize_t       expo;              
        hssize_t       shift_val;         
        ssize_t        msb_pos_s;         
        ssize_t        new_msb_pos;       
        bool           truncated = false; 
        bool           reverse   = true;  

        
        s = sp;
        if (direction > 0)
            d = elmtno < olap ? dbuf : dp;
        else
            d = elmtno + olap >= nelmts ? dbuf : dp;
        if (d == dbuf)
            memset(dbuf, 0, sizeof(dbuf));

#ifndef NDEBUG
        if (d == dbuf) {
            assert((dp >= sp && dp < sp + src_p->shared->size) ||
                   (sp >= dp && sp < dp + dst_p->shared->size));
        }
        else {
            assert((dp < sp && dp + dst_p->shared->size <= sp) ||
                   (sp < dp && sp + src_p->shared->size <= dp));
        }
#endif

        
        if (H5T_ORDER_BE == src_atomic.order) {
            size_t half_size = src_p->shared->size / 2;

            if (H5T_FLOAT == src_p->shared->type) {
                for (size_t i = 0; i < half_size; i++)
                    H5_SWAP_BYTES(s, i, src_p->shared->size - (i + 1));
            }
            else {
                uint8_t *cur_part = s;
                
                for (size_t i = 0; i < half_size / 2; i++)
                    H5_SWAP_BYTES(cur_part, i, half_size - (i + 1));
                
                cur_part += half_size;
                for (size_t i = 0; i < half_size / 2; i++)
                    H5_SWAP_BYTES(cur_part, i, half_size - (i + 1));
            }
        }
        else if (H5T_ORDER_VAX == src_atomic.order) {
            if (H5T_FLOAT == src_p->shared->type) {
                uint8_t tmp1, tmp2;
                size_t  tsize = src_p->shared->size;
                assert(0 == tsize % 2);

                for (size_t i = 0; i < tsize; i += 4) {
                    tmp1 = s[i];
                    tmp2 = s[i + 1];

                    s[i]     = s[(tsize - 2) - i];
                    s[i + 1] = s[(tsize - 1) - i];

                    s[(tsize - 2) - i] = tmp1;
                    s[(tsize - 1) - i] = tmp2;
                }
            }
            else
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "VAX byte ordering is unsupported for complex number type conversions");
        }

        
        H5T__bit_set(d, dst_atomic.offset, dst_atomic.prec, false);

        
        specval_type = H5T__conv_float_find_special(s, &src_atomic, &sign);
        if (specval_type == H5T_CONV_FLOAT_SPECVAL_POSZERO ||
            specval_type == H5T_CONV_FLOAT_SPECVAL_NEGZERO) {
            
            goto padding;
        }
        else if (specval_type != H5T_CONV_FLOAT_SPECVAL_REGULAR) {
            
            if (conv_ctx->u.conv.cb_struct.func) {
                H5T_conv_except_t except_type; 

                
                H5T__reverse_order(src_rev, s, src_p);

                if (specval_type == H5T_CONV_FLOAT_SPECVAL_POSINF)
                    except_type = H5T_CONV_EXCEPT_PINF;
                else if (specval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF)
                    except_type = H5T_CONV_EXCEPT_NINF;
                else
                    except_type = H5T_CONV_EXCEPT_NAN;

                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        except_ret = (conv_ctx->u.conv.cb_struct.func)(
                            except_type, conv_ctx->u.conv.src_type_id, conv_ctx->u.conv.dst_type_id, src_rev,
                            d, conv_ctx->u.conv.cb_struct.user_data);
                    }
                H5_AFTER_USER_CB(FAIL)
            }

            if (except_ret == H5T_CONV_UNHANDLED) {
                if (specval_type == H5T_CONV_FLOAT_SPECVAL_NAN)
                    goto padding; 
                else if (specval_type == H5T_CONV_FLOAT_SPECVAL_POSINF) {
                    if (H5T_SGN_NONE == dst_atomic.u.i.sign)
                        H5T__bit_set(d, dst_atomic.offset, dst_atomic.prec, true);
                    else if (H5T_SGN_2 == dst_atomic.u.i.sign)
                        H5T__bit_set(d, dst_atomic.offset, dst_atomic.prec - 1, true);
                }
                else if (specval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF) {
                    if (H5T_SGN_2 == dst_atomic.u.i.sign)
                        H5T__bit_set(d, dst_atomic.prec - 1, (size_t)1, true);
                }
            }
            else if (except_ret == H5T_CONV_HANDLED) {
                
                reverse = false;
                goto next;
            }
            else if (except_ret == H5T_CONV_ABORT)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");

            goto padding;
        }

        
        expo = (hssize_t)H5T__bit_get_d(s, src_atomic.u.f.epos, src_atomic.u.f.esize);

        
        if (0 == expo || H5T_NORM_NONE == src_atomic.u.f.norm)
            expo -= (hssize_t)(src_atomic.u.f.ebias - 1);
        else if (H5T_NORM_IMPLIED == src_atomic.u.f.norm)
            expo -= (hssize_t)src_atomic.u.f.ebias;
        else
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "normalization method not implemented yet");

        
        H5T__bit_copy(int_buf, (size_t)0, s, src_atomic.u.f.mpos, src_atomic.u.f.msize);

        
        if (H5T_NORM_IMPLIED == src_atomic.u.f.norm)
            H5T__bit_inc(int_buf, src_atomic.u.f.msize, 8 * int_buf_size - src_atomic.u.f.msize);

        
        msb_pos_s = H5T__bit_find(int_buf, (size_t)0, src_atomic.prec, H5T_BIT_MSB, true);

        
        if (msb_pos_s < 0)
            goto padding;

        
        shift_val = expo - (ssize_t)src_atomic.u.f.msize;
        H5T__bit_shift(int_buf, shift_val, (size_t)0, int_buf_size * 8);

        
        new_msb_pos = msb_pos_s + shift_val;
        if (new_msb_pos == -1)
            goto padding;

        
        if ((size_t)expo < src_atomic.u.f.msize && conv_ctx->u.conv.cb_struct.func)
            truncated = true;

        if (H5T_SGN_NONE == dst_atomic.u.i.sign) { 
            
            if (sign) { 
                
                if (conv_ctx->u.conv.cb_struct.func) {
                    
                    H5T__reverse_order(src_rev, s, src_p);

                    
                    H5_BEFORE_USER_CB(FAIL)
                        {
                            except_ret = (conv_ctx->u.conv.cb_struct.func)(
                                H5T_CONV_EXCEPT_RANGE_LOW, conv_ctx->u.conv.src_type_id,
                                conv_ctx->u.conv.dst_type_id, src_rev, d,
                                conv_ctx->u.conv.cb_struct.user_data);
                        }
                    H5_AFTER_USER_CB(FAIL)

                    if (except_ret == H5T_CONV_HANDLED) {
                        
                        reverse = false;
                        goto next;
                    }
                    else if (except_ret == H5T_CONV_ABORT)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
                }
            }
            else { 
                if (new_msb_pos >= (ssize_t)dst_atomic.prec) {
                    
                    if (conv_ctx->u.conv.cb_struct.func) {
                        
                        H5T__reverse_order(src_rev, s, src_p);

                        
                        H5_BEFORE_USER_CB(FAIL)
                            {
                                except_ret = (conv_ctx->u.conv.cb_struct.func)(
                                    H5T_CONV_EXCEPT_RANGE_HI, conv_ctx->u.conv.src_type_id,
                                    conv_ctx->u.conv.dst_type_id, src_rev, d,
                                    conv_ctx->u.conv.cb_struct.user_data);
                            }
                        H5_AFTER_USER_CB(FAIL)
                    }

                    if (except_ret == H5T_CONV_UNHANDLED)
                        H5T__bit_set(d, dst_atomic.offset, dst_atomic.prec, true);
                    else if (except_ret == H5T_CONV_HANDLED) {
                        
                        reverse = false;
                        goto next;
                    }
                    else if (except_ret == H5T_CONV_ABORT)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
                }
                else {
                    
                    if (truncated && conv_ctx->u.conv.cb_struct.func) {
                        
                        H5T__reverse_order(src_rev, s, src_p);

                        
                        H5_BEFORE_USER_CB(FAIL)
                            {
                                except_ret = (conv_ctx->u.conv.cb_struct.func)(
                                    H5T_CONV_EXCEPT_TRUNCATE, conv_ctx->u.conv.src_type_id,
                                    conv_ctx->u.conv.dst_type_id, src_rev, d,
                                    conv_ctx->u.conv.cb_struct.user_data);
                            }
                        H5_AFTER_USER_CB(FAIL)
                    }

                    if (except_ret == H5T_CONV_UNHANDLED) {
                        
                        if (new_msb_pos >= 0)
                            H5T__bit_copy(d, dst_atomic.offset, int_buf, (size_t)0, (size_t)new_msb_pos + 1);
                    }
                    else if (except_ret == H5T_CONV_HANDLED) {
                        
                        reverse = false;
                        goto next;
                    }
                    else if (except_ret == H5T_CONV_ABORT)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
                }
            }
        }
        else if (H5T_SGN_2 == dst_atomic.u.i.sign) { 
            if (sign) {                              
                if ((new_msb_pos >= 0) && ((size_t)new_msb_pos < dst_atomic.prec - 1)) {
                    
                    if (truncated && conv_ctx->u.conv.cb_struct.func) {
                        
                        H5T__reverse_order(src_rev, s, src_p);

                        
                        H5_BEFORE_USER_CB(FAIL)
                            {
                                except_ret = (conv_ctx->u.conv.cb_struct.func)(
                                    H5T_CONV_EXCEPT_TRUNCATE, conv_ctx->u.conv.src_type_id,
                                    conv_ctx->u.conv.dst_type_id, src_rev, d,
                                    conv_ctx->u.conv.cb_struct.user_data);
                            }
                        H5_AFTER_USER_CB(FAIL)
                    }

                    if (except_ret == H5T_CONV_UNHANDLED) { 
                        
                        H5T__bit_dec(int_buf, (size_t)0, dst_atomic.prec);
                        H5T__bit_neg(int_buf, (size_t)0, dst_atomic.prec);

                        
                        H5T__bit_copy(d, dst_atomic.offset, int_buf, (size_t)0, dst_atomic.prec - 1);
                        H5T__bit_set(d, (dst_atomic.offset + dst_atomic.prec - 1), (size_t)1, true);
                    }
                    else if (except_ret == H5T_CONV_HANDLED) {
                        
                        reverse = false;
                        goto next;
                    }
                    else if (except_ret == H5T_CONV_ABORT)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
                }
                else {
                    
                    if (conv_ctx->u.conv.cb_struct.func) {
                        
                        H5T__reverse_order(src_rev, s, src_p);

                        
                        H5_BEFORE_USER_CB(FAIL)
                            {
                                except_ret = (conv_ctx->u.conv.cb_struct.func)(
                                    H5T_CONV_EXCEPT_RANGE_LOW, conv_ctx->u.conv.src_type_id,
                                    conv_ctx->u.conv.dst_type_id, src_rev, d,
                                    conv_ctx->u.conv.cb_struct.user_data);
                            }
                        H5_AFTER_USER_CB(FAIL)
                    }

                    if (except_ret == H5T_CONV_UNHANDLED)
                        H5T__bit_set(d, (dst_atomic.offset + dst_atomic.prec - 1), (size_t)1, true);
                    else if (except_ret == H5T_CONV_HANDLED) {
                        
                        reverse = false;
                        goto next;
                    }
                    else if (except_ret == H5T_CONV_ABORT)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
                }
            }
            else { 
                if (new_msb_pos >= (ssize_t)dst_atomic.prec - 1) {
                    
                    if (conv_ctx->u.conv.cb_struct.func) {
                        
                        H5T__reverse_order(src_rev, s, src_p);

                        
                        H5_BEFORE_USER_CB(FAIL)
                            {
                                except_ret = (conv_ctx->u.conv.cb_struct.func)(
                                    H5T_CONV_EXCEPT_RANGE_HI, conv_ctx->u.conv.src_type_id,
                                    conv_ctx->u.conv.dst_type_id, src_rev, d,
                                    conv_ctx->u.conv.cb_struct.user_data);
                            }
                        H5_AFTER_USER_CB(FAIL)
                    }

                    if (except_ret == H5T_CONV_UNHANDLED)
                        H5T__bit_set(d, dst_atomic.offset, dst_atomic.prec - 1, true);
                    else if (except_ret == H5T_CONV_HANDLED) {
                        
                        reverse = false;
                        goto next;
                    }
                    else if (except_ret == H5T_CONV_ABORT)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
                }
                else if (new_msb_pos < (ssize_t)dst_atomic.prec - 1) {
                    
                    if (truncated && conv_ctx->u.conv.cb_struct.func) {
                        
                        H5T__reverse_order(src_rev, s, src_p);

                        
                        H5_BEFORE_USER_CB(FAIL)
                            {
                                except_ret = (conv_ctx->u.conv.cb_struct.func)(
                                    H5T_CONV_EXCEPT_TRUNCATE, conv_ctx->u.conv.src_type_id,
                                    conv_ctx->u.conv.dst_type_id, src_rev, d,
                                    conv_ctx->u.conv.cb_struct.user_data);
                            }
                        H5_AFTER_USER_CB(FAIL)
                    }

                    if (except_ret == H5T_CONV_UNHANDLED) {
                        
                        if (new_msb_pos >= 0)
                            H5T__bit_copy(d, dst_atomic.offset, int_buf, (size_t)0, (size_t)new_msb_pos + 1);
                    }
                    else if (except_ret == H5T_CONV_HANDLED) {
                        
                        reverse = false;
                        goto next;
                    }
                    else if (except_ret == H5T_CONV_ABORT)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
                }
            }
        }

padding:
        
        if (dst_atomic.offset > 0) {
            assert(H5T_PAD_ZERO == dst_atomic.lsb_pad || H5T_PAD_ONE == dst_atomic.lsb_pad);
            H5T__bit_set(d, (size_t)0, dst_atomic.offset, (bool)(H5T_PAD_ONE == dst_atomic.lsb_pad));
        }
        if (dst_atomic.offset + dst_atomic.prec != 8 * dst_p->shared->size) {
            assert(H5T_PAD_ZERO == dst_atomic.msb_pad || H5T_PAD_ONE == dst_atomic.msb_pad);
            H5T__bit_set(d, dst_atomic.offset + dst_atomic.prec,
                         8 * dst_p->shared->size - (dst_atomic.offset + dst_atomic.prec),
                         (bool)(H5T_PAD_ONE == dst_atomic.msb_pad));
        }

        
        if (H5T_ORDER_BE == dst_atomic.order && reverse)
            for (size_t i = 0; i < dst_p->shared->size / 2; i++)
                H5_SWAP_BYTES(d, i, dst_p->shared->size - (i + 1));

next:
        
        if (d == dbuf)
            H5MM_memcpy(dp, d, dst_p->shared->size);

        
        sp += src_delta;
        dp += dst_delta;

        memset(int_buf, 0, int_buf_size);
    } 

done:
    H5MM_free(src_rev);
    H5MM_free(int_buf);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__conv_f_complex(const H5T_t *src_p, const H5T_t *dst_p, H5T_cdata_t *cdata,
                    const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                    size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    bool   equal_cplx_conv = false; 
    herr_t ret_value       = SUCCEED;

    FUNC_ENTER_PACKAGE

    switch (cdata->command) {
        case H5T_CONV_INIT: {
            H5T_atomic_t src_atomic; 
            H5T_atomic_t dst_atomic; 

            if (!src_p || !dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (!H5T_IS_ATOMIC(dst_p->shared->parent->shared))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid complex number datatype");
            src_atomic = src_p->shared->u.atomic;
            dst_atomic = dst_p->shared->parent->shared->u.atomic;
            if (H5T_ORDER_LE != src_atomic.order && H5T_ORDER_BE != src_atomic.order &&
                H5T_ORDER_VAX != src_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "unsupported byte order for source datatype");
            if (H5T_ORDER_LE != dst_atomic.order && H5T_ORDER_BE != dst_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "unsupported byte order for destination datatype");
            if (dst_p->shared->size > TEMP_FLOAT_CONV_BUFFER_SIZE)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "destination datatype size is too large");
            if (8 * sizeof(int64_t) - 1 < src_atomic.u.f.esize ||
                8 * sizeof(int64_t) - 1 < dst_atomic.u.f.esize)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "exponent field is too large");
            cdata->need_bkg = H5T_BKG_NO;

            break;
        }

        case H5T_CONV_FREE:
            break;

        case H5T_CONV_CONV:
            if (!src_p || !dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (NULL == conv_ctx)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid datatype conversion context pointer");

            
            equal_cplx_conv = (0 == H5T_cmp(src_p, dst_p->shared->parent, false));
            if (!equal_cplx_conv) {
                
                if (H5T__conv_f_f_loop(src_p, dst_p, conv_ctx, nelmts, buf_stride, buf) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "unable to convert data values");
            }
            else {
                
                if (H5T__conv_complex_f_matched(src_p, dst_p, conv_ctx, nelmts, buf_stride, buf) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "unable to convert data values");
            }

            break;

        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown conversion command");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5_HAVE__FLOAT16

herr_t
H5T__conv__Float16_schar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT16, SCHAR, H5__Float16, signed char, SCHAR_MIN, SCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv__Float16_uchar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT16, UCHAR, H5__Float16, unsigned char, 0, UCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv__Float16_short(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT16, SHORT, H5__Float16, short, SHRT_MIN, SHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv__Float16_ushort(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fX(FLOAT16, USHORT, H5__Float16, unsigned short, 0, USHRT_MAX);
}

herr_t
H5T__conv__Float16_int(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fX(FLOAT16, INT, H5__Float16, int, INT_MIN, INT_MAX);
}

herr_t
H5T__conv__Float16_uint(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fX(FLOAT16, UINT, H5__Float16, unsigned int, 0, UINT_MAX);
}

herr_t
H5T__conv__Float16_long(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fX(FLOAT16, LONG, H5__Float16, long, LONG_MIN, LONG_MAX);
}

herr_t
H5T__conv__Float16_ulong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fX(FLOAT16, ULONG, H5__Float16, unsigned long, 0, ULONG_MAX);
}

herr_t
H5T__conv__Float16_llong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fX(FLOAT16, LLONG, H5__Float16, long long, LLONG_MIN, LLONG_MAX);
}

herr_t
H5T__conv__Float16_ullong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fX(FLOAT16, ULLONG, H5__Float16, unsigned long long, 0, ULLONG_MAX);
}

herr_t
H5T__conv__Float16_float(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fF(FLOAT16, FLOAT, H5__Float16, float, -, -);
}

herr_t
H5T__conv__Float16_double(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fF(FLOAT16, DOUBLE, H5__Float16, double, -, -);
}

herr_t
H5T__conv__Float16_ldouble(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                           const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                           size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fF(FLOAT16, LDOUBLE, H5__Float16, long double, -, -);
}

#ifdef H5_HAVE_COMPLEX_NUMBERS

herr_t
H5T__conv__Float16_fcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fZ(FLOAT16, FLOAT_COMPLEX, H5__Float16, H5_float_complex, -, -);
}

herr_t
H5T__conv__Float16_dcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fZ(FLOAT16, DOUBLE_COMPLEX, H5__Float16, H5_double_complex, -, -);
}

herr_t
H5T__conv__Float16_lcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fZ(FLOAT16, LDOUBLE_COMPLEX, H5__Float16, H5_ldouble_complex, -, -);
}
#endif
#endif

herr_t
H5T__conv_float_schar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                      size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                      void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT, SCHAR, float, signed char, SCHAR_MIN, SCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_float_uchar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                      size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                      void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT, UCHAR, float, unsigned char, 0, UCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_float_short(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                      size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                      void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT, SHORT, float, short, SHRT_MIN, SHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_float_ushort(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT, USHORT, float, unsigned short, 0, USHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_float_int(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                    size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                    void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT, INT, float, int, INT_MIN, INT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_float_uint(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                     size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                     void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT, UINT, float, unsigned int, 0, UINT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_float_long(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                     size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                     void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT, LONG, float, long, LONG_MIN, LONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_float_ulong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                      size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                      void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT, ULONG, float, unsigned long, 0, ULONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_float_llong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                      size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                      void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT, LLONG, float, long long, LLONG_MIN, LLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_float_ullong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(FLOAT, ULLONG, float, unsigned long long, 0, ULLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

#ifdef H5_HAVE__FLOAT16

herr_t
H5T__conv_float__Float16(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    
    H5_WARN_NONSTD_SUFFIX_OFF
    H5T_CONV_Ff(FLOAT, FLOAT16, float, H5__Float16, -FLT16_MAX, FLT16_MAX);
    H5_WARN_NONSTD_SUFFIX_ON
}
#endif

herr_t
H5T__conv_float_double(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fF(FLOAT, DOUBLE, float, double, -, -);
}

herr_t
H5T__conv_float_ldouble(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fF(FLOAT, LDOUBLE, float, long double, -, -);
}

#ifdef H5_HAVE_COMPLEX_NUMBERS

herr_t
H5T__conv_float_fcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fz(FLOAT, FLOAT_COMPLEX, float, H5_float_complex, -, -);
}

herr_t
H5T__conv_float_dcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fZ(FLOAT, DOUBLE_COMPLEX, float, H5_double_complex, -, -);
}

herr_t
H5T__conv_float_lcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fZ(FLOAT, LDOUBLE_COMPLEX, float, H5_ldouble_complex, -, -);
}
#endif

herr_t
H5T__conv_double_schar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(DOUBLE, SCHAR, double, signed char, SCHAR_MIN, SCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_double_uchar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(DOUBLE, UCHAR, double, unsigned char, 0, UCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_double_short(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(DOUBLE, SHORT, double, short, SHRT_MIN, SHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_double_ushort(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(DOUBLE, USHORT, double, unsigned short, 0, USHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_double_int(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                     size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                     void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(DOUBLE, INT, double, int, INT_MIN, INT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_double_uint(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                      size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                      void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(DOUBLE, UINT, double, unsigned int, 0, UINT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_double_long(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                      size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                      void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(DOUBLE, LONG, double, long, LONG_MIN, LONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_double_ulong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(DOUBLE, ULONG, double, unsigned long, 0, ULONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_double_llong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(DOUBLE, LLONG, double, long long, LLONG_MIN, LLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_double_ullong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(DOUBLE, ULLONG, double, unsigned long long, 0, ULLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

#ifdef H5_HAVE__FLOAT16

herr_t
H5T__conv_double__Float16(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    
    H5_WARN_NONSTD_SUFFIX_OFF
    H5T_CONV_Ff(DOUBLE, FLOAT16, double, H5__Float16, -FLT16_MAX, FLT16_MAX);
    H5_WARN_NONSTD_SUFFIX_ON
}
#endif

herr_t
H5T__conv_double_float(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Ff(DOUBLE, FLOAT, double, float, -FLT_MAX, FLT_MAX);
}

herr_t
H5T__conv_double_ldouble(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fF(DOUBLE, LDOUBLE, double, long double, -, -);
}

#ifdef H5_HAVE_COMPLEX_NUMBERS

herr_t
H5T__conv_double_fcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Fz(DOUBLE, FLOAT_COMPLEX, double, H5_float_complex, -FLT_MAX, FLT_MAX);
}

herr_t
H5T__conv_double_dcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fz(DOUBLE, DOUBLE_COMPLEX, double, H5_double_complex, -, -);
}

herr_t
H5T__conv_double_lcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fZ(DOUBLE, LDOUBLE_COMPLEX, double, H5_ldouble_complex, -, -);
}
#endif

herr_t
H5T__conv_ldouble_schar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(LDOUBLE, SCHAR, long double, signed char, SCHAR_MIN, SCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_ldouble_uchar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(LDOUBLE, UCHAR, long double, unsigned char, 0, UCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_ldouble_short(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(LDOUBLE, SHORT, long double, short, SHRT_MIN, SHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_ldouble_ushort(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(LDOUBLE, USHORT, long double, unsigned short, 0, USHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_ldouble_int(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                      size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                      void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(LDOUBLE, INT, long double, int, INT_MIN, INT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_ldouble_uint(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(LDOUBLE, UINT, long double, unsigned int, 0, UINT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_ldouble_long(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(LDOUBLE, LONG, long double, long, LONG_MIN, LONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_ldouble_ulong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(LDOUBLE, ULONG, long double, unsigned long, 0, ULONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

#ifdef H5T_CONV_INTERNAL_LDOUBLE_LLONG
herr_t
H5T__conv_ldouble_llong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(LDOUBLE, LLONG, long double, long long, LLONG_MIN, LLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}
#endif 

#ifdef H5T_CONV_INTERNAL_LDOUBLE_ULLONG
herr_t
H5T__conv_ldouble_ullong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Fx(LDOUBLE, ULLONG, long double, unsigned long long, 0, ULLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}
#endif 

#ifdef H5_HAVE__FLOAT16
#ifdef H5T_CONV_INTERNAL_LDOUBLE_FLOAT16

herr_t
H5T__conv_ldouble__Float16(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                           const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                           size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    
    H5_WARN_NONSTD_SUFFIX_OFF
    H5T_CONV_Ff(LDOUBLE, FLOAT16, long double, H5__Float16, -FLT16_MAX, FLT16_MAX);
    H5_WARN_NONSTD_SUFFIX_ON
}
#endif
#endif

herr_t
H5T__conv_ldouble_float(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Ff(LDOUBLE, FLOAT, long double, float, -FLT_MAX, FLT_MAX);
}

herr_t
H5T__conv_ldouble_double(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Ff(LDOUBLE, DOUBLE, long double, double, -DBL_MAX, DBL_MAX);
}

#ifdef H5_HAVE_COMPLEX_NUMBERS

herr_t
H5T__conv_ldouble_fcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                           const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                           size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Fz(LDOUBLE, FLOAT_COMPLEX, long double, H5_float_complex, -FLT_MAX, FLT_MAX);
}

herr_t
H5T__conv_ldouble_dcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                           const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                           size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Fz(LDOUBLE, DOUBLE_COMPLEX, long double, H5_double_complex, -DBL_MAX, DBL_MAX);
}

herr_t
H5T__conv_ldouble_lcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                           const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                           size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_fz(LDOUBLE, LDOUBLE_COMPLEX, long double, H5_ldouble_complex, -, -);
}
#endif
