/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5D_FRIEND     
#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      

static void  *H5O__layout_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                 size_t p_size, const uint8_t *p);
static herr_t H5O__layout_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                                 const void *_mesg);
static void  *H5O__layout_copy(const void *_mesg, void *_dest);
static size_t H5O__layout_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__layout_reset(void *_mesg);
static herr_t H5O__layout_free(void *_mesg);
static herr_t H5O__layout_delete(H5F_t *f, H5O_t *open_oh, void *_mesg);
static herr_t H5O__layout_pre_copy_file(H5F_t *file_src, const void *mesg_src, bool *deleted,
                                        const H5O_copy_t *cpy_info, void *udata);
static void  *H5O__layout_copy_file(H5F_t *file_src, void *mesg_src, H5F_t *file_dst, bool *recompute_size,
                                    unsigned *mesg_flags, H5O_copy_t *cpy_info, void *udata);
static herr_t H5O__layout_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_LAYOUT[1] = {{
    H5O_LAYOUT_ID,             
    "layout",                  
    sizeof(H5O_layout_t),      
    0,                         
    H5O__layout_decode,        
    H5O__layout_encode,        
    H5O__layout_copy,          
    H5O__layout_size,          
    H5O__layout_reset,         
    H5O__layout_free,          
    H5O__layout_delete,        
    NULL,                      
    NULL,                      
    NULL,                      
    H5O__layout_pre_copy_file, 
    H5O__layout_copy_file,     
    NULL,                      
    NULL,                      
    NULL,                      
    H5O__layout_debug          
}};

H5FL_DEFINE(H5O_layout_t);

static void *
H5O__layout_decode(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                   unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    const uint8_t *p_end     = p + p_size - 1; 
    H5O_layout_t  *mesg      = NULL;
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    if (NULL == (mesg = H5FL_CALLOC(H5O_layout_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, NULL, "memory allocation failed");
    mesg->storage.type = H5D_LAYOUT_ERROR;

    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    mesg->version = *p++;

    if (mesg->version < H5O_LAYOUT_VERSION_1 || mesg->version > H5O_LAYOUT_VERSION_LATEST)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "bad version number for layout message");

    if (mesg->version < H5O_LAYOUT_VERSION_3) {
        unsigned ndims; 

        
        if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        ndims = *p++;

        if (!ndims || ndims > H5O_LAYOUT_NDIMS)
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "dimensionality is out of range");

        
        if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        mesg->type = (H5D_layout_t)*p++;

        if (H5D_CONTIGUOUS != mesg->type && H5D_CHUNKED != mesg->type && H5D_COMPACT != mesg->type)
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "bad layout type for layout message");

        
        mesg->storage.type = mesg->type;

        
        if (H5_IS_BUFFER_OVERFLOW(p, 5, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        p += 5;

        
        if (mesg->type == H5D_CONTIGUOUS) {
            if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            H5F_addr_decode(f, &p, &(mesg->storage.u.contig.addr));

            
            mesg->ops = H5D_LOPS_CONTIG;
        }
        else if (mesg->type == H5D_CHUNKED) {
            if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            H5F_addr_decode(f, &p, &(mesg->storage.u.chunk.idx_addr));

            
            mesg->ops = H5D_LOPS_CHUNK;

            
            mesg->storage.u.chunk.idx_type = H5D_CHUNK_IDX_BTREE;
            mesg->storage.u.chunk.ops      = H5D_COPS_BTREE;
        }
        else if (mesg->type == H5D_COMPACT) {
            
            mesg->ops = H5D_LOPS_COMPACT;
        }
        else
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "invalid layout type");

        
        if (mesg->type != H5D_CHUNKED) {
            
            if (H5_IS_BUFFER_OVERFLOW(p, (ndims * 4), p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            p += ndims * sizeof(uint32_t); 
        }
        else {
            if (ndims < 2)
                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "bad dimensions for chunked storage");
            mesg->u.chunk.ndims = ndims;

            if (H5_IS_BUFFER_OVERFLOW(p, (ndims * 4), p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            for (unsigned u = 0; u < ndims; u++) {

                UINT32DECODE(p, mesg->u.chunk.dim[u]);

                
                if (mesg->u.chunk.dim[u] == 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                "bad chunk dimension value when parsing layout message - chunk dimension "
                                "must be positive: mesg->u.chunk.dim[%u] = %" PRIuHSIZE,
                                u, mesg->u.chunk.dim[u]);
            }

            
            mesg->u.chunk.size = mesg->u.chunk.dim[0];
            for (unsigned u = 1; u < ndims; u++)
                mesg->u.chunk.size *= mesg->u.chunk.dim[u];
        }

        if (mesg->type == H5D_COMPACT) {
            if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            UINT32DECODE(p, mesg->storage.u.compact.size);

            if (mesg->storage.u.compact.size > 0) {
                
                if (H5_IS_BUFFER_OVERFLOW(p, mesg->storage.u.compact.size, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

                if (NULL == (mesg->storage.u.compact.buf = H5MM_malloc(mesg->storage.u.compact.size)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, NULL,
                                "memory allocation failed for compact data buffer");
                H5MM_memcpy(mesg->storage.u.compact.buf, p, mesg->storage.u.compact.size);
                p += mesg->storage.u.compact.size;
            }
        }
    }
    else {
        
        if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        mesg->type = mesg->storage.type = (H5D_layout_t)*p++;

        
        switch (mesg->type) {
            case H5D_COMPACT:
                
                if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
                UINT16DECODE(p, mesg->storage.u.compact.size);

                if (mesg->storage.u.compact.size > 0) {
                    
                    if (H5_IS_BUFFER_OVERFLOW(p, mesg->storage.u.compact.size, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");

                    
                    if (NULL == (mesg->storage.u.compact.buf = H5MM_malloc(mesg->storage.u.compact.size)))
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, NULL,
                                    "memory allocation failed for compact data buffer");

                    
                    H5MM_memcpy(mesg->storage.u.compact.buf, p, mesg->storage.u.compact.size);
                    p += mesg->storage.u.compact.size;
                }

                
                mesg->ops = H5D_LOPS_COMPACT;
                break;

            case H5D_CONTIGUOUS:
                
                if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
                H5F_addr_decode(f, &p, &(mesg->storage.u.contig.addr));

                
                if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_size(f), p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
                H5F_DECODE_LENGTH(f, p, mesg->storage.u.contig.size);

                
                mesg->ops = H5D_LOPS_CONTIG;
                break;

            case H5D_CHUNKED:
                if (mesg->version < H5O_LAYOUT_VERSION_4) {
                    
                    mesg->u.chunk.flags = (uint8_t)0;

                    
                    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");
                    mesg->u.chunk.ndims = *p++;

                    if (mesg->u.chunk.ndims > H5O_LAYOUT_NDIMS)
                        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "dimensionality is too large");
                    if (mesg->u.chunk.ndims < 2)
                        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "bad dimensions for chunked storage");

                    
                    if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");
                    H5F_addr_decode(f, &p, &(mesg->storage.u.chunk.idx_addr));

                    if (H5_IS_BUFFER_OVERFLOW(p, (mesg->u.chunk.ndims * 4), p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");

                    
                    for (unsigned u = 0; u < mesg->u.chunk.ndims; u++) {

                        UINT32DECODE(p, mesg->u.chunk.dim[u]);

                        
                        if (mesg->u.chunk.dim[u] == 0)
                            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                        "bad chunk dimension value when parsing layout message - chunk "
                                        "dimension must be positive: mesg->u.chunk.dim[%u] = %" PRIuHSIZE,
                                        u, mesg->u.chunk.dim[u]);
                    }

                    
                    mesg->u.chunk.size = mesg->u.chunk.dim[0];
                    for (unsigned u = 1; u < mesg->u.chunk.ndims; u++)
                        mesg->u.chunk.size *= mesg->u.chunk.dim[u];

                    
                    mesg->storage.u.chunk.idx_type = H5D_CHUNK_IDX_BTREE;
                    mesg->storage.u.chunk.ops      = H5D_COPS_BTREE;
                }
                else {
                    
                    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");
                    mesg->u.chunk.flags = *p++;

                    
                    
                    if (mesg->u.chunk.flags & ~H5O_LAYOUT_ALL_CHUNK_FLAGS)
                        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "bad flag value for message");

                    
                    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");
                    mesg->u.chunk.ndims = *p++;

                    if (mesg->u.chunk.ndims > H5O_LAYOUT_NDIMS)
                        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "dimensionality is too large");

                    
                    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");
                    mesg->u.chunk.enc_bytes_per_dim = *p++;

                    if (mesg->u.chunk.enc_bytes_per_dim == 0 || mesg->u.chunk.enc_bytes_per_dim > 8)
                        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                    "encoded chunk dimension size is too large");

                    if (H5_IS_BUFFER_OVERFLOW(p, (mesg->u.chunk.ndims * mesg->u.chunk.enc_bytes_per_dim),
                                              p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");

                    
                    for (unsigned u = 0; u < mesg->u.chunk.ndims; u++) {
                        UINT64DECODE_VAR(p, mesg->u.chunk.dim[u], mesg->u.chunk.enc_bytes_per_dim);

                        
                        if (mesg->u.chunk.dim[u] == 0)
                            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                        "bad chunk dimension value when parsing layout message - chunk "
                                        "dimension must be positive: mesg->u.chunk.dim[%u] = %" PRIuHSIZE,
                                        u, mesg->u.chunk.dim[u]);
                    }

                    
                    mesg->u.chunk.size = mesg->u.chunk.dim[0];
                    for (unsigned u = 1; u < mesg->u.chunk.ndims; u++)
                        mesg->u.chunk.size *= mesg->u.chunk.dim[u];

                    
                    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");
                    mesg->u.chunk.idx_type = (H5D_chunk_index_t)*p++;

                    if (mesg->u.chunk.idx_type >= H5D_CHUNK_IDX_NTYPES)
                        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "unknown chunk index type");
                    mesg->storage.u.chunk.idx_type = mesg->u.chunk.idx_type;

                    switch (mesg->u.chunk.idx_type) {
                        case H5D_CHUNK_IDX_BTREE:
                            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                        "v1 B-tree index type should never be in a v4 layout message");
                            break;

                        case H5D_CHUNK_IDX_NONE: 
                            mesg->storage.u.chunk.ops = H5D_COPS_NONE;
                            break;

                        case H5D_CHUNK_IDX_SINGLE: 
                            if (mesg->u.chunk.flags & H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER) {
                                uint64_t nbytes = 0;

                                if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_size(f) + 4, p_end))
                                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                                "ran off end of input buffer while decoding");

                                H5F_DECODE_LENGTH(f, p, nbytes);
                                H5_CHECKED_ASSIGN(mesg->storage.u.chunk.u.single.nbytes, uint32_t, nbytes,
                                                  uint64_t);

                                UINT32DECODE(p, mesg->storage.u.chunk.u.single.filter_mask);
                            }

                            
                            mesg->storage.u.chunk.ops = H5D_COPS_SINGLE;
                            break;

                        case H5D_CHUNK_IDX_FARRAY:
                            
                            if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                            "ran off end of input buffer while decoding");
                            mesg->u.chunk.u.farray.cparam.max_dblk_page_nelmts_bits = *p++;

                            if (0 == mesg->u.chunk.u.farray.cparam.max_dblk_page_nelmts_bits)
                                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                            "invalid fixed array creation parameter");

                            
                            mesg->storage.u.chunk.ops = H5D_COPS_FARRAY;
                            break;

                        case H5D_CHUNK_IDX_EARRAY:
                            
                            if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                            "ran off end of input buffer while decoding");
                            mesg->u.chunk.u.earray.cparam.max_nelmts_bits = *p++;

                            if (0 == mesg->u.chunk.u.earray.cparam.max_nelmts_bits)
                                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                            "invalid extensible array creation parameter");

                            if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                            "ran off end of input buffer while decoding");
                            mesg->u.chunk.u.earray.cparam.idx_blk_elmts = *p++;

                            if (0 == mesg->u.chunk.u.earray.cparam.idx_blk_elmts)
                                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                            "invalid extensible array creation parameter");

                            if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                            "ran off end of input buffer while decoding");
                            mesg->u.chunk.u.earray.cparam.sup_blk_min_data_ptrs = *p++;

                            if (0 == mesg->u.chunk.u.earray.cparam.sup_blk_min_data_ptrs)
                                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                            "invalid extensible array creation parameter");

                            if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                            "ran off end of input buffer while decoding");
                            mesg->u.chunk.u.earray.cparam.data_blk_min_elmts = *p++;

                            if (0 == mesg->u.chunk.u.earray.cparam.data_blk_min_elmts)
                                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                            "invalid extensible array creation parameter");

                            if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                            "ran off end of input buffer while decoding");
                            mesg->u.chunk.u.earray.cparam.max_dblk_page_nelmts_bits = *p++;

                            if (0 == mesg->u.chunk.u.earray.cparam.max_dblk_page_nelmts_bits)
                                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                            "invalid extensible array creation parameter");

                            
                            mesg->storage.u.chunk.ops = H5D_COPS_EARRAY;
                            break;

                        case H5D_CHUNK_IDX_BT2: 
                            if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
                                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                            "ran off end of input buffer while decoding");
                            UINT32DECODE(p, mesg->u.chunk.u.btree2.cparam.node_size);

                            if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                            "ran off end of input buffer while decoding");
                            mesg->u.chunk.u.btree2.cparam.split_percent = *p++;

                            if (mesg->u.chunk.u.btree2.cparam.split_percent == 0 ||
                                mesg->u.chunk.u.btree2.cparam.split_percent > 100)
                                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                            "bad value for v2 B-tree split percent value - must be > 0 and "
                                            "<= 100: split percent = %" PRIu8,
                                            mesg->u.chunk.u.btree2.cparam.split_percent);

                            if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                            "ran off end of input buffer while decoding");
                            mesg->u.chunk.u.btree2.cparam.merge_percent = *p++;

                            if (mesg->u.chunk.u.btree2.cparam.merge_percent == 0 ||
                                mesg->u.chunk.u.btree2.cparam.merge_percent > 100)
                                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                            "bad value for v2 B-tree merge percent value - must be > 0 and "
                                            "<= 100: merge percent = %" PRIu8,
                                            mesg->u.chunk.u.btree2.cparam.merge_percent);

                            
                            mesg->storage.u.chunk.ops = H5D_COPS_BT2;
                            break;

                        case H5D_CHUNK_IDX_NTYPES:
                        default:
                            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "Invalid chunk index type");
                    }

                    
                    if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");
                    H5F_addr_decode(f, &p, &(mesg->storage.u.chunk.idx_addr));
                }

                
                mesg->ops = H5D_LOPS_CHUNK;
                break;

            case H5D_VIRTUAL:
                
                if (mesg->version < H5O_LAYOUT_VERSION_4)
                    HGOTO_ERROR(H5E_OHDR, H5E_VERSION, NULL, "invalid layout version with virtual layout");

                
                if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
                H5F_addr_decode(f, &p, &(mesg->storage.u.virt.serial_list_hobjid.addr));
                

                if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
                UINT32DECODE(p, mesg->storage.u.virt.serial_list_hobjid.idx);

                
                mesg->storage.u.virt.list_nused  = 0;
                mesg->storage.u.virt.list        = NULL;
                mesg->storage.u.virt.list_nalloc = 0;
                mesg->storage.u.virt.view        = H5D_VDS_ERROR;
                mesg->storage.u.virt.printf_gap  = HSIZE_UNDEF;
                mesg->storage.u.virt.source_fapl = -1;
                mesg->storage.u.virt.source_dapl = -1;
                mesg->storage.u.virt.init        = false;

                
                mesg->ops = H5D_LOPS_VIRTUAL;

                break;

            case H5D_LAYOUT_ERROR:
            case H5D_NLAYOUTS:
            default:
                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "Invalid layout class");
        }
    }

    
    ret_value = mesg;

done:
    if (ret_value == NULL) {
        if (mesg) {
            if (mesg->type == H5D_VIRTUAL)
                if (H5D__virtual_reset_layout(mesg) < 0)
                    HDONE_ERROR(H5E_OHDR, H5E_CANTFREE, NULL, "unable to reset virtual layout");
            H5FL_FREE(H5O_layout_t, mesg);
        }
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__layout_encode(H5F_t *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                   const void *_mesg)
{
    const H5O_layout_t *mesg = (const H5O_layout_t *)_mesg;
    unsigned            u;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(mesg);
    assert(p);

    
    *p++ = (uint8_t)((mesg->version < H5O_LAYOUT_VERSION_3) ? H5O_LAYOUT_VERSION_3 : mesg->version);

    
    *p++ = (uint8_t)mesg->type;

    
    switch (mesg->type) {
        case H5D_COMPACT:
            
            UINT16ENCODE(p, mesg->storage.u.compact.size);

            
            if (mesg->storage.u.compact.size > 0) {
                if (mesg->storage.u.compact.buf)
                    H5MM_memcpy(p, mesg->storage.u.compact.buf, mesg->storage.u.compact.size);
                else
                    memset(p, 0, mesg->storage.u.compact.size);
                p += mesg->storage.u.compact.size;
            } 
            break;

        case H5D_CONTIGUOUS:
            
            H5F_addr_encode(f, &p, mesg->storage.u.contig.addr);

            
            H5F_ENCODE_LENGTH(f, p, mesg->storage.u.contig.size);
            break;

        case H5D_CHUNKED:
            if (mesg->version < H5O_LAYOUT_VERSION_4) {
                
                assert(mesg->u.chunk.ndims > 0 && mesg->u.chunk.ndims <= H5O_LAYOUT_NDIMS);
                *p++ = (uint8_t)mesg->u.chunk.ndims;

                
                H5F_addr_encode(f, &p, mesg->storage.u.chunk.idx_addr);

                
                for (u = 0; u < mesg->u.chunk.ndims; u++)
                    UINT32ENCODE(p, mesg->u.chunk.dim[u]);
            } 
            else {
                
                *p++ = mesg->u.chunk.flags;

                
                assert(mesg->u.chunk.ndims > 0 && mesg->u.chunk.ndims <= H5O_LAYOUT_NDIMS);
                *p++ = (uint8_t)mesg->u.chunk.ndims;

                
                assert(mesg->u.chunk.enc_bytes_per_dim > 0 && mesg->u.chunk.enc_bytes_per_dim <= 8);
                *p++ = (uint8_t)mesg->u.chunk.enc_bytes_per_dim;

                
                for (u = 0; u < mesg->u.chunk.ndims; u++)
                    UINT64ENCODE_VAR(p, mesg->u.chunk.dim[u], mesg->u.chunk.enc_bytes_per_dim);

                
                *p++ = (uint8_t)mesg->u.chunk.idx_type;

                switch (mesg->u.chunk.idx_type) {
                    case H5D_CHUNK_IDX_BTREE:
                        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL,
                                    "v1 B-tree index type should never be in a v4 layout message");
                        break;

                    case H5D_CHUNK_IDX_NONE: 
                        break;

                    case H5D_CHUNK_IDX_SINGLE: 
                        
                        if (mesg->u.chunk.flags & H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER) {
                            H5F_ENCODE_LENGTH(f, p, mesg->storage.u.chunk.u.single.nbytes);
                            UINT32ENCODE(p, mesg->storage.u.chunk.u.single.filter_mask);
                        } 
                        break;

                    case H5D_CHUNK_IDX_FARRAY:
                        
                        *p++ = mesg->u.chunk.u.farray.cparam.max_dblk_page_nelmts_bits;
                        break;

                    case H5D_CHUNK_IDX_EARRAY:
                        
                        *p++ = mesg->u.chunk.u.earray.cparam.max_nelmts_bits;
                        *p++ = mesg->u.chunk.u.earray.cparam.idx_blk_elmts;
                        *p++ = mesg->u.chunk.u.earray.cparam.sup_blk_min_data_ptrs;
                        *p++ = mesg->u.chunk.u.earray.cparam.data_blk_min_elmts;
                        *p++ = mesg->u.chunk.u.earray.cparam.max_dblk_page_nelmts_bits;
                        break;

                    case H5D_CHUNK_IDX_BT2: 
                        UINT32ENCODE(p, mesg->u.chunk.u.btree2.cparam.node_size);
                        *p++ = mesg->u.chunk.u.btree2.cparam.split_percent;
                        *p++ = mesg->u.chunk.u.btree2.cparam.merge_percent;
                        break;

                    case H5D_CHUNK_IDX_NTYPES:
                    default:
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, FAIL, "Invalid chunk index type");
                } 

                
                H5F_addr_encode(f, &p, mesg->storage.u.chunk.idx_addr);
            } 
            break;

        case H5D_VIRTUAL:
            
            H5F_addr_encode(f, &p, mesg->storage.u.virt.serial_list_hobjid.addr);
            UINT32ENCODE(p, mesg->storage.u.virt.serial_list_hobjid.idx);
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, FAIL, "Invalid layout class");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__layout_copy(const void *_mesg, void *_dest)
{
    const H5O_layout_t *mesg      = (const H5O_layout_t *)_mesg;
    H5O_layout_t       *dest      = (H5O_layout_t *)_dest;
    void               *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(mesg);

    
    if (!dest && NULL == (dest = H5FL_MALLOC(H5O_layout_t)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, NULL, "layout message allocation failed");

    
    *dest = *mesg;

    
    switch (mesg->type) {
        case H5D_COMPACT:
            
            if (mesg->storage.u.compact.size > 0) {
                
                assert(mesg->storage.u.compact.buf);

                
                if (NULL == (dest->storage.u.compact.buf = H5MM_malloc(dest->storage.u.compact.size)))
                    HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, NULL, "unable to allocate memory for compact dataset");

                
                H5MM_memcpy(dest->storage.u.compact.buf, mesg->storage.u.compact.buf,
                            dest->storage.u.compact.size);
            } 
            else
                assert(dest->storage.u.compact.buf == NULL);
            break;

        case H5D_CONTIGUOUS:
            
            break;

        case H5D_CHUNKED:
            
            if (dest->storage.u.chunk.ops)
                H5D_chunk_idx_reset(&dest->storage.u.chunk, false);
            break;

        case H5D_VIRTUAL:
            if (H5D__virtual_copy_layout(dest) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, NULL, "unable to copy virtual layout");
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, NULL, "Invalid layout class");
    } 

    
    ret_value = dest;

done:
    if (ret_value == NULL)
        if (NULL == _dest)
            dest = H5FL_FREE(H5O_layout_t, dest);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__layout_size(const H5F_t *f, bool H5_ATTR_UNUSED disable_shared, const void *_mesg)
{
    const H5O_layout_t *mesg      = (const H5O_layout_t *)_mesg;
    size_t              ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);

    
    
    ret_value = H5D__layout_meta_size(f, mesg, true);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__layout_reset(void *_mesg)
{
    H5O_layout_t *mesg      = (H5O_layout_t *)_mesg;
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (mesg) {
        
        if (H5D_COMPACT == mesg->type)
            mesg->storage.u.compact.buf = H5MM_xfree(mesg->storage.u.compact.buf);
        else if (H5D_VIRTUAL == mesg->type)
            
            if (H5D__virtual_reset_layout(mesg) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to reset virtual layout");

        
        mesg->type    = H5D_CONTIGUOUS;
        mesg->version = H5O_LAYOUT_VERSION_DEFAULT;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__layout_free(void *_mesg)
{
    H5O_layout_t *mesg = (H5O_layout_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    
    H5O__layout_reset(mesg);

    (void)H5FL_FREE(H5O_layout_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__layout_delete(H5F_t *f, H5O_t *open_oh, void *_mesg)
{
    H5O_layout_t *mesg      = (H5O_layout_t *)_mesg;
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(open_oh);
    assert(mesg);

    
    switch (mesg->type) {
        case H5D_COMPACT: 
            
            break;

        case H5D_CONTIGUOUS: 
            
            if (H5D__contig_delete(f, &mesg->storage) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to free raw data");
            break;

        case H5D_CHUNKED: 
            
            if (H5D__chunk_delete(f, open_oh, mesg) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to free raw data");
            break;

        case H5D_VIRTUAL: 
            
            if (H5D__virtual_delete(f, &mesg->storage) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to free raw data");
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            HGOTO_ERROR(H5E_OHDR, H5E_BADTYPE, FAIL, "not valid storage type");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__layout_pre_copy_file(H5F_t H5_ATTR_UNUSED *file_src, const void *mesg_src, bool H5_ATTR_UNUSED *deleted,
                          const H5O_copy_t *cpy_info, void H5_ATTR_UNUSED *udata)
{
    const H5O_layout_t *layout_src = (const H5O_layout_t *)mesg_src; 
    herr_t              ret_value  = SUCCEED;                        

    FUNC_ENTER_PACKAGE

    
    assert(cpy_info);
    assert(cpy_info->file_dst);

    
    if (layout_src->version > H5O_layout_ver_bounds[H5F_HIGH_BOUND(cpy_info->file_dst)])
        HGOTO_ERROR(H5E_OHDR, H5E_BADRANGE, FAIL, "layout message version out of bounds");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__layout_copy_file(H5F_t *file_src, void *mesg_src, H5F_t *file_dst, bool H5_ATTR_UNUSED *recompute_size,
                      unsigned H5_ATTR_UNUSED *mesg_flags, H5O_copy_t *cpy_info, void *_udata)
{
    H5D_copy_file_ud_t *udata      = (H5D_copy_file_ud_t *)_udata; 
    H5O_layout_t       *layout_src = (H5O_layout_t *)mesg_src;
    H5O_layout_t       *layout_dst = NULL;
    void               *ret_value  = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(file_src);
    assert(layout_src);
    assert(file_dst);

    if (layout_src->type == H5D_VIRTUAL)
        if (H5D__virtual_load_layout(file_src, layout_src) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTLOAD, NULL, "unable to load virtual layout information");

    
    if (NULL == (layout_dst = (H5O_layout_t *)H5O__layout_copy(layout_src, NULL)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, NULL, "unable to copy layout");

    
    switch (layout_src->type) {
        case H5D_COMPACT:
            if (layout_src->storage.u.compact.buf) {
                
                if (H5D__compact_copy(file_src, &layout_src->storage.u.compact, file_dst,
                                      &layout_dst->storage.u.compact, udata->src_dtype, cpy_info) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, NULL, "unable to copy chunked storage");
            } 
            break;

        case H5D_CONTIGUOUS: {
            hsize_t nelmts;  
            size_t  dt_size; 
            
            if (H5D__contig_check(file_src, layout_src, udata->src_space_extent, udata->src_dtype) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "invalid layout / dataspace / datatype info");

            
            nelmts  = H5S_extent_nelem(udata->src_space_extent);
            dt_size = H5T_get_size(udata->src_dtype);
            if (layout_src->version < H5O_LAYOUT_VERSION_3)
                layout_dst->storage.u.contig.size = nelmts * dt_size;
            else
                
                if (layout_dst->storage.u.contig.size != (nelmts * dt_size))
                    HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "invalid layout storage size ");

            if (H5D__contig_is_space_alloc(&layout_src->storage) ||
                (cpy_info->shared_fo &&
                 H5D__contig_is_data_cached((const H5D_shared_t *)cpy_info->shared_fo))) {
                
                if (H5D__contig_copy(file_src, &layout_src->storage.u.contig, file_dst,
                                     &layout_dst->storage.u.contig, udata->src_dtype, cpy_info) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, NULL, "unable to copy contiguous storage");
            } 
        } break;

        case H5D_CHUNKED:
            if (H5D__chunk_is_space_alloc(&layout_src->storage) ||
                (cpy_info->shared_fo &&
                 H5D__chunk_is_data_cached((const H5D_shared_t *)cpy_info->shared_fo))) {
                
                if (H5D__chunk_copy(file_src, layout_src, file_dst, layout_dst, udata->src_space_extent,
                                    udata->src_dtype, udata->common.src_pline, cpy_info) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, NULL, "unable to copy chunked storage");
            } 
            break;

        case H5D_VIRTUAL:
            
            if (H5D__virtual_copy(file_dst, layout_dst) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, NULL, "unable to copy virtual storage");
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "Invalid layout class");
    } 

    
    ret_value = layout_dst;

done:
    if (!ret_value)
        if (layout_dst)
            layout_dst = H5FL_FREE(H5O_layout_t, layout_dst);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__layout_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_layout_t *mesg = (const H5O_layout_t *)_mesg;
    size_t              u;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Version:", mesg->version);
    switch (mesg->type) {
        case H5D_CHUNKED:
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Type:", "Chunked");

            
            Rfprintf(stream, "%*s%-*s %lu\n", indent, "", fwidth,
                    "Number of dimensions:", (unsigned long)(mesg->u.chunk.ndims));
            Rfprintf(stream, "%*s%-*s {", indent, "", fwidth, "Size:");
            for (u = 0; u < (size_t)mesg->u.chunk.ndims; u++)
                Rfprintf(stream, "%s%" PRIuHSIZE, u ? ", " : "", mesg->u.chunk.dim[u]);
            Rfprintf(stream, "}\n");

            
            switch (mesg->u.chunk.idx_type) {
                case H5D_CHUNK_IDX_BTREE:
                    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Index Type:", "v1 B-tree");
                    break;

                case H5D_CHUNK_IDX_NONE:
                    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Index Type:", "Implicit");
                    break;

                case H5D_CHUNK_IDX_SINGLE:
                    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Index Type:", "Single Chunk");
                    break;

                case H5D_CHUNK_IDX_FARRAY:
                    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Index Type:", "Fixed Array");
                    
                    break;

                case H5D_CHUNK_IDX_EARRAY:
                    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Index Type:", "Extensible Array");
                    
                    break;

                case H5D_CHUNK_IDX_BT2:
                    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Index Type:", "v2 B-tree");
                    
                    break;

                case H5D_CHUNK_IDX_NTYPES:
                default:
                    Rfprintf(stream, "%*s%-*s %s (%u)\n", indent, "", fwidth, "Index Type:", "Unknown",
                            (unsigned)mesg->u.chunk.idx_type);
                    break;
            } 
            Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
                    "Index address:", mesg->storage.u.chunk.idx_addr);
            break;

        case H5D_CONTIGUOUS:
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Type:", "Contiguous");
            Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
                    "Data address:", mesg->storage.u.contig.addr);
            Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
                    "Data Size:", mesg->storage.u.contig.size);
            break;

        case H5D_COMPACT:
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Type:", "Compact");
            Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, "Data Size:", (unsigned long long)mesg->storage.u.compact.size);
            break;

        case H5D_VIRTUAL:
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Type:", "Virtual");
            Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
                    "Global heap address:", mesg->storage.u.virt.serial_list_hobjid.addr);
            Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
                    "Global heap index:", (unsigned long long)mesg->storage.u.virt.serial_list_hobjid.idx);
            for (u = 0; u < mesg->storage.u.virt.list_nused; u++) {
                Rfprintf(stream, "%*sMapping %llu:\n", indent, "", (unsigned long long)u);
                Rfprintf(stream, "%*s%-*s %s\n", indent + 3, "", fwidth - 3,
                        "Virtual selection:", "<Not yet implemented>");
                Rfprintf(stream, "%*s%-*s %s\n", indent + 3, "", fwidth - 3,
                        "Source file name:", mesg->storage.u.virt.list[u].source_file_name);
                Rfprintf(stream, "%*s%-*s %s\n", indent + 3, "", fwidth - 3,
                        "Source dataset name:", mesg->storage.u.virt.list[u].source_dset_name);
                Rfprintf(stream, "%*s%-*s %s\n", indent + 3, "", fwidth - 3,
                        "Source selection:", "<Not yet implemented>");
            } 
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            Rfprintf(stream, "%*s%-*s %s (%u)\n", indent, "", fwidth, "Type:", "Unknown",
                    (unsigned)mesg->type);
            break;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
