/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 

void *
H5MM_realloc(void *mem, size_t size)
{
    void *ret_value = NULL;

    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (NULL == mem && 0 == size)
        
        ret_value = NULL;
    else {
        ret_value = realloc(mem, size);

        
        if (0 == size)
            ret_value = NULL;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

char *
H5MM_xstrdup(const char *s)
{
    char *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    if (s)
        if (NULL == (ret_value = strdup(s)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "string duplication failed");
done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

char *
H5MM_strdup(const char *s)
{
    char *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    if (!s)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "NULL string not allowed");
    if (NULL == (ret_value = strdup(s)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "string duplication failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

char *
H5MM_strndup(const char *s, size_t n)
{
    char *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    if (!s)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "NULL string not allowed");

    if (NULL == (ret_value = HDstrndup(s, n)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "string duplication failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5MM_xfree(void *mem)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    free(mem);

    FUNC_LEAVE_NOAPI(NULL)
} 

void *
H5MM_xfree_const(const void *mem)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    H5MM_xfree((void *)(uintptr_t)mem);

    FUNC_LEAVE_NOAPI(NULL)
} 

#ifdef H5MM_DEBUG

void *
H5MM_memcpy(void *dest, const void *src, size_t n)
{
    void *ret = NULL;

    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(dest);
    assert(src);

    
    assert((char *)dest >= (const char *)src + n || (const char *)src >= (char *)dest + n);

    
    ret = memcpy(dest, src, n);

    FUNC_LEAVE_NOAPI(ret)

} 

#endif 
