/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5HFpkg.h"     
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 

#define H5HF_HUGE_BT2_NODE_SIZE  512
#define H5HF_HUGE_BT2_SPLIT_PERC 100
#define H5HF_HUGE_BT2_MERGE_PERC 40

static herr_t H5HF__huge_bt2_create(H5HF_hdr_t *hdr);

static hsize_t H5HF__huge_new_id(H5HF_hdr_t *hdr);
static herr_t  H5HF__huge_op_real(H5HF_hdr_t *hdr, const uint8_t *id, bool is_read, H5HF_operator_t op,
                                  void *op_data);

static herr_t
H5HF__huge_bt2_create(H5HF_hdr_t *hdr)
{
    H5B2_create_t bt2_cparam;          
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    
    if (hdr->huge_ids_direct) {
        if (hdr->filter_len > 0) {
            bt2_cparam.rrec_size =
                (uint32_t)((unsigned)hdr->sizeof_addr     
                           + (unsigned)hdr->sizeof_size   
                           + (unsigned)4                  
                           + (unsigned)hdr->sizeof_size); 
            bt2_cparam.cls = H5HF_HUGE_BT2_FILT_DIR;
        } 
        else {
            bt2_cparam.rrec_size = (uint32_t)((unsigned)hdr->sizeof_addr     
                                              + (unsigned)hdr->sizeof_size); 
            bt2_cparam.cls       = H5HF_HUGE_BT2_DIR;
        } 
    }     
    else {
        if (hdr->filter_len > 0) {
            bt2_cparam.rrec_size =
                (uint32_t)((unsigned)hdr->sizeof_addr     
                           + (unsigned)hdr->sizeof_size   
                           + (unsigned)4                  
                           + (unsigned)hdr->sizeof_size   
                           + (unsigned)hdr->sizeof_size); 
            bt2_cparam.cls = H5HF_HUGE_BT2_FILT_INDIR;
        } 
        else {
            bt2_cparam.rrec_size = (uint32_t)((unsigned)hdr->sizeof_addr     
                                              + (unsigned)hdr->sizeof_size   
                                              + (unsigned)hdr->sizeof_size); 
            bt2_cparam.cls       = H5HF_HUGE_BT2_INDIR;
        } 
    }     
    bt2_cparam.node_size     = (size_t)H5HF_HUGE_BT2_NODE_SIZE;
    bt2_cparam.split_percent = H5HF_HUGE_BT2_SPLIT_PERC;
    bt2_cparam.merge_percent = H5HF_HUGE_BT2_MERGE_PERC;

    
    if (NULL == (hdr->huge_bt2 = H5B2_create(hdr->f, &bt2_cparam, hdr->f)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTCREATE, FAIL,
                    "can't create v2 B-tree for tracking 'huge' heap objects");

    
    if (H5B2_get_addr(hdr->huge_bt2, &hdr->huge_bt2_addr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL,
                    "can't get v2 B-tree address for tracking 'huge' heap objects");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__huge_init(H5HF_hdr_t *hdr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);

    

    
    if (hdr->filter_len > 0) {
        if ((hdr->id_len - 1) >= (unsigned)(hdr->sizeof_addr + hdr->sizeof_size + 4 + hdr->sizeof_size)) {
            
            hdr->huge_ids_direct = true;

            
            hdr->huge_id_size = (uint8_t)(hdr->sizeof_addr + hdr->sizeof_size + hdr->sizeof_size);
        } 
        else
            
            hdr->huge_ids_direct = false;
    } 
    else {
        if ((unsigned)(hdr->sizeof_addr + hdr->sizeof_size) <= (unsigned)(hdr->id_len - 1)) {
            
            hdr->huge_ids_direct = true;

            
            hdr->huge_id_size = (uint8_t)(hdr->sizeof_addr + hdr->sizeof_size);
        } 
        else
            
            hdr->huge_ids_direct = false;
    } 
    if (!hdr->huge_ids_direct) {
        
        if ((hdr->id_len - 1) < sizeof(hsize_t)) {
            hdr->huge_id_size = (uint8_t)(hdr->id_len - 1);
            hdr->huge_max_id  = ((hsize_t)1 << (hdr->huge_id_size * 8)) - 1;
        } 
        else {
            hdr->huge_id_size = sizeof(hsize_t);
            hdr->huge_max_id  = HSIZET_MAX;
        } 
    }     
    hdr->huge_bt2 = NULL;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static hsize_t
H5HF__huge_new_id(H5HF_hdr_t *hdr)
{
    hsize_t new_id;        
    hsize_t ret_value = 0; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->huge_ids_wrapped)
        
        HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, 0, "wrapping 'huge' object IDs not supported yet");
    else {
        
        
        new_id = ++hdr->huge_next_id;

        
        if (hdr->huge_next_id == hdr->huge_max_id)
            hdr->huge_ids_wrapped = true;
    } 

    
    ret_value = new_id;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__huge_insert(H5HF_hdr_t *hdr, size_t obj_size, void *obj, void *_id)
{
    uint8_t *id = (uint8_t *)_id;   
    haddr_t  obj_addr;              
    void    *write_buf;             
    size_t   write_size;            
    unsigned filter_mask = 0;       
    herr_t   ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(obj_size > hdr->max_man_size);
    assert(obj);
    assert(id);

    
    if (!H5_addr_defined(hdr->huge_bt2_addr)) {
        
        if (H5HF__huge_bt2_create(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCREATE, FAIL,
                        "can't create v2 B-tree for tracking 'huge' heap objects");
    } 
    else {
        
        if (NULL == hdr->huge_bt2) {
            
            if (NULL == (hdr->huge_bt2 = H5B2_open(hdr->f, hdr->huge_bt2_addr, hdr->f)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTOPENOBJ, FAIL,
                            "unable to open v2 B-tree for tracking 'huge' heap objects");
        } 
    }     
    assert(hdr->huge_bt2);

    
    if (hdr->filter_len > 0) {
        H5Z_cb_t filter_cb; 
        size_t   nbytes;    

        
        filter_cb.op_data = NULL;
        filter_cb.func    = NULL; 

        
        write_size = obj_size;
        if (NULL == (write_buf = H5MM_malloc(write_size)))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "memory allocation failed for pipeline buffer");
        H5MM_memcpy(write_buf, obj, write_size);

        
        nbytes = write_size;
        if (H5Z_pipeline(&(hdr->pline), 0, &filter_mask, H5Z_NO_EDC, filter_cb, &nbytes, &write_size,
                         &write_buf) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTFILTER, FAIL, "output pipeline failed");

        
        write_size = nbytes;
    } 
    else {
        write_buf  = obj;
        write_size = obj_size;
    } 

    
    if (HADDR_UNDEF == (obj_addr = H5MF_alloc(hdr->f, H5FD_MEM_FHEAP_HUGE_OBJ, (hsize_t)write_size)))
        HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "file allocation failed for fractal heap huge object");

    
    if (H5F_block_write(hdr->f, H5FD_MEM_FHEAP_HUGE_OBJ, obj_addr, write_size, write_buf) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_WRITEERROR, FAIL, "writing 'huge' object to file failed");

    
    if (write_buf != obj) {
        assert(hdr->filter_len > 0);
        H5MM_xfree(write_buf);
    } 

    
    if (hdr->huge_ids_direct) {
        if (hdr->filter_len > 0) {
            H5HF_huge_bt2_filt_dir_rec_t obj_rec; 

            
            obj_rec.addr        = obj_addr;
            obj_rec.len         = write_size;
            obj_rec.filter_mask = filter_mask;
            obj_rec.obj_size    = obj_size;

            
            if (H5B2_insert(hdr->huge_bt2, &obj_rec) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINSERT, FAIL,
                            "couldn't insert object tracking record in v2 B-tree");

            
            *id++ = H5HF_ID_VERS_CURR | H5HF_ID_TYPE_HUGE;
            H5F_addr_encode(hdr->f, &id, obj_addr);
            H5F_ENCODE_LENGTH(hdr->f, id, (hsize_t)write_size);
            UINT32ENCODE(id, filter_mask);
            H5F_ENCODE_LENGTH(hdr->f, id, (hsize_t)obj_size);
        } 
        else {
            H5HF_huge_bt2_dir_rec_t obj_rec; 

            
            obj_rec.addr = obj_addr;
            obj_rec.len  = write_size;

            
            if (H5B2_insert(hdr->huge_bt2, &obj_rec) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINSERT, FAIL,
                            "couldn't insert object tracking record in v2 B-tree");

            
            *id++ = H5HF_ID_VERS_CURR | H5HF_ID_TYPE_HUGE;
            H5F_addr_encode(hdr->f, &id, obj_addr);
            H5F_ENCODE_LENGTH(hdr->f, id, (hsize_t)write_size);
        } 
    }     
    else {
        H5HF_huge_bt2_filt_indir_rec_t filt_indir_rec; 
        H5HF_huge_bt2_indir_rec_t      indir_rec;      
        void                          *ins_rec;        
        hsize_t                        new_id;         

        
        if (0 == (new_id = H5HF__huge_new_id(hdr)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't generate new ID for object");

        if (hdr->filter_len > 0) {
            
            filt_indir_rec.addr        = obj_addr;
            filt_indir_rec.len         = write_size;
            filt_indir_rec.filter_mask = filter_mask;
            filt_indir_rec.obj_size    = obj_size;
            filt_indir_rec.id          = new_id;

            
            ins_rec = &filt_indir_rec;
        } 
        else {
            
            indir_rec.addr = obj_addr;
            indir_rec.len  = write_size;
            indir_rec.id   = new_id;

            
            ins_rec = &indir_rec;
        } 

        
        if (H5B2_insert(hdr->huge_bt2, ins_rec) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINSERT, FAIL,
                        "couldn't insert object tracking record in v2 B-tree");

        
        *id++ = H5HF_ID_VERS_CURR | H5HF_ID_TYPE_HUGE;
        UINT64ENCODE_VAR(id, new_id, hdr->huge_id_size);
    } 

    
    hdr->huge_size += obj_size;
    hdr->huge_nobjs++;

    
    if (H5HF__hdr_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark heap header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__huge_get_obj_len(H5HF_hdr_t *hdr, const uint8_t *id, size_t *obj_len_p)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(hdr->huge_bt2_addr));
    assert(id);
    assert(obj_len_p);

    
    id++;

    
    if (hdr->huge_ids_direct) {
        if (hdr->filter_len > 0) {
            
            id += hdr->sizeof_addr + hdr->sizeof_size + 4;

            
            H5F_DECODE_LENGTH(hdr->f, id, *obj_len_p);
        } 
        else {
            
            id += hdr->sizeof_addr;

            
            H5F_DECODE_LENGTH(hdr->f, id, *obj_len_p);
        } 
    }     
    else {
        bool found = false; 

        
        if (NULL == hdr->huge_bt2) {
            
            if (NULL == (hdr->huge_bt2 = H5B2_open(hdr->f, hdr->huge_bt2_addr, hdr->f)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTOPENOBJ, FAIL,
                            "unable to open v2 B-tree for tracking 'huge' heap objects");
        } 

        if (hdr->filter_len > 0) {
            H5HF_huge_bt2_filt_indir_rec_t found_rec;  
            H5HF_huge_bt2_filt_indir_rec_t search_rec; 

            
            UINT64DECODE_VAR(id, search_rec.id, hdr->huge_id_size);

            
            if (H5B2_find(hdr->huge_bt2, &search_rec, &found, H5HF__huge_bt2_filt_indir_found, &found_rec) <
                0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTFIND, FAIL, "can't check for object in v2 B-tree");
            if (!found)
                HGOTO_ERROR(H5E_HEAP, H5E_NOTFOUND, FAIL, "can't find object in v2 B-tree");

            
            *obj_len_p = (size_t)found_rec.obj_size;
        } 
        else {
            H5HF_huge_bt2_indir_rec_t found_rec;  
            H5HF_huge_bt2_indir_rec_t search_rec; 

            
            UINT64DECODE_VAR(id, search_rec.id, hdr->huge_id_size);

            
            if (H5B2_find(hdr->huge_bt2, &search_rec, &found, H5HF__huge_bt2_indir_found, &found_rec) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTFIND, FAIL, "can't check for object in v2 B-tree");
            if (!found)
                HGOTO_ERROR(H5E_HEAP, H5E_NOTFOUND, FAIL, "can't find object in v2 B-tree");

            
            *obj_len_p = (size_t)found_rec.len;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__huge_get_obj_off(H5HF_hdr_t *hdr, const uint8_t *id, hsize_t *obj_off_p)
{
    haddr_t obj_addr;            
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(hdr->huge_bt2_addr));
    assert(id);
    assert(obj_off_p);

    
    id++;

    
    if (hdr->huge_ids_direct) {
        
        H5F_addr_decode(hdr->f, &id, &obj_addr);
    } 
    else {
        bool found = false; 

        
        assert(H5_addr_defined(hdr->huge_bt2_addr));

        
        if (NULL == hdr->huge_bt2) {
            
            if (NULL == (hdr->huge_bt2 = H5B2_open(hdr->f, hdr->huge_bt2_addr, hdr->f)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTOPENOBJ, FAIL,
                            "unable to open v2 B-tree for tracking 'huge' heap objects");
        } 

        if (hdr->filter_len > 0) {
            H5HF_huge_bt2_filt_indir_rec_t found_rec;  
            H5HF_huge_bt2_filt_indir_rec_t search_rec; 

            
            UINT64DECODE_VAR(id, search_rec.id, hdr->huge_id_size);

            
            if (H5B2_find(hdr->huge_bt2, &search_rec, &found, H5HF__huge_bt2_filt_indir_found, &found_rec) <
                0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTFIND, FAIL, "can't check for object in v2 B-tree");
            if (!found)
                HGOTO_ERROR(H5E_HEAP, H5E_NOTFOUND, FAIL, "can't find object in v2 B-tree");

            
            obj_addr = found_rec.addr;
        } 
        else {
            H5HF_huge_bt2_indir_rec_t found_rec;  
            H5HF_huge_bt2_indir_rec_t search_rec; 

            
            UINT64DECODE_VAR(id, search_rec.id, hdr->huge_id_size);

            
            if (H5B2_find(hdr->huge_bt2, &search_rec, &found, H5HF__huge_bt2_indir_found, &found_rec) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTFIND, FAIL, "can't check for object in v2 B-tree");
            if (!found)
                HGOTO_ERROR(H5E_HEAP, H5E_NOTFOUND, FAIL, "can't find object in v2 B-tree");

            
            obj_addr = found_rec.addr;
        } 
    }     

    
    *obj_off_p = (hsize_t)obj_addr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__huge_op_real(H5HF_hdr_t *hdr, const uint8_t *id, bool is_read, H5HF_operator_t op, void *op_data)
{
    void    *read_buf = NULL;       
    haddr_t  obj_addr;              
    size_t   obj_size    = 0;       
    unsigned filter_mask = 0;       
    herr_t   ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);
    assert(is_read || op);

    
    id++;

    
    if (hdr->huge_ids_direct) {
        
        H5F_addr_decode(hdr->f, &id, &obj_addr);
        H5F_DECODE_LENGTH(hdr->f, id, obj_size);

        
        if (hdr->filter_len > 0)
            UINT32DECODE(id, filter_mask);
    } 
    else {
        bool found = false; 

        
        assert(H5_addr_defined(hdr->huge_bt2_addr));

        
        if (NULL == hdr->huge_bt2) {
            
            if (NULL == (hdr->huge_bt2 = H5B2_open(hdr->f, hdr->huge_bt2_addr, hdr->f)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTOPENOBJ, FAIL,
                            "unable to open v2 B-tree for tracking 'huge' heap objects");
        } 

        if (hdr->filter_len > 0) {
            H5HF_huge_bt2_filt_indir_rec_t found_rec;  
            H5HF_huge_bt2_filt_indir_rec_t search_rec; 

            
            UINT64DECODE_VAR(id, search_rec.id, hdr->huge_id_size);

            
            if (H5B2_find(hdr->huge_bt2, &search_rec, &found, H5HF__huge_bt2_filt_indir_found, &found_rec) <
                0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTFIND, FAIL, "can't check for object in v2 B-tree");
            if (!found)
                HGOTO_ERROR(H5E_HEAP, H5E_NOTFOUND, FAIL, "can't find object in v2 B-tree");

            
            obj_addr = found_rec.addr;
            H5_CHECKED_ASSIGN(obj_size, size_t, found_rec.len, hsize_t);
            filter_mask = found_rec.filter_mask;
        } 
        else {
            H5HF_huge_bt2_indir_rec_t found_rec;  
            H5HF_huge_bt2_indir_rec_t search_rec; 

            
            UINT64DECODE_VAR(id, search_rec.id, hdr->huge_id_size);

            
            if (H5B2_find(hdr->huge_bt2, &search_rec, &found, H5HF__huge_bt2_indir_found, &found_rec) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTFIND, FAIL, "can't check for object in v2 B-tree");
            if (!found)
                HGOTO_ERROR(H5E_HEAP, H5E_NOTFOUND, FAIL, "can't find object in v2 B-tree");

            
            obj_addr = found_rec.addr;
            H5_CHECKED_ASSIGN(obj_size, size_t, found_rec.len, hsize_t);
        } 
    }     

    
    if (hdr->filter_len > 0 || !is_read) {
        if (NULL == (read_buf = H5MM_malloc((size_t)obj_size)))
            HGOTO_ERROR(H5E_HEAP, H5E_NOSPACE, FAIL, "memory allocation failed for pipeline buffer");
    } 
    else
        read_buf = op_data;

    
    
    if (H5F_block_read(hdr->f, H5FD_MEM_FHEAP_HUGE_OBJ, obj_addr, (size_t)obj_size, read_buf) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_READERROR, FAIL, "can't read 'huge' object's data from the file");

    
    if (hdr->filter_len > 0) {
        H5Z_cb_t filter_cb; 
        size_t   read_size; 
        size_t   nbytes;    

        
        filter_cb.op_data = NULL;
        filter_cb.func    = NULL; 

        
        read_size = nbytes = obj_size;
        if (H5Z_pipeline(&(hdr->pline), H5Z_FLAG_REVERSE, &filter_mask, H5Z_NO_EDC, filter_cb, &nbytes,
                         &read_size, &read_buf) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTFILTER, FAIL, "input filter failed");
        obj_size = nbytes;
    } 

    
    if (is_read) {
        
        
        if (hdr->filter_len > 0)
            H5MM_memcpy(op_data, read_buf, (size_t)obj_size);
    } 
    else {
        
        if (op(read_buf, (size_t)obj_size, op_data) < 0) {
            
            read_buf = H5MM_xfree(read_buf);

            
            HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "application's callback failed");
        } 
    }     

done:
    
    if (read_buf && read_buf != op_data)
        read_buf = H5MM_xfree(read_buf);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__huge_write(H5HF_hdr_t *hdr, const uint8_t *id, const void *obj)
{
    haddr_t obj_addr  = HADDR_UNDEF; 
    size_t  obj_size  = 0;           
    herr_t  ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(hdr);
    assert(id);
    assert(obj);

    
    if (hdr->filter_len > 0)
        HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL,
                    "modifying 'huge' object with filters not supported yet");

    
    id++;

    
    if (hdr->huge_ids_direct) {
        
        H5F_addr_decode(hdr->f, &id, &obj_addr);
        H5F_DECODE_LENGTH(hdr->f, id, obj_size);
    }
    else {
        H5HF_huge_bt2_indir_rec_t found_rec;     
        H5HF_huge_bt2_indir_rec_t search_rec;    
        bool                      found = false; 

        
        assert(H5_addr_defined(hdr->huge_bt2_addr));

        
        if (NULL == hdr->huge_bt2) {
            
            if (NULL == (hdr->huge_bt2 = H5B2_open(hdr->f, hdr->huge_bt2_addr, hdr->f)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTOPENOBJ, FAIL,
                            "unable to open v2 B-tree for tracking 'huge' heap objects");
        }

        
        UINT64DECODE_VAR(id, search_rec.id, hdr->huge_id_size);

        
        if (H5B2_find(hdr->huge_bt2, &search_rec, &found, H5HF__huge_bt2_indir_found, &found_rec) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTFIND, FAIL, "can't check for object in v2 B-tree");
        if (!found)
            HGOTO_ERROR(H5E_HEAP, H5E_NOTFOUND, FAIL, "can't find object in v2 B-tree");

        
        obj_addr = found_rec.addr;
        H5_CHECKED_ASSIGN(obj_size, size_t, found_rec.len, hsize_t);
    }

    
    
    if (H5F_block_write(hdr->f, H5FD_MEM_FHEAP_HUGE_OBJ, obj_addr, obj_size, obj) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_WRITEERROR, FAIL, "writing 'huge' object to file failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__huge_read(H5HF_hdr_t *hdr, const uint8_t *id, void *obj)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);
    assert(obj);

    
    if (H5HF__huge_op_real(hdr, id, true, NULL, obj) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "unable to operate on heap object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__huge_op(H5HF_hdr_t *hdr, const uint8_t *id, H5HF_operator_t op, void *op_data)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(id);
    assert(op);

    
    if (H5HF__huge_op_real(hdr, id, false, op, op_data) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTOPERATE, FAIL, "unable to operate on heap object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__huge_remove(H5HF_hdr_t *hdr, const uint8_t *id)
{
    H5HF_huge_remove_ud_t udata;               
    herr_t                ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(hdr->huge_bt2_addr));
    assert(id);

    
    if (NULL == hdr->huge_bt2) {
        
        if (NULL == (hdr->huge_bt2 = H5B2_open(hdr->f, hdr->huge_bt2_addr, hdr->f)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTOPENOBJ, FAIL,
                        "unable to open v2 B-tree for tracking 'huge' heap objects");
    } 

    
    id++;

    
    udata.hdr = hdr;

    
    if (hdr->huge_ids_direct) {
        if (hdr->filter_len > 0) {
            H5HF_huge_bt2_filt_dir_rec_t search_rec; 

            
            
            H5F_addr_decode(hdr->f, &id, &search_rec.addr);
            H5F_DECODE_LENGTH(hdr->f, id, search_rec.len);

            
            
            if (H5B2_remove(hdr->huge_bt2, &search_rec, H5HF__huge_bt2_filt_dir_remove, &udata) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "can't remove object from B-tree");
        } 
        else {
            H5HF_huge_bt2_dir_rec_t search_rec; 

            
            
            H5F_addr_decode(hdr->f, &id, &search_rec.addr);
            H5F_DECODE_LENGTH(hdr->f, id, search_rec.len);

            
            
            if (H5B2_remove(hdr->huge_bt2, &search_rec, H5HF__huge_bt2_dir_remove, &udata) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "can't remove object from B-tree");
        } 
    }     
    else {
        if (hdr->filter_len > 0) {
            H5HF_huge_bt2_filt_indir_rec_t search_rec; 

            
            UINT64DECODE_VAR(id, search_rec.id, hdr->huge_id_size);

            
            
            if (H5B2_remove(hdr->huge_bt2, &search_rec, H5HF__huge_bt2_filt_indir_remove, &udata) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "can't remove object from B-tree");
        } 
        else {
            H5HF_huge_bt2_indir_rec_t search_rec; 

            
            UINT64DECODE_VAR(id, search_rec.id, hdr->huge_id_size);

            
            
            if (H5B2_remove(hdr->huge_bt2, &search_rec, H5HF__huge_bt2_indir_remove, &udata) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "can't remove object from B-tree");
        } 
    }     

    
    hdr->huge_size -= udata.obj_len;
    hdr->huge_nobjs--;

    
    if (H5HF__hdr_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark heap header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__huge_term(H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->huge_bt2) {
        
        assert(H5_addr_defined(hdr->huge_bt2_addr));

        
        if (H5B2_close(hdr->huge_bt2) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree");
        hdr->huge_bt2 = NULL;
    } 

    
    if (H5_addr_defined(hdr->huge_bt2_addr) && hdr->huge_nobjs == 0) {
        
        assert(hdr->huge_size == 0);

        
        
        if (H5B2_delete(hdr->f, hdr->huge_bt2_addr, hdr->f, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDELETE, FAIL, "can't delete v2 B-tree");

        
        hdr->huge_bt2_addr    = HADDR_UNDEF;
        hdr->huge_next_id     = 0;
        hdr->huge_ids_wrapped = false;

        
        if (H5HF__hdr_dirty(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark heap header as dirty");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__huge_delete(H5HF_hdr_t *hdr)
{
    H5HF_huge_remove_ud_t udata;               
    H5B2_remove_t         op;                  
    herr_t                ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(hdr->huge_bt2_addr));
    assert(hdr->huge_nobjs);
    assert(hdr->huge_size);

    
    udata.hdr = hdr;

    
    if (hdr->huge_ids_direct) {
        if (hdr->filter_len > 0)
            op = H5HF__huge_bt2_filt_dir_remove;
        else
            op = H5HF__huge_bt2_dir_remove;
    } 
    else {
        if (hdr->filter_len > 0)
            op = H5HF__huge_bt2_filt_indir_remove;
        else
            op = H5HF__huge_bt2_indir_remove;
    } 

    
    if (H5B2_delete(hdr->f, hdr->huge_bt2_addr, hdr->f, op, &udata) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDELETE, FAIL, "can't delete v2 B-tree");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
