% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gompertz.R
\name{gompertz}
\alias{gompertz}
\alias{gompertz_curve}
\alias{gompertz_objfun}
\title{Gompertz Growth Model}
\usage{
gompertz(par, data, silent = TRUE, ...)

gompertz_curve(t, L1, L2, k, t1, t2)

gompertz_objfun(par, data)
}
\arguments{
\item{par}{a parameter list.}

\item{data}{a data list.}

\item{silent}{passed to \code{\link[RTMB]{MakeADFun}}.}

\item{\dots}{passed to \code{\link[RTMB]{MakeADFun}}.}

\item{t}{age (vector).}

\item{L1}{predicted length at age \code{t1}.}

\item{L2}{predicted length at age \code{t2}.}

\item{k}{growth coefficient.}

\item{t1}{age where predicted length is \code{L1}.}

\item{t2}{age where predicted length is \code{L2}.}
}
\value{
The \code{gompertz} function returns a TMB model object, produced by
\code{\link[RTMB]{MakeADFun}}.

The \code{gompertz_curve} function returns a numeric vector of predicted
length at age.

The \code{gompertz_objfun} function returns the negative log-likelihood as a
single number, describing the goodness of fit of \code{par} to the
\code{data}.
}
\description{
Fit a Gompertz growth model to otoliths and/or tags, using the Schnute
parametrization.
}
\details{
The main function \code{gompertz} creates a model object, ready for parameter
estimation. The auxiliary functions \code{gompertz_curve} and
\code{gompertz_objfun} are called by the main function to calculate the
regression curve and objective function value. The user can also call the
auxiliary functions directly for plotting and model exploration.

The \code{par} list contains the following elements:
\itemize{
  \item \code{log_L1}, predicted length at age \code{t1}
  \item \code{log_L2}, predicted length at age \code{t2}
  \item \code{log_k}, growth coefficient
  \item \code{log_sigma_min}, growth variability at the shortest observed
        length in the data
  \item \code{log_sigma_max} (*), growth variability at the longest observed
        length in the data
  \item \code{log_age} (*), age at release of tagged individuals (vector)
}

*: The parameter \code{log_sigma_max} can be omitted to estimate growth
variability that does not vary with length. The parameter vector
\code{log_age} can be omitted to fit to otoliths only.

The \code{data} list contains the following elements:
\itemize{
  \item \code{Aoto} (*), age from otoliths (vector)
  \item \code{Loto} (*), length from otoliths (vector)
  \item \code{Lrel} (*), length at release of tagged individuals (vector)
  \item \code{Lrec} (*), length at recapture of tagged individuals (vector)
  \item \code{liberty} (*), time at liberty of tagged individuals in years
        (vector)
  \item \code{t1}, age where predicted length is \code{L1}
  \item \code{t2}, age where predicted length is \code{L2}
}

*: The data vectors \code{Aoto} and \code{Loto} can be omitted to fit to
tagging data only. The data vectors \code{Lrel}, \code{Lrec}, and
\code{liberty} can be omitted to fit to otoliths only.
}
\note{
The Schnute parametrization used in \code{gompertz} reduces parameter
correlation and improves convergence reliability compared to the traditional
parametrization used in \code{\link{gompertzo}}. Therefore, the
\code{gompertz} parametrization can be recommended for general usage, as both
parametrizations produce the same growth curve. However, there can be some
use cases where the traditional parametrization (\code{Linf}, \code{k},
\code{tau}) is preferred over the Schnute parametrization (\code{L1},
\code{L2}, \code{k}).

Gompertz is a special case of the Richards (1959) model, where \eqn{b=0}. If
the best model fit of a \code{\link{richards}} model to a particular dataset
involves a very small estimated value of \eqn{b}, then the \code{gompertz}
model offers a preferable parametrization, as it produces the same curve
using fewer parameters.

The Gompertz (1825) growth model, as parametrized by Schnute (1981, Eq. 16)
predicts length at age as:

\deqn{\hat L_t ~=~ L_1\exp\!\left[\,\log(L_2/L_1)\,
      \frac{1-e^{-k(t-t_1)}}{1-e^{-k(t_2-t_1)}}\,\right]}{
      Lhat = L1 * exp(log(L2/L1) * (1-exp(-k*(t-t1))) / (1-exp(-k*(t2-t1))))}

The variability of length at age increases linearly with length,

\deqn{\sigma_L ~=~ \alpha \,+\, \beta \hat L}{sigma = alpha + beta * Lhat}

where the intercept is \eqn{\alpha=\sigma_{\min} - \beta L_{\min}}{alpha =
sigma_min - beta * L_min}, the slope is
\eqn{\beta=(\sigma_{\max}-\sigma_{\min}) / (L_{\max}-L_{\min})}{beta =
(sigma_max-sigma_min) / (L_max-L_min)}, and \eqn{L_{\min}}{L_min} and
\eqn{L_{\max}}{L_max} are the shortest and longest observed lengths in the
data. Alternatively, growth variability can be modelled as a constant
\eqn{\sigma_L=\sigma_{\min}}{sigma_L=sigma_min} that does not vary with
length, by omitting \code{log_sigma_max} from the parameter list (see above).

The negative log-likelihood objective function integrates (sums) the
likelihood components from the otoliths and tags,

\deqn{f ~=~ \sum_{i=1}^{N_\mathrm{oto}}\, 0.5\log(2\pi) \,+\,
      \log\sigma_i \,+\, \frac{(L_i-\hat L_i)^2}{2\sigma_i^2}}{f
      = sum(0.5*log(2*pi) + log(sigma_i) + ((L_i-Lhat_i)^2)/(2*sigma_i^2))}
\deqn{~~~~ ~+\,~ \sum_{j=1}^{N_\mathrm{tag}}\, 0.5\log(2\pi) \,+\,
      \log\sigma_j \,+\, \frac{(L_j-\hat L_j)^2}{2\sigma_j^2}}{
      + sum(0.5*log(2*pi) + log(sigma_j) + ((L_j-Lhat_j)^2)/(2*sigma_j^2))}
\deqn{~~~~ ~+\,~ \sum_{k=1}^{N_\mathrm{tag}}\, 0.5\log(2\pi) \,+\,
      \log\sigma_k \,+\, \frac{(L_k-\hat L_k)^2}{2\sigma_k^2}}{
      + sum(0.5*log(2*pi) + log(sigma_k) + ((L_k-Lhat_k)^2)/(2*sigma_k^2))}

where \eqn{L_i} are length measurements from the otolith data, \eqn{L_j} are
length measurements at tag release, and \eqn{L_k} are length measurements at
tag recapture. \if{html,latex}{\eqn{N_\mathrm{oto}} is the number of fish in
the otolith data and \eqn{N_\mathrm{tag}} is the number of fish in the
tagging data.}
}
\examples{
# Model 1: Fit to haddock otoliths

# Explore initial parameter values
plot(len~age, otoliths_had, xlim=c(0,18), ylim=c(0,105), pch=16,
     col="#0080a010")
x <- seq(1, 18, 0.1)
lines(x, gompertz_curve(x, L1=15, L2=70, k=0.4, t1=1, t2=10), lty=3)

# Prepare parameters and data
init <- list(log_L1=log(20), log_L2=log(70), log_k=log(0.1),
             log_sigma_min=log(3), log_sigma_max=log(3))
dat <- list(Aoto=otoliths_had$age, Loto=otoliths_had$len, t1=1, t2=10)
gompertz_objfun(init, dat)

# Fit model
model <- gompertz(init, dat)
fit <- nlminb(model$par, model$fn, model$gr,
              control=list(eval.max=1e4, iter.max=1e4))
report <- model$report()
sdreport <- sdreport(model)

# Plot results
Lhat <- with(report, gompertz_curve(x, L1, L2, k, t1, t2))
lines(x, Lhat, lwd=2, col=2)
legend("bottomright", c("initial curve","model fit"), col=c(1,2), lty=c(3,1),
       lwd=c(1,2), bty="n", inset=0.02, y.intersp=1.25)

# Model summary
report[c("L1", "L2", "k", "sigma_min", "sigma_max")]
fit[-1]
summary(sdreport)

# Plot 95\% prediction band
band <- pred_band(x, model)
areaplot::confplot(cbind(lower,upper)~age, band, xlim=c(0,18), ylim=c(0,100),
         ylab="len", col="mistyrose")
points(len~age, otoliths_had, pch=16, col="#0080a010")
lines(Lhat~age, band, lwd=2, col=2)
lines(lower~age, band, lty=1, lwd=0.5, col=2)
lines(upper~age, band, lty=1, lwd=0.5, col=2)

#############################################################################

# Model 2: Fit to skipjack otoliths and tags

# Explore initial parameter values
plot(len~age, otoliths_skj, xlim=c(0,4), ylim=c(0,100))
x <- seq(0, 4, 0.1)
points(lenRel~I(lenRel/60), tags_skj, col=4)
points(lenRec~I(lenRel/60+liberty), tags_skj, col=3)
lines(x, gompertz_curve(x, L1=28, L2=74, k=1, t1=0, t2=4), lty=2)
legend("bottomright", c("otoliths","tag releases","tag recaptures",
       "initial curve"), col=c(1,4,3,1), pch=c(1,1,1,NA), lty=c(0,0,0,2),
       lwd=c(1.2,1.2,1.2,1), bty="n", inset=0.02, y.intersp=1.25)

# Prepare parameters and data
init <- list(log_L1=log(28), log_L2=log(74), log_k=log(1),
             log_sigma_min=log(3), log_sigma_max=log(3),
             log_age=log(tags_skj$lenRel/60))
dat <- list(Aoto=otoliths_skj$age, Loto=otoliths_skj$len,
            Lrel=tags_skj$lenRel, Lrec=tags_skj$lenRec,
            liberty=tags_skj$liberty, t1=0, t2=4)
gompertz_objfun(init, dat)

# Fit model
model <- gompertz(init, dat)
fit <- nlminb(model$par, model$fn, model$gr,
              control=list(eval.max=1e4, iter.max=1e4))
report <- model$report()
sdreport <- sdreport(model)

# Plot results
plot(len~age, otoliths_skj, xlim=c(0,4), ylim=c(0,100))
points(report$age, report$Lrel, col=4)
points(report$age+report$liberty, report$Lrec, col=3)
Lhat <- with(report, gompertz_curve(x, L1, L2, k, t1, t2))
lines(x, Lhat, lwd=2)
legend("bottomright", c("otoliths","tag releases","tag recaptures",
       "model fit"), col=c(1,4,3,1), pch=c(1,1,1,NA), lty=c(0,0,0,1),
       lwd=c(1.2,1.2,1.2,2), bty="n", inset=0.02, y.intersp=1.25)

# Model summary
report[c("L1", "L2", "k", "sigma_min", "sigma_max")]
fit[-1]
head(summary(sdreport), 5)

#############################################################################

# Model 3: Fit to skipjack otoliths only

init <- list(log_L1=log(28), log_L2=log(74), log_k=log(1),
             log_sigma_min=log(3), log_sigma_max=log(3))
dat <- list(Aoto=otoliths_skj$age, Loto=otoliths_skj$len, t1=0, t2=4)
model <- gompertz(init, dat)
fit <- nlminb(model$par, model$fn, model$gr,
              control=list(eval.max=1e4, iter.max=1e4))
model$report()[c("L1", "L2", "k", "sigma_min", "sigma_max")]

#############################################################################

# Model 4: Fit to skipjack otoliths only,
# but now estimating constant sigma instead of sigma varying by length

# We do this by omitting log_sigma_max
init <- list(log_L1=log(28), log_L2=log(74), log_k=log(1),
             log_sigma_min=log(3))
dat <- list(Aoto=otoliths_skj$age, Loto=otoliths_skj$len, t1=0, t2=4)
model <- gompertz(init, dat)
fit <- nlminb(model$par, model$fn, model$gr,
                    control=list(eval.max=1e4, iter.max=1e4))
model$report()[c("L1", "L2", "k", "sigma_min")]

#############################################################################

# Model 5: Fit to skipjack tags only

init <- list(log_L1=log(28), log_L2=log(74), log_k=log(1),
             log_sigma_min=log(3), log_sigma_max=log(3),
             log_age=log(tags_skj$lenRel/60))
dat <- list(Lrel=tags_skj$lenRel, Lrec=tags_skj$lenRec,
            liberty=tags_skj$liberty, t1=0, t2=4)
model <- gompertz(init, dat)
fit <- nlminb(model$par, model$fn, model$gr,
                   control=list(eval.max=1e4, iter.max=1e4))
model$report()[c("L1", "L2", "k", "sigma_min", "sigma_max")]

}
\references{
Gompertz, B. (1825).
On the nature of the function expressive of the law of human mortality, and
on a new mode of determining the value of life contingencies.
\emph{Philosophical Transactions of the Royal Society}, \bold{115}, 513-583.

Schnute, J. (1981).
A versatile growth model with statistically stable parameters.
\emph{Canadian Journal of Fisheries and Aquatic Science}, \bold{38},
1128-1140.
\doi{10.1139/f81-153}.

The \code{\link{fishgrowth-package}} help page includes references describing
the parameter estimation method.
}
\seealso{
\code{\link{gcm}}, \code{gompertz}, \code{\link{gompertzo}},
\code{\link{richards}}, \code{\link{richards}}, \code{\link{schnute3}},
\code{\link{vonbert}}, and \code{\link{vonberto}} are alternative growth
models.

\code{\link{pred_band}} calculates a prediction band for a fitted growth
model.

\code{\link{otoliths_had}}, \code{\link{otoliths_skj}}, and
\code{\link{tags_skj}} are example datasets.

\code{\link{fishgrowth-package}} gives an overview of the package.
}
