% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv_boost.R
\name{cv_boost_imputed}
\alias{cv_boost_imputed}
\title{Cross-validated boosting on already-imputed data}
\usage{
cv_boost_imputed(
  X_train_list,
  y_train_list,
  X_val_list,
  y_val_list,
  X_full,
  y_full,
  ny = 0.1,
  mstop = 250,
  type = c("gaussian", "logistic"),
  MIBoost = TRUE,
  pool = TRUE,
  pool_threshold = 0,
  show_progress = TRUE,
  center = c("auto", "off", "force")
)
}
\arguments{
\item{X_train_list}{A list of length \eqn{k}. Element \code{i} is itself a
list of length \eqn{M} containing the \eqn{n_{train} \times p} numeric
design matrices for each imputed dataset in CV fold \code{i}.}

\item{y_train_list}{A list of length \eqn{k}. Element \code{i} is a list of
length \eqn{M}, where each element is a length-\eqn{n_{train}} numeric
response vector aligned with \code{X_train_list[[i]][[m]]}.}

\item{X_val_list}{A list of length \eqn{k}. Element \code{i} is a list of
length \eqn{M} containing the \eqn{n_{val} \times p} numeric validation
matrices matched to the corresponding imputed training dataset
in fold \code{i}.}

\item{y_val_list}{A list of length \eqn{k}. Element \code{i} is a list of
length \eqn{M}, where each element is a length-\eqn{n_{val}} continuous
response vector aligned with \code{X_val_list[[i]][[m]]}.}

\item{X_full}{A list of length \eqn{M} with the \eqn{n \times p} numeric
full-data design matrices (one per imputed dataset) used to fit
the final model.}

\item{y_full}{A list of length \eqn{M}, where each element is a
length-\eqn{n} continuous response vector corresponding to the
imputed dataset in \code{X_full}.}

\item{ny}{Learning rate. Defaults to \code{0.1}.}

\item{mstop}{Maximum number of boosting iterations to evaluate during
cross-validation. The selected \code{mstop} is the value that minimizes
the mean CV error over \code{1:mstop}. Default is \code{250}.}

\item{type}{Type of loss function. One of:
\code{"gaussian"} (mean squared error) for continuous responses,
or \code{"logistic"} (binomial deviance) for binary responses.}

\item{MIBoost}{Logical. If \code{TRUE}, applies the MIBoost algorithm,
which enforces uniform variable selection across all imputed datasets. If
\code{FALSE}, variables are selected independently within each imputed
dataset, and pooling is governed by \code{pool_threshold}.}

\item{pool}{Logical. If \code{TRUE}, models across the \eqn{M} imputed
datasets are aggregated into a single final model. If \code{FALSE},
\eqn{M} separate models are returned.}

\item{pool_threshold}{Only used when \code{MIBoost = FALSE} and \code{pool = TRUE}.
Controls the pooling rule when aggregating the \eqn{M} models obtained from
the imputed datasets into a single final model. A candidate variable is
included only if it is selected in at least \code{pool_threshold} (a value
in (0, 1]) proportion of the imputed datasets; coefficients of all other
variables are set to zero. A value of \code{0} corresponds to
estimate-averaging, while values \code{> 0} correspond to
selection-frequency thresholding.}

\item{show_progress}{Logical; print fold-level progress and summary timings.
Default \code{TRUE}.}

\item{center}{One of \code{c("auto", "off", "force")}. Controls
centering of \code{X} within each imputed dataset.
With \code{"auto"} (recommended), centering is applied only if the training
matrix is not already centered. With \code{"force"}, centering is always
applied. With \code{"off"}, centering is skipped. If \code{X_val_list} is
provided, validation sets are centered using the means from the
corresponding training set.}
}
\value{
A list with:
\itemize{
\item \code{CV_error}: numeric vector of length \code{mstop} with the mean
cross-validated loss across folds (and imputations).
\item \code{best_mstop}: integer index of the minimizing entry in
\code{CV_error}.
\item \code{final_model}: numeric vector of length \code{1 + p} containing
the intercept followed by \eqn{p} coefficients of the final pooled
model fitted at \code{best_mstop} on \code{X_full}/\code{y_full}.
}
}
\description{
Performs k-fold cross-validation for \code{\link{impu_boost}} to determine
the optimal value of \code{mstop} before fitting the final model on the
full dataset. This function should only be used when data have already
been imputed. In most cases, it is preferable to provide unimputed data
and use \code{\link{cv_boost_raw}} instead.
}
\details{
To avoid data leakage, each CV fold should first be split into training and
validation subsets, after which imputation is performed. For the final model,
all data should be imputed independently.

The recommended workflow is illustrated in the examples.

Centering affects only \code{X}; \code{y} is left unchanged. For
\code{type = "logistic"}, responses are treated as numeric \code{0/1}
via the logistic link. Validation loss is averaged over
imputations and then over folds.
}
\examples{
\dontshow{if (requireNamespace("mice", quietly = TRUE) && requireNamespace("miceadds", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
  set.seed(123)
  utils::data(booami_sim)
  k <- 2; M <- 2
  n <- nrow(booami_sim); p <- ncol(booami_sim) - 1
  folds <- sample(rep(seq_len(k), length.out = n))

  X_train_list <- vector("list", k)
  y_train_list <- vector("list", k)
  X_val_list   <- vector("list", k)
  y_val_list   <- vector("list", k)

  for (cv in seq_len(k)) {
    tr <- folds != cv
    va <- !tr
    dat_tr <- booami_sim[tr, , drop = FALSE]
    dat_va <- booami_sim[va, , drop = FALSE]
    pm_tr  <- mice::quickpred(dat_tr, method = "spearman", mincor = 0.30, minpuc = 0.60)
    imp_tr <- mice::mice(dat_tr, m = M, predictorMatrix = pm_tr, maxit = 1, printFlag = FALSE)
    imp_va <- mice::mice.mids(imp_tr, newdata = dat_va, maxit = 1, printFlag = FALSE)
    X_train_list[[cv]] <- vector("list", M)
    y_train_list[[cv]] <- vector("list", M)

    X_val_list[[cv]]   <- vector("list", M)
    y_val_list[[cv]]   <- vector("list", M)
    for (m in seq_len(M)) {
      tr_m <- mice::complete(imp_tr, m)
      va_m <- mice::complete(imp_va, m)
      X_train_list[[cv]][[m]] <- data.matrix(tr_m[, 1:p, drop = FALSE])
      y_train_list[[cv]][[m]] <- tr_m$y
      X_val_list[[cv]][[m]]   <- data.matrix(va_m[, 1:p, drop = FALSE])
      y_val_list[[cv]][[m]]   <- va_m$y
    }
  }

  pm_full  <- mice::quickpred(booami_sim, method = "spearman", mincor = 0.30, minpuc = 0.60)
  imp_full <- mice::mice(booami_sim, m = M, predictorMatrix = pm_full, maxit = 1, printFlag = FALSE)
  X_full <- lapply(seq_len(M),
  function(m) data.matrix(
  mice::complete(imp_full, m)[, 1:p, drop = FALSE]))
  y_full <- lapply(seq_len(M), function(m) mice::complete(imp_full, m)$y)

  res <- cv_boost_imputed(
    X_train_list, y_train_list,
    X_val_list,   y_val_list,
    X_full,       y_full,
    ny = 0.1, mstop = 50, type = "gaussian",
    MIBoost = TRUE, pool = TRUE, center = "auto",
    show_progress = FALSE
  )
\dontshow{invisible(utils::capture.output(str(res)))}
}

\donttest{
  set.seed(2025)
  utils::data(booami_sim)
  k <- 5; M <- 10
  n <- nrow(booami_sim); p <- ncol(booami_sim) - 1
  folds <- sample(rep(seq_len(k), length.out = n))

  X_train_list <- vector("list", k)
  y_train_list <- vector("list", k)
  X_val_list   <- vector("list", k)
  y_val_list   <- vector("list", k)
  for (cv in seq_len(k)) {
    tr <- folds != cv; va <- !tr
    dat_tr <- booami_sim[tr, , drop = FALSE]
    dat_va <- booami_sim[va, , drop = FALSE]
    pm_tr  <- mice::quickpred(dat_tr, method = "spearman", mincor = 0.20, minpuc = 0.40)
    imp_tr <- mice::mice(dat_tr, m = M, predictorMatrix = pm_tr, maxit = 5, printFlag = TRUE)
    imp_va <- mice::mice.mids(imp_tr, newdata = dat_va, maxit = 1, printFlag = FALSE)
    X_train_list[[cv]] <- vector("list", M)
    y_train_list[[cv]] <- vector("list", M)
    X_val_list[[cv]]   <- vector("list", M)
    y_val_list[[cv]]   <- vector("list", M)
    for (m in seq_len(M)) {
      tr_m <- mice::complete(imp_tr, m); va_m <- mice::complete(imp_va, m)
      X_train_list[[cv]][[m]] <- data.matrix(tr_m[, 1:p, drop = FALSE])
      y_train_list[[cv]][[m]] <- tr_m$y
      X_val_list[[cv]][[m]]   <- data.matrix(va_m[, 1:p, drop = FALSE])
      y_val_list[[cv]][[m]]   <- va_m$y
    }
  }
  pm_full  <- mice::quickpred(booami_sim, method = "spearman", mincor = 0.20, minpuc = 0.40)
  imp_full <- mice::mice(booami_sim, m = M, predictorMatrix = pm_full, maxit = 5, printFlag = TRUE)
  X_full <- lapply(seq_len(M),
  function(m) data.matrix(mice::complete(imp_full, m)[, 1:p, drop = FALSE]))
  y_full <- lapply(seq_len(M),
  function(m) mice::complete(imp_full, m)$y)

  res_heavy <- cv_boost_imputed(
    X_train_list, y_train_list,
    X_val_list,   y_val_list,
    X_full,       y_full,
    ny = 0.1, mstop = 250, type = "gaussian",
    MIBoost = TRUE, pool = TRUE, center = "auto",
    show_progress = TRUE
  )
  str(res_heavy)
}
\dontshow{\}) # examplesIf}
}
\references{
Kuchen, R. (2025). \emph{MIBoost: A Gradient Boosting Algorithm for Variable
Selection After Multiple Imputation}. arXiv:2507.21807.
\doi{10.48550/arXiv.2507.21807} \url{https://arxiv.org/abs/2507.21807}.
}
\seealso{
\code{\link{impu_boost}}, \code{\link{cv_boost_raw}}
}
