% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AGB.R
\name{AGB}
\alias{AGB}
\title{Estimate the above-ground biomass}
\usage{
AGB(
  x,
  measure.label,
  h,
  taxon = "taxon",
  dead = "dead",
  circumference = TRUE,
  su = "quadrat",
  area,
  coord,
  rm.dead = TRUE,
  check.spelling = FALSE,
  correct.taxon = TRUE,
  sort = TRUE,
  decreasing = TRUE,
  cache = FALSE,
  long = FALSE
)
}
\arguments{
\item{x}{A \code{data.frame} with the community sample data. See \strong{Details}.}

\item{measure.label}{Name of the column with circumference/diameter at breast height (cm).}

\item{h}{Name of the column with tree height (m). If omitted in \code{x}, height is estimated from \code{coord}.}

\item{taxon}{Name of the column with sampled taxon names. Default \code{"taxon"}. Use UTF-8; accents and special characters are not allowed.}

\item{dead}{String used to identify dead individuals. Default \code{"dead"}.}

\item{circumference}{Logical; if \code{TRUE} (default), CBH is assumed; otherwise DBH is assumed.}

\item{su}{Name of the column with sample-unit identifiers. Default \code{"quadrat"}.}

\item{area}{Numeric scalar: total sampled area (ha).}

\item{coord}{A vector \code{c(longitude, latitude)} or a two-column matrix/\code{data.frame} of site coordinates (decimal degrees).
Required when \code{h} is missing in \code{x}.}

\item{rm.dead}{Logical; if \code{TRUE} (default) dead individuals are removed prior to biomass calculation.}

\item{check.spelling}{Logical; if \code{TRUE}, near-matching taxon names are flagged for correction. Default \code{FALSE}.}

\item{correct.taxon}{Logical; if \code{TRUE} (default) taxon names are standardized via TNRS.}

\item{sort}{Logical; if \code{TRUE} (default) taxa are sorted by AGB.}

\item{decreasing}{Logical; if \code{TRUE} (default) sorting is in decreasing order.}

\item{cache}{Logical or \code{NULL}; if \code{TRUE} the function with write and use a cache to reduce online search of taxa names.
(\code{NULL} means use cache but clear it first). Default is \code{cache = FALSE}.}

\item{long}{Logical; if \code{FALSE} (default) the \code{$tree} component is omitted (see \strong{Value}).}
}
\value{
An object of class \code{"biomass"} with up to four components:
\itemize{
  \item \code{$tree}: individual records (taxon, wood density \code{wd}, assignment level, \code{AGB}, \code{C}, \code{CO2e});
  \item \code{$taxon}: AGB, C, and CO\eqn{_2}e per taxon (Mg per ha);
  \item \code{$total}: total AGB, C, and CO\eqn{_2}e per ha;
  \item \code{$WD.level}: percentage of wood-density assignments at \code{species}, \code{genus}, and sample-unit levels.
}
}
\description{
Estimate the above-ground biomass (AGB), carbon (C) and CO\eqn{_2} equivalent (CO\eqn{_2}e) of trees.
}
\details{
\code{AGB} is a wrapper around \pkg{BIOMASS} functions \code{getWoodDensity}, \code{computeAGB},
and \code{correctTaxo} (Rejou-Mechain et al., 2017). Tree biomasses are computed using the
allometric model of Chave et al. (2014).

It is expected that taxon names are binomials (genus and species). The function splits the taxon
string into two columns (\code{genus}, \code{species}) to retrieve wood density. Single-word
taxa (e.g., \code{Indet}, \code{Dead}) receive \code{species = NA}.

Wood density (g/cm\eqn{^3}) is obtained from a global database (~16,500 species). If a species
is missing, the genus mean is used; if the genus is missing, the sample-unit (\code{su}) mean is used.

The input \code{x} must include columns for sample-unit labels, taxon names, CBH/DBH, and optionally
height. If height is absent, \code{coord} is mandatory to allow height estimation.

The CBH/DBH column allows multi-stem notation such as \code{"17.1+8+5.7+6.8"}. The plus sign separates
stems; decimal separators can be points or commas; spaces around \code{"+"} are ignored. Column names
in \code{x} are coerced to lower case at runtime, making the function case-insensitive.

Measurement units: CBH/DBH in centimeters; height in meters.
}
\examples{
data <- quadrat.df
head(data)
\donttest{
resul1 <- AGB(
  data, measure.label = "CBH", h = "h", taxon = "Species", dead = "Morta",
  circumference = TRUE, su = "Plot", area = 0.0625, rm.dead = TRUE,
  check.spelling = FALSE, correct.taxon = TRUE, sort = TRUE,
  decreasing = TRUE, long = TRUE
)
head(resul1$tree)
resul1$taxon
resul1$total
resul1$WD.level

quadrat.default <- quadrat.df
colnames(quadrat.default) <- c("quadrat", "family", "taxon", "cbh", "h")

Resul2 <- AGB(x = quadrat.default, measure.label = "cbh",
              circumference = TRUE, h = "h", dead = "Morta", area = 0.0625)
head(Resul2$tree)
Resul2$taxon
Resul2$total
Resul2$WD.level
\dontrun{
Resul3 <- AGB(data, measure.label = "CBH", taxon = "Species", dead = "Morta",
              circumference = TRUE, su = "Plot", area = 0.0625,
              coord = c(-47.85, -21.17))
Resul3$taxon
Resul3$total
Resul3$WD.level
}
}
}
\references{
Boyle, B. et al. (2013) \emph{BMC Bioinformatics} 14:16.\cr
Chave, J. et al. (2014) \emph{Global Change Biology} 20(10):3177--3190.\cr
Rejou-Mechain, M. et al. (2017) \emph{Methods in Ecology and Evolution} 8:1163--1167.\cr
Zanne, A.E. et al. (2009) Global wood density database. Dryad.\cr
}
\seealso{
\pkg{BIOMASS} (\code{\link[BIOMASS]{getWoodDensity}},
  \code{\link[BIOMASS]{computeAGB}}, \code{\link[BIOMASS]{correctTaxo}})
}
