% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/KTPsi.r
\name{KTPsi}
\alias{KTPsi}
\alias{KTPsi.start}
\alias{KTPsi.step}
\alias{KTPsi.final}
\alias{KTPsi.stop}
\title{An implementation of Kontsevich and Tyler \eqn{\Psi} algorithm.}
\usage{
KTPsi(
  domains = list(slopes = 1:5, thresholds = 20:40, fps = c(0, 0.025, 0.05, 0.1, 0.2), fns
    = c(0, 0.025, 0.05, 0.1, 0.2)),
  priors = list(slopes = rep(1, length(domains$slopes))/length(domains$slopes),
    thresholds = rep(1, length(domains$thresholds))/length(domains$thresholds), fps =
    rep(1, length(domains$fps))/length(domains$fps), fns = rep(1,
    length(domains$fns))/length(domains$fns)),
  stimValues = 17:40,
  stopType = "N",
  stopValue = 140,
  maxPresentations = 200,
  minInterStimInterval = NA,
  maxInterStimInterval = NA,
  verbose = 0,
  makeStim,
  ...
)

KTPsi.start(
  domains = list(slopes = 1:5, thresholds = 20:40, fps = c(0, 0.025, 0.05, 0.1, 0.2), fns
    = c(0, 0.025, 0.05, 0.1, 0.2)),
  priors = list(slopes = rep(1, length(domains$slopes))/length(domains$slopes),
    thresholds = rep(1, length(domains$thresholds))/length(domains$thresholds), fps =
    rep(1, length(domains$fps))/length(domains$fps), fns = rep(1,
    length(domains$fns))/length(domains$fns)),
  stimValues = 17:40,
  stopType = "N",
  stopValue = 140,
  maxPresentations = 200,
  minInterStimInterval = NA,
  maxInterStimInterval = NA,
  verbose = 0,
  makeStim,
  ...
)

KTPsi.step(state, nextStim = NULL, fixedStimValue = NA)

KTPsi.final(state, method = "expectation")

KTPsi.stop(state)
}
\arguments{
\item{domains}{A list of 4 vectors:
\itemize{
\item \code{slopes} The valid slopes in the domain of psychometric functions.
\item \code{thresholds} The valid thresholds in the domain of psychometric functions.
\item \code{fps} The valid upper asymptotes (false positives) in the domain of psychometric functions.
\item \code{fns} The valid lower asymptotes (false negatives) in the domain of psychometric functions.
}}

\item{priors}{A list of 4 vectors:
\itemize{
\item \code{slopes} The prior probability vector for \code{domains$slopes}.
\item \code{thresholds} The prior probability vector for \code{domains$thresholds}.
\item \code{fps} The prior probability vector for \code{domains$fps}.
\item \code{fns} The prior probability vector for \code{domains$fns}.
}

Each prior should the same length as its \code{domains} counterpart and sum to 1.}

\item{stimValues}{Vector of allowable stimulus values.}

\item{stopType}{\code{N}, for number of presentations and \code{H}, for the entropy  of the pdf.}

\item{stopValue}{Value for number of presentations (\code{stopType=N}), or Entropy (\code{stopType=H}).}

\item{maxPresentations}{Maximum number of presentations regardless of \code{stopType}.}

\item{minInterStimInterval}{If both \code{minInterStimInterval} and \code{maxInterStimInterval}
are not \code{NA}, then between each stimuli there is a random wait period drawn uniformly
between \code{minInterStimInterval} and \code{maxInterStimInterval}.}

\item{maxInterStimInterval}{\code{minInterStimInterval}.}

\item{verbose}{\code{verbose=0} does nothing, \code{verbose=1} stores pdfs for returning,
and \code{verbose=2} stores pdfs and also prints each presentation.}

\item{makeStim}{A function that takes a stimulus value and numPresentations and returns an OPI datatype
ready for passing to opiPresent. See examples.}

\item{...}{Extra parameters to pass to the opiPresent function}

\item{state}{Current state of the KTPsi as returned by (eg) \code{KTPsi.start}.}

\item{nextStim}{The next stimulus to present in a suitable format for passing to \code{\link{opiPresent}}}

\item{fixedStimValue}{Currently ignored.}

\item{method}{Either \code{"expectation"} or \code{"MAP"}.}
}
\value{
\subsection{Single location}{

\code{KTPsi} returns a list containing
\itemize{
\item \code{npres} Total number of presentations used.
\item \code{respSeq} Response sequence stored as a matrix: row 1 is dB values of stimuli, row 2 is 1/0 for seen/not-seen, row 3 is fixated 1/0 (always 1 if \code{checkFixationOK} not present in stim objects returned from \code{makeStim}).
\item \code{pdfs} If \code{verbose} is bigger than 0, then this is a list of the pdfs used for each presentation, otherwise NULL.
\item \code{final} The mean/median/mode of the final pdf, depending on \code{stimChoice}, which is the determined threshold.
\item \code{opiResp} A list of responses received from each successful call to \code{opiPresent} within \code{KTPsi}.
}
}

\subsection{Multilple locations}{

\code{KTPsi.start} returns a list that can be passed to \code{KTPsi.step}, \code{KTPsi.stop}, and \code{KTPsi.final}. It represents the state of a KTPsi at a single location at a point in time and contains the following.
\itemize{
\item \code{name} \code{KTPsi}
\item A copy of all of the parameters supplied to KTPsi.start: \code{domains}, \code{priors}, \code{stimValues}, \code{stopType}, \code{stopValue}, \code{maxPresentations}, \code{makeStim} and \code{opiParams}.
\item \code{psi} A matrix where \code{psi[domain_index, stim]} is the probability of seeing \code{stim} assuming the psychometric function for the domain index \code{domain_index}.
\item \code{labels} A text representation of \code{psi[domain_index, ]}, or the the psychometric function for the domain index \code{domain_index}.
\item \code{pdf} Current pdf: vector of probabilities the same length as product of lengths of \code{domain} elements.
\item \code{numPresentations} The number of times \code{KTPsi.step} has been called on this state.
\item \code{stimuli} A vector containing the stimuli used at each call of \code{KTPsi.step}.
\item \code{responses} A vector containing the responses received at each call of \code{KTPsi.step}.
\item \code{responseTimes} A vector containing the response times received at each call of \code{KTPsi.step}.
\item \code{fixated} A vector containing TRUE/FALSE if fixation was OK according to \code{checkFixationOK} for each call of \code{KTPsi.step} (defaults to TRUE if \code{checkFixationOK} not present).
\item \code{opiResp} A list of responses received from each call to \code{opiPresent} within \code{KTPsi.step}.
}

\code{KTPsi.step} returns a list containing
\itemize{
\item \code{stat:} The new state after presenting a stimuli and getting a response.
\item \code{resp} The return from the \code{opiPresent} call that was made.
}

\code{KTPsi.stop} returns \code{TRUE} if the KTPsi has reached its stopping criteria, and \code{FALSE} otherwise.

\code{KTPsi.final} returns an estimate of threshold based on state based on its parameter.
}

TRUE if the \code{state} has reached its stopping criteria, and FALSE otherwise.
}
\description{
An implementation of Kontsevich and Tyler (Vis Res 39 (1999) pages 2729--2737
default parameterised for Standard Automated Perimetry.
based on
A. Turpin, D. Jankovic and A.M. McKendrick,
"Identifying Steep Psychometric Function Slope Quickly in Clinical Applications",
Vision Research, 50(23). November 2010. Pages 2476-2485
}
\details{
The assumed psychometric function is the cumulative Gaussian:
\deqn{\mbox{fp}+(1-\mbox{fp}-\mbox{fn})(1-\mbox{pnorm}(x, \mbox{threshold}, \mbox{slope})}
hence \code{domain$slopes} are standard deviations and \code{domain$thresholds} are the mean.

While it is assumed that \code{domains$thresholds} and \code{stimValues} are in dB, this need not be the case.
As long as the \code{makeStim} function converts \code{stimValues} into cd/\eqn{\mbox{m}^2}{m^2}
for the \code{opiPresent} function, then any units should work.

The \code{checkFixationOK} function is called (if present in stim made from \code{makeStim})
after each presentation, and if it returns FALSE, the pdf for that state is not changed
(ie the presentation is ignored), but the stim, number of presentations etc is recorded in
the state.

If more than one KTPsi is to be interleaved (for example, testing multiple locations), then thePsi
\code{KTPsi.start}, \code{KTPsi.step}, \code{KTPsi.stop} and \code{KTPsi.final} calls can maintain
the state of the KTPsi after each presentation, and should be used. If only a single KTPsi is
required, then the simpler \code{KTPsi} function can be used, which is a wrapper for the four functions
that maintain state. See examples below.
}
\examples{
chooseOpi("SimGaussian")
if(!is.null(opiInitialize(sd = 2)$err))
  stop("opiInitialize failed")

     # This section is for single location KTPsi
     # Stimulus is Size III white-on-white as in the HFA
makeStim <- function(db, n) {
  s <- list(x=9, y=9, level=dbTocd(db), size=0.43, color="white",
            duration=200, responseWindow=1500, checkFixationOK=NULL)
  class(s) <- "opiStaticStimulus"
  return(s)
}

KTPsi(makeStim = makeStim, stopType="H", stopValue=  3, tt=30, fpr=0.03)
KTPsi(makeStim = makeStim, stopType="N", stopValue= 27, verbose = 0, tt=30, fpr=0.03)

     # For multiple locations...
\dontrun{
states <- lapply(1:10, function(loc) KTPsi.start(makeStim = makeStim))
unfinished <- 1:10
while (length(unfinished) > 0) {
     loc <- unfinished[[1]]
     states[[loc]] <- KTPsi.step(states[[loc]])$state
     if (KTPsi.stop(states[[loc]]))
         unfinished <- unfinished[-1]
}
}

}
\references{
Kontsevich and Tyler. Vision Research 39 (1999) pages 2729--2737.

A. Turpin, D. Jankovic and A.M. McKendrick,
"Identifying Steep Psychometric Function Slope Quickly in Clinical Applications",
Vision Research, 50(23). November 2010. Pages 2476-2485

A. Turpin, P.H. Artes and A.M. McKendrick "The Open Perimetry Interface: An enabling tool for
clinical visual psychophysics", Journal of Vision 12(11) 2012.
}
\seealso{
\code{\link{dbTocd}}, \code{\link{opiPresent}}
}
