% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nonparametric_fits.R
\name{growth.gcFitSpline}
\alias{growth.gcFitSpline}
\title{Perform a smooth spline fit on growth data}
\usage{
growth.gcFitSpline(
  time,
  data,
  gcID = "undefined",
  control = growth.control(biphasic = FALSE)
)
}
\arguments{
\item{time}{Vector of the independent variable (usually time).}

\item{data}{Vector of dependent variable (usually: growth values).}

\item{gcID}{(Character) The name of the analyzed sample.}

\item{control}{A \code{grofit.control} object created with \code{\link{growth.control}},
defining relevant fitting options.}

\item{biphasic}{(Logical) Shall \code{growth.gcFitSpline} try to extract growth
parameters for two different growth phases (as observed with, e.g., diauxic shifts)
(\code{TRUE}) or not (\code{FALSE})?}
}
\value{
A \code{gcFitSpline} object. The lag time is estimated as the intersection between the
tangent at the maximum slope and the horizontal line with \eqn{y = y_0}, where
\code{y0} is the first value of the dependent variable. Use \code{\link{plot.gcFitSpline}} to
visualize the spline fit and derivative over time.
\item{time.in}{Raw time values provided to the function as \code{time}.}
\item{data.in}{Raw growth data provided to the function as \code{data}.}
\item{raw.time}{Filtered time values used for the spline fit.}
\item{raw.data}{Filtered growth values used for the spline fit.}
\item{gcID}{(Character) Identifies the tested sample.}
\item{fit.time}{Fitted time values.}
\item{fit.data}{Fitted growth values.}
\item{parameters}{List of determined growth parameters.}
\itemize{
\item \code{A}: {Maximum growth.}
\item \code{dY}: {Difference in maximum growth and minimum growth.}
\item \code{mu}: {Maximum growth rate (i.e., maximum in first derivative of the spline).}
\item \code{tD}: {Doubling time.}
\item \code{t.max}: {Time at the maximum growth rate.}
\item \code{lambda}: {Lag time.}
\item \code{b.tangent}: {Intersection of the tangent at the maximum growth rate with the abscissa.}
\item \code{mu2}: {For biphasic growth: Growth rate of the second growth phase.}
\item \code{tD2}: {Doubling time of the second growth phase.}
\item \code{lambda2}: {For biphasic growth: Lag time determined for the second growth phase.}
\item \code{t.max2}: {For biphasic growth: Time at the maximum growth rate of the second growth phase.}
\item \code{b.tangent2}: {For biphasic growth: Intersection of the tangent at the maximum growth rate of the second growth phase with the abscissa.}
\item \code{integral}: {Area under the curve of the spline fit.}
}
\item{spline}{\code{smooth.spline} object generated by the \code{\link{smooth.spline}} function.}
\item{spline.deriv1}{list of time ('x') and growth ('y') values describing the first derivative of the spline fit.}
\item{reliable}{(Logical) Indicates whether the performed fit is reliable (to be set manually).}
\item{fitFlag}{(Logical) Indicates whether a spline fit was successfully performed on the data.}
\item{fitFlag2}{(Logical) Indicates whether a second growth phase was identified.}
\item{control}{Object of class \code{grofit.control} containing list of options passed to the function as \code{control}.}
}
\description{
\code{growth.gcFitSpline} performs a smooth spline fit on the dataset and determines
the highest growth rate as the global maximum in the first derivative of the spline.
}
\details{
If \code{biphasic = TRUE}, the following steps are performed to define a
second growth phase: \enumerate{ \item Determine local minima within the first
derivative of the smooth spline fit. \item Remove the 'peak' containing the highest
value of the first derivative (i.e., \eqn{mu_{max}}) that is flanked by two local
minima. \item Repeat the smooth spline fit and identification of maximum slope for
later time values than the local minimum after \eqn{mu_{max}}. \item Repeat the
smooth spline fit and identification of maximum slope for earlier time values than
the local minimum before \eqn{mu_{max}}. \item Choose the greater of the two
independently determined slopes as \eqn{mu_{max}2}. }
}
\examples{
# Create random growth dataset
rnd.dataset <- rdm.data(d = 35, mu = 0.8, A = 5, label = 'Test1')

# Extract time and growth data for single sample
time <- rnd.dataset$time[1,]
data <- rnd.dataset$data[1,-(1:3)] # Remove identifier columns

# Perform spline fit
TestFit <- growth.gcFitSpline(time, data, gcID = 'TestFit',
                 control = growth.control(fit.opt = 's'))

plot(TestFit)


}
\references{
Matthias Kahm, Guido Hasenbrink, Hella Lichtenberg-Frate, Jost Ludwig, Maik Kschischo (2010). \emph{grofit: Fitting Biological Growth Curves with R}. Journal of Statistical Software, 33(7), 1-21. DOI: 10.18637/jss.v033.i07
}
\seealso{
Other growth fitting functions: 
\code{\link{growth.drFit}()},
\code{\link{growth.gcBootSpline}()},
\code{\link{growth.gcFitLinear}()},
\code{\link{growth.gcFitModel}()},
\code{\link{growth.gcFit}()},
\code{\link{growth.workflow}()}
}
\concept{growth fitting functions}
