/*-
 * Copyright (c) 2002 Jaromir Dolecek <jdolecek@NetBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jaromir Dolecek nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE JAROMIR DOLECEK AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/* $Id: kqueuemodule.c,v 1.2 2002/11/24 09:19:42 dolecek Exp $ */

#include <Python.h>

#include <sys/event.h>

/* -=-=-=-=-=-=-=-=-=-=-=-= KEvent Object =-=-=-=-=-=-=-=-=-=-=-= */
staticforward PyTypeObject kqueue_keventType;

typedef struct {
	PyObject_HEAD
	struct kevent kev;
} kqueue_keventObject;

static PyObject *
kqueue_new_kevent(PyObject *self, PyObject *args)
{
	kqueue_keventObject *keo;
	long ident=0, filter=0, flags=0, fflags=0;
	LONG_LONG data=0;
	PyObject *udata = Py_None;	/* use None by default */

	/* ident, filter, flags is required, rest is optional */
	if (!PyArg_ParseTuple(args, "lll|lLO",
		&ident, &filter, &flags, &fflags, &data, &udata)) {
		return NULL;
	}

	keo = PyObject_New(kqueue_keventObject, &kqueue_keventType);
	memset(&keo->kev, 0, sizeof(keo->kev));
	
	/* setup args */
	keo->kev.ident = (intptr_t) ident;
	keo->kev.filter = (uint32_t) filter;
	keo->kev.flags = (uint32_t) flags;
	keo->kev.fflags = (uint32_t) fflags;
	keo->kev.data = (int64_t) data;
	Py_INCREF(udata);
	keo->kev.udata = (intptr_t) udata;

	return (PyObject *) keo;
}

static void
kqueue_kevent_dealloc(PyObject *self)
{
	kqueue_keventObject *keo = (void *)self;

	Py_DECREF((PyObject *) keo->kev.udata);

	/* poof! poof! */
	PyObject_Del(self);
}

#if 0
/* Definition of struct kevent on NetBSD */
struct kevent {
	uintptr_t	ident;		/* identifier for this event */
	uint32_t	filter;		/* filter for event */
	uint32_t	flags;		/* action flags for kqueue */
	uint32_t	fflags;		/* filter flag value */
	int64_t		data;		/* filter data value */
	intptr_t	udata;		/* opaque user data identifier */
};
#endif
static const char * const members[] = {
	"ident", "filter", "flags", "fflags", "data", "udata",
};

static PyObject *
kqueue_keventType_getattr(PyObject *self, char *name)
{
	kqueue_keventObject *keo = (void *)self;
	PyObject *v;

	if (strcmp(name, "ident") == 0)
		v = PyLong_FromLong((long) keo->kev.ident);
	else if (strcmp(name, "filter") == 0)
		v = PyInt_FromLong((long) keo->kev.filter);
	else if (strcmp(name, "flags") == 0)
		v = PyInt_FromLong((long) keo->kev.flags);
	else if (strcmp(name, "fflags") == 0)
		v = PyInt_FromLong((long) keo->kev.fflags);
	else if (strcmp(name, "data") == 0)
		v = PyLong_FromLongLong((LONG_LONG) keo->kev.data);
	else if (strcmp(name, "udata") == 0) {
		v = (PyObject *) keo->kev.udata;
		Py_INCREF(v);
	} else if (strcmp(name, "__members__") == 0) {
		/* this is needed to tell which attributes we have */
		int i = sizeof(members) / sizeof(char *);
		PyObject *list;

		list = PyList_New(i);
		if (!list)
			return NULL;

		for(i=0; members[i]; i++) {
			v = PyString_FromString(members[i]);
			if (!v || PyList_SetItem(list, i, v) < 0) {
				Py_DECREF(list);
				return NULL;
			}
		}

		v = list;
	} else {
		PyErr_SetString(PyExc_AttributeError, name);
		v = NULL;
	}

	return v;
}

/*
 * Set kevent structure parameters. We allow either (int) or (long)
 * for numeric arguments. udata keeps the pointer to Python
 * object.
 */
static int
kqueue_keventType_setattr(PyObject *self, char *name, PyObject *v)
{
	kqueue_keventObject *keo = (void *) self;

	if (v == NULL) {
		PyErr_SetString(PyExc_TypeError,
			"can't delete kevent attributes");
		return -1;
	}

	if (strcmp(name, "ident") == 0) {
		/* allow either int or long */
		if (PyLong_Check(v))
			keo->kev.ident = (intptr_t) PyLong_AsLong(v);
		else if (PyInt_Check(v))
			keo->kev.ident = (intptr_t) PyInt_AsLong(v);
		else {
			PyErr_SetString(PyExc_TypeError, "ident must be long or int");
			return -1;
		}
	} else if (strcmp(name, "filter") == 0) {
		/* allow either int or long */
		if (PyLong_Check(v))
			keo->kev.filter = (uint32_t) PyLong_AsLong(v);
		else if (PyInt_Check(v))
			keo->kev.filter = (uint32_t) PyInt_AsLong(v);
		else {
			PyErr_SetString(PyExc_TypeError, "filter must be long or int");
			return -1;
		}
	} else if (strcmp(name, "flags") == 0) {
		/* allow either int or long */
		if (PyLong_Check(v))
			keo->kev.flags = (uint32_t) PyLong_AsLong(v);
		else if (PyInt_Check(v))
			keo->kev.flags = (uint32_t) PyInt_AsLong(v);
		else {
			PyErr_SetString(PyExc_TypeError, "flags must be long or int");
			return -1;
		}
	} else if (strcmp(name, "fflags") == 0) {
		/* allow either int or long */
		if (PyLong_Check(v))
			keo->kev.fflags = (uint32_t) PyLong_AsLong(v);
		else if (PyInt_Check(v))
			keo->kev.fflags = (uint32_t) PyInt_AsLong(v);
		else {
			PyErr_SetString(PyExc_TypeError, "fflags must be long or int");
			return -1;
		}
	} else if (strcmp(name, "data") == 0) {
		/* allow either int or long */
		if (PyLong_Check(v))
			keo->kev.data = (int64_t) PyLong_AsLongLong(v);
		else if (PyInt_Check(v))
			keo->kev.data = (int64_t) PyInt_AsLong(v);
		else {
			PyErr_SetString(PyExc_TypeError, "data must be long or int");
			return -1;
		}
	} else if (strcmp(name, "udata") == 0) {
		/* any Python object, store pointer to it */
		Py_DECREF((PyObject *) keo->kev.udata);
		Py_INCREF(v);
		keo->kev.udata = (intptr_t) v;
	} else {
		/* some other attribute, error out */
		PyErr_SetString(PyExc_AttributeError, name);
		return -1;
	}

	return (0);
}

/*
 * Nice external representation of the object, for convenience only.
 */
static PyObject *
kqueue_keventType_repr(PyObject *self)
{
	kqueue_keventObject *keo = (void *) self;
	char buf[512], *urep;
	PyObject *uo;

	uo = PyObject_Repr((PyObject *) keo->kev.udata);
	urep = PyString_AsString(uo);

	PyOS_snprintf(buf, sizeof(buf),
		"<kevent object at %p, ident=0x%lx, filter=%u, flags=0x%x, fflags=0x%x, data=%lld, udata=%s>",
		keo, (long) keo->kev.ident, (unsigned int) keo->kev.filter,
		(unsigned int) keo->kev.flags,
		(unsigned int) keo->kev.fflags,
		(long long int) keo->kev.data,
		urep);
	Py_DECREF(uo);

	return PyString_FromString(buf);
}

static PyTypeObject kqueue_keventType = {
    PyObject_HEAD_INIT(NULL)
    0,
    "kevent",
    sizeof(kqueue_keventObject),
    0,
    kqueue_kevent_dealloc,	/*tp_dealloc*/
    0,				/*tp_print*/
    kqueue_keventType_getattr,	/*tp_getattr*/
    kqueue_keventType_setattr,	/*tp_setattr*/
    0,				/*tp_compare*/
    kqueue_keventType_repr,	/*tp_repr*/
    0,				/*tp_as_number*/
    0,				/*tp_as_sequence*/
    0,				/*tp_as_mapping*/
    0,				/*tp_hash */
};

/* -=-=-=-=-=-=-=-=-=-=-=-= KQueue Object =-=-=-=-=-=-=-=-=-=-=-= */
staticforward PyTypeObject kqueue_kqueueType;

typedef struct {
	PyObject_HEAD
	int kq;			/* kqueue descriptor */

	PyObject *_kevl;	/* internal list of registered events */
} kqueue_kqueueObject;

static PyObject *
kqueue_new_kqueue(PyObject *self, PyObject *args)
{
	int kq;
	kqueue_kqueueObject *kqo;

	kq = kqueue();
	if (kq < 0) {
		PyErr_SetFromErrno(PyExc_OSError);
		return NULL;
	}

	kqo = PyObject_New(kqueue_kqueueObject, &kqueue_kqueueType);
	if (!kqo)
		return NULL;

	kqo->_kevl = PyList_New(0);
	if (!kqo->_kevl) {
		close(kq);
		PyObject_Del(kqo);
		return NULL;
	}

	kqo->kq = kq;

	return (PyObject *) kqo;
}

static void
kqueue_kqueue_dealloc(PyObject *self)
{
	kqueue_kqueueObject *kq = (kqueue_kqueueObject *)self;

	close(kq->kq);
	Py_DECREF(kq->_kevl);
	PyObject_Del(self);
}

/*
 * fileno() implemented along the Python file descriptor fileno().
 */
static PyObject *
kqueue_kqueue_fileno(PyObject *self)
{
	kqueue_kqueueObject *kq = (kqueue_kqueueObject *)self;
	
	return PyInt_FromLong((long) kq->kq);
}

/*
 * Add/delete given kevent from internal kqueue object list. This
 * is necessary to hold correct reference count to embedded
 * udata Python object, since kevent() call can return new kevents
 * with new reference to the embedded udata object.
 * We try to remove references on EV_DELETE request or when EV_ONESHOT
 * kevent is received, but some events might still fall through
 * cracks.
 * Yeah, this is lame :(
 */
static void
kqueue_kevent_collect(kqueue_kqueueObject *kq, kqueue_keventObject *ke,
	int rem)
{
	/* don't bother if udata is None */
	if ((PyObject *) ke->kev.udata == Py_None)
		return;

	if (rem) {
		PyObject *v, *l = kq->_kevl;
		int i;
		int sz = PyList_Size(l);

		for(i=0; i < sz; i++) {
			v = PyList_GetItem(l, i);

			if ((intptr_t) v != (intptr_t) ke->kev.udata)
				continue;

			/* Replace with None. setitem() drops reference
			 * of the old item */
			Py_INCREF(Py_None);
			PyList_SetItem(l, i, Py_None);

			break;
		}
	} else {
		/* add */
		PyObject *l = kq->_kevl, *v;
		int sz = PyList_Size(l);
		int i;

		/* find empty slot */
		for(i=0; i < sz; i++) {
			v = PyList_GetItem(l, i);

			if (v == Py_None)
				break;
		}


		if (i == sz) {
			/* append() increases refcount */
			i = PyList_Append(l, (PyObject *) ke->kev.udata);
		} else {
			v = (PyObject *) ke->kev.udata;
			Py_INCREF(v);
			PyList_SetItem(l, i, v);
		}
	}
}

/*
 * Call kevent(2).
 */
static PyObject *
kqueue_kqueue_kevent(PyObject *self, PyObject *args)
{
	PyObject *chl = Py_None, *timo = Py_None, *list;
	kqueue_kqueueObject *kq = (kqueue_kqueueObject *)self;
	long evl_size = 0;
	struct kevent *ke;
	int i, n;
	struct timespec *tsp, ts;
	
	if (!PyArg_ParseTuple(args, "O|lO", &chl, &evl_size, &timo))
		return NULL;

	/* Push any changelist to system */
	if (chl == Py_None)
		;	/* no action */
	else if (PyList_Check(chl)) {
		PyObject *v;
		kqueue_keventObject *keo;
		int sz = PyList_Size(chl);

		for(i=0; i < sz; i++) {
			v = PyList_GetItem(chl, i);
			if (!v) {
				/* exception set by GetItem */
				return NULL;
			}

			if (!PyObject_TypeCheck(v, &kqueue_keventType)) {
				PyErr_SetString(PyExc_TypeError, "changelist must contain only kevent objects");
				return NULL;
			}

			keo = (void *)v;

			n = kevent(kq->kq, &keo->kev, 1, NULL, 0, NULL);
			if (n < 0) {
				PyErr_SetFromErrno(PyExc_OSError);
				return NULL;
			}

			if (keo->kev.flags & (EV_ADD|EV_DELETE)) {
				kqueue_kevent_collect(kq, keo,
					keo->kev.flags & EV_DELETE);
			}
		}
	} else if (PyObject_TypeCheck(chl, &kqueue_keventType)) {
		/* for convenience, support passing single kevent */
		kqueue_keventObject *keo = (void *)chl;
		int n;
		
		n = kevent(kq->kq, &keo->kev, 1, NULL, 0, NULL);
		if (n < 0) {
			PyErr_SetFromErrno(PyExc_OSError);
			return NULL;
		}

		if (keo->kev.flags & (EV_ADD|EV_DELETE)) {
			kqueue_kevent_collect(kq, keo,
				keo->kev.flags & EV_DELETE);
		}
	} else  {
		PyErr_SetString(PyExc_TypeError, "changelist must be list of kevent objects, or kevent object");
		return NULL;
	}


	if (evl_size <= 0) {
		/* no events requested, don't return anything */
		Py_INCREF(Py_None);
		return Py_None;
	}

	/* set timeout */
	if (timo == Py_None)
		tsp = NULL;
	else if (PyTuple_Check(timo) && PyTuple_Size(timo) == 2) {
		ts.tv_sec = (time_t) PyLong_AsLong(PyTuple_GetItem(timo, 0));
		ts.tv_nsec = (time_t) PyLong_AsLong(PyTuple_GetItem(timo, 1));
		tsp = &ts;
	} else if (PyLong_Check(timo) || PyInt_Check(timo)) {
		long to;

		if (PyLong_Check(timo))
			to = PyLong_AsLong(timo);
		else
			to = PyInt_AsLong(timo);
		ts.tv_sec = to / 1000000000;
		ts.tv_nsec = to % 1000000000;
		tsp = &ts;
	} else {
		PyErr_SetString(PyExc_ValueError, "timeout must be tuple (sec, nsec), or int/long nsec");
		return NULL;
	}

	/* allocate array for the retrieved events */
	ke = PyMem_Malloc(evl_size * sizeof(struct kevent));
	if (!ke) {
		PyErr_NoMemory();
		return NULL;
	}

	/* retrieve events */
	Py_BEGIN_ALLOW_THREADS
	n = kevent(kq->kq, NULL, 0, ke, (size_t) evl_size, tsp);
	Py_END_ALLOW_THREADS
	if (n < 0) {
		PyErr_SetFromErrno(PyExc_OSError);
		PyMem_Free(ke);
		return NULL;
	}

	/* put events in list and return the list */
	list = PyList_New(0);
	if (!list) {
		PyMem_Free(ke);
		return NULL;
	}

	for(i=0; i < n; i++) {
		kqueue_keventObject *keo;

		keo = PyObject_New(kqueue_keventObject, &kqueue_keventType);

		Py_INCREF((PyObject *) ke[i].udata);
		keo->kev = ke[i];

		PyList_Append(list, (PyObject *)keo);

		if (ke[i].flags & EV_ONESHOT)
			kqueue_kevent_collect(kq, keo, 1);

		Py_DECREF((PyObject *) keo);
	}

	PyMem_Free(ke);
	return list;
}

#ifdef KFILTER_BYFILTER
/*
 * Call KFILTER_BYFILTER ioctl - map kfilter code to name.
 */
static PyObject *
kqueue_kqueue_byfilter(PyObject *self, PyObject *args)
{
	kqueue_kqueueObject *kq = (kqueue_kqueueObject *)self;
	struct kfilter_mapping kfm;
	long filter;
	char buf[128];
	int error;

	if (!PyArg_ParseTuple(args, "l", &filter))
		return NULL;

	kfm.name = buf;
	kfm.len = sizeof(buf);
	kfm.filter = (uint32_t) filter;

	error = ioctl(kq->kq, KFILTER_BYFILTER, &kfm);
	if (error != 0) {
		PyErr_SetFromErrno(PyExc_OSError);
		return NULL;
	}

	return PyString_FromString(buf);
}
#endif /* KFILTER_BYFILTER */

#ifdef KFILTER_BYNAME
/*
 * Call KFILTER_BYNAME ioctl - map kfilter name to code.
 */
static PyObject *
kqueue_kqueue_byname(PyObject *self, PyObject *args)
{
	kqueue_kqueueObject *kq = (kqueue_kqueueObject *)self;
	struct kfilter_mapping kfm;
	char *name;
	int error;

	if (!PyArg_ParseTuple(args, "s", &name))
		return NULL;

	kfm.name = name;

	error = ioctl(kq->kq, KFILTER_BYNAME, &kfm);
	if (error != 0) {
		PyErr_SetFromErrno(PyExc_OSError);
		return NULL;
	}

	return PyLong_FromLong((long)kfm.filter);
}
#endif /* KFILTER_BYNAME */

static PyMethodDef kqueue_kqueueType_methods[] = {
	{"fileno", (PyCFunction)kqueue_kqueue_fileno, METH_NOARGS,
		"Return the kqueue descriptor." },
	{"kevent", (PyCFunction)kqueue_kqueue_kevent, METH_VARARGS,
		"Register or retrieve events." }, 
#ifdef KFILTER_BYFILTER
	{"kfilter_byfilter", (PyCFunction)kqueue_kqueue_byfilter, METH_VARARGS,
		"Retrieve name for specified kqfilter code" },
#endif
#ifdef KFILTER_BYNAME
	{"kfilter_byname", (PyCFunction)kqueue_kqueue_byname, METH_VARARGS,
		"Retrieve code for specified kqfilter name" },
#endif
	{NULL, NULL, 0, NULL}		/* sentinel */
};

static PyObject *
kqueue_kqueueType_getattr(PyObject *obj, char *name)
{
	return Py_FindMethod(kqueue_kqueueType_methods, obj, name);
}

static PyTypeObject kqueue_kqueueType = {
    PyObject_HEAD_INIT(NULL)
    0,
    "kqueue",
    sizeof(kqueue_kqueueObject),
    0,
    kqueue_kqueue_dealloc,	/*tp_dealloc*/
    0,				/*tp_print*/
    kqueue_kqueueType_getattr,	/*tp_getattr*/
    0,				/*tp_setattr*/
    0,				/*tp_compare*/
    0,				/*tp_repr*/
    0,				/*tp_as_number*/
    0,				/*tp_as_sequence*/
    0,				/*tp_as_mapping*/
    0,				/*tp_hash */
};

static PyMethodDef kqueue_methods[] = {
    {"kqueue", kqueue_new_kqueue, METH_NOARGS,
     "Create a new kqueue object."},
    {"EV_SET", kqueue_new_kevent, METH_VARARGS,
     "Create and initialize new kevent."},
    {NULL, NULL, 0, NULL}
};

/*
 * kqueue/kevent contants we want to export for use by programs.
 */

#define KQ_SYMBOL(n, sig)		{ #n, n, sig }

static const struct {
	char *name;
	long value;
	short hassign;
} kqueue_symbols[] = {
	/* filters */
	KQ_SYMBOL(EVFILT_READ, 1),
	KQ_SYMBOL(EVFILT_WRITE, 1),
	KQ_SYMBOL(EVFILT_AIO, 1),
	KQ_SYMBOL(EVFILT_VNODE, 1),
	KQ_SYMBOL(EVFILT_PROC, 1),
	KQ_SYMBOL(EVFILT_SIGNAL, 1),
#ifdef EVFILT_TIMER
	KQ_SYMBOL(EVFILT_TIMER, 1),
#endif

	/* actions */
	KQ_SYMBOL(EV_ADD, 0),
	KQ_SYMBOL(EV_DELETE, 0),
	KQ_SYMBOL(EV_ENABLE, 0),
	KQ_SYMBOL(EV_DISABLE, 0),

	/* flags */
	KQ_SYMBOL(EV_ONESHOT, 0),
	KQ_SYMBOL(EV_CLEAR, 0),

	/* returned values */
	KQ_SYMBOL(EV_EOF, 0),
	KQ_SYMBOL(EV_ERROR, 0),

	/* data/hint flags for EVFILT_{READ|WRITE}, shared with userspace */
	KQ_SYMBOL(NOTE_LOWAT, 0),

	/* data/hint flags for EVFILT_VNODE, shared with userspace */
	KQ_SYMBOL(NOTE_DELETE, 0),
	KQ_SYMBOL(NOTE_WRITE, 0),
	KQ_SYMBOL(NOTE_EXTEND, 0),
	KQ_SYMBOL(NOTE_ATTRIB, 0),
	KQ_SYMBOL(NOTE_LINK, 0),
	KQ_SYMBOL(NOTE_RENAME, 0),
	KQ_SYMBOL(NOTE_REVOKE, 0),

	/* data/hint flags for EVFILT_PROC, shared with userspace */
	KQ_SYMBOL(NOTE_EXIT, 0),
	KQ_SYMBOL(NOTE_FORK, 0),
	KQ_SYMBOL(NOTE_EXEC, 0),

	/* additional flags for EVFILT_PROC */
	KQ_SYMBOL(NOTE_TRACK, 0),
	KQ_SYMBOL(NOTE_TRACKERR, 0),
	KQ_SYMBOL(NOTE_CHILD, 0),

	{ NULL, 0 }
};

static void
kqueue_all_ins(PyObject* d)
{
	int i;
        PyObject* v;
	char *symb;
	static const char * const pykq_version = "2.0";

	for(i=0; (symb = kqueue_symbols[i].name); i++) {
		if (kqueue_symbols[i].hassign)
			v = PyInt_FromLong(kqueue_symbols[i].value);
		else {
			v = PyLong_FromUnsignedLong(
				(unsigned long) kqueue_symbols[i].value);
		}

	        if (!v)
			continue;
		
		PyDict_SetItemString(d, symb, v);
       		Py_DECREF(v);
	}

	/* export kqueue module version */
	v = PyString_FromString(pykq_version);
	if (v) {
		PyDict_SetItemString(d, "PYKQ_VERSION", v);
		Py_DECREF(v);
	}
}

DL_EXPORT(void)
initkqueue(void)
{
	PyObject *m, *d;

	kqueue_kqueueType.ob_type = &PyType_Type;
	kqueue_keventType.ob_type = &PyType_Type;

	/* Create the module and add the function */
	m = Py_InitModule3("kqueue", kqueue_methods,
		"KQueue provides a generic method of notifying the user when\
an event happens\n\
or a condition holds, based on the results of small pieces of kernel code\n\
termed filters.");

	/* Add some symbolic constants to the module */
	d = PyModule_GetDict(m);
	kqueue_all_ins(d);
}
