//
// MODTOOL
//
// Copyright (c) !998 David Lindauer (LADSOFT)
//
// see license.txt for licensing info
//
// ==============================================================
//
// SMTP.CPP
//
// mail sender, send one message.  You can actually do without headers
// the way this is set up, but we won't.
//
// This is a state machine.  Rather than comment every state I give a brief
// summary of transactions here:
//
// local:  issues winsock connect command
// remote: returns code 220
// local:  sends "HELO localhost" where local host is the name of this machine
// remote: returns code 250
// local:  sends "MAIL FROM: <me>"  with the moderators address
// remote: returns code 250
// local:  sends "RCPT TO: <rcpt>" with the recipients address
// remote: returns code 250
// local:  sends "DATA"
// remote: returns code 354
//
// local:  sends headers and body
//
// local:  sends a ','
// remote: returns code 250
// local:  sends 'QUIT'
// remote: some reply which I discard
// disconnect from network
//
// All commands must be on a single line.  All lines are terminated with
// the sequence "\r\n".  When we send the text we could send as much
// data as we want in a packet, but, it is convenient to send a line at
// a time.  
//
// Note:  the mail message is terminated by the sequence
// "\r\n.\r\n" after which the remote starts responding again.
// For transparency, any other line in the message which begins with
// a dot gets another dot added.  It is the responsibility of the eventual
// receiver to strip this out.
//
// 

#define STRICT
#include <windows.h>
#include <winsock.h>
#include <stdio.h>
#include <stdarg.h>

#include "profile.h"
#include "mailfile.h"
#include "network.h"
#include "effects.h"
#include "progress.h"
#include "send.h"

// ==============================================================
// states
//
#define MS_EXPCONNECT 0
#define MS_SENDHELO   1
#define MS_RECEIVEHELO 2 
#define MS_RECEIVEHELO2 3
#define MS_SENDFROM    4
#define MS_RECEIVEFROM 5
#define MS_SENDTO      6
#define MS_RECEIVETO   7
#define MS_SENDDATA    8
#define MS_RECEIVEDATA 9
#define MS_SENDLINES   10
#define MS_SENDDOT     11
#define MS_RECEIVEDOT  12
#define MS_ERROR       13
#define MS_ABORT       14
#define MS_IGNORE      15

// So we can keep track of what WINSOCK has sent
#define FL_SEND 1
#define FL_RECEIVE 2

extern HINSTANCE hInst;
extern HWND hWndProgress,hWndMain;
extern int networkinuse;
extern int batchflags;

// ==============================================================
//
// globals
extern long int currentsize;		// size sent
extern long int totalsize;			// total size to send
extern int isaoneshot;		// false if not doing it from command line
static int flags; 		// the WINSOCK flags

static char **rcpt;		// cached recipient
static int rcptcount;		// number of recipients
static FILE *workfil;		// the input file
static int state;		// the state we are in
static SOCKET worksocket;	// the socket
static char *server;		// cached name of server we are using
static int updatesize;		// if updating the size
static int smtprejected;	// if smtp system rejected message

// ==============================================================
// state machine
//
static int callback(HWND hWnd, int mode, int error)
{
	static char string[10000];
	static int lp;
	int t;

	// If the socket has an error
	if (error == SOCKET_ERROR)  {
		state = MS_ERROR;
	}

	// if they hit cancel
	if (!hWndProgress) {
		NetworkMessageBox(MB_ICONINFORMATION,"User canceled transfer");
		state = MS_ABORT;
	}

	// ignore close events from winsock 
	if (mode == FD_CLOSE) {
		return 0;
	}

	// WINSOCK connect event, make sure the host was there
	if (mode == FD_CONNECT) {
		if (error == SOCKET_ERROR) {
			NoConnect(server);
			networkinuse = 0;
		}
		state = MS_EXPCONNECT;
		return 0;
	}

	// Get the WINSOCK mode flags to local
	if (mode == FD_WRITE)
		flags |= FL_SEND;
	if (mode == FD_READ)
		flags |= FL_RECEIVE;

	switch (state) {
		case MS_EXPCONNECT:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 220) {
				NetworkMessageBox(MB_ICONERROR,"SMTP host refused connection");
				goto abort;
			}
			state = MS_SENDFROM;
		case MS_SENDHELO:
			if (!(flags & FL_SEND))
				break;
			flags &= ~FL_SEND;
			if (StreamedNetworkSend(worksocket,"HELO %s",smtpsourcehost) != NETWORK_OK)
				goto doerror;
			state = MS_RECEIVEHELO;
			break;
		case MS_RECEIVEHELO:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 250) {
				smtprejected = TRUE;
				NetworkMessageBox(MB_ICONERROR,"SMTP host failed HELO command, aborting");
				goto abort;
			}
helolp:
			if (string[strlen(string)-1] == '\n')
				state = MS_SENDFROM;
			else {
				state = MS_RECEIVEHELO2;
				break;
			}
		case MS_SENDFROM:
			if (StreamedNetworkSend(worksocket,"MAIL FROM: <%s>",smtpsourcepath) != NETWORK_OK)
				goto doerror;
			state = MS_RECEIVEFROM;
			break;
		case MS_RECEIVEHELO2:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			goto helolp;
		case MS_RECEIVEFROM:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 250) {
				NetworkMessageBox(MB_ICONERROR,"SMTP host failed MAIL FROM command: \nMAIL FROM: <%s>\nResponse: \"%s\"", smtpsourcepath,string);
				smtprejected = TRUE;
				goto abort;
			}
			lp = 0;
sendto:
			state = MS_SENDTO;
		case MS_SENDTO:
			if (StreamedNetworkSend(worksocket,"RCPT TO: <%s>",rcpt[lp]) != NETWORK_OK)
				goto doerror;
			state = MS_RECEIVETO;
			break;
		case MS_RECEIVETO:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 250) {
				NetworkMessageBox(MB_ICONERROR,"SMTP host failed RCPT TO command\nRCPT TO: <%>\nResponse: \"%s\"",rcpt[lp],string);
				smtprejected = TRUE;
				goto abort;
			}
			free(rcpt[lp]);
			if (++lp < rcptcount)
				goto sendto;
			state = MS_SENDDATA;
		case MS_SENDDATA:
			if (StreamedNetworkSend(worksocket,"DATA") != NETWORK_OK)
				goto doerror;
			state = MS_RECEIVEDATA;
			break;
		case MS_RECEIVEDATA:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 354) {
				smtprejected = TRUE;
				NetworkMessageBox(MB_ICONERROR,"SMTP host failed DATA command, aborting");
				goto abort;
			}
			state = MS_SENDLINES;
		case MS_SENDLINES:
			while (!feof(workfil)) {
				char buf[512];
				buf[0] = 0;

				if (!GetMailString(buf,workfil)) {
					if (StreamedNetworkSend(worksocket,buf)) {
						ExtendedMessageBox("File error",MB_ICONERROR,"Can't read from temp mail file, aborting");
						goto abort;
					}
					if (updatesize) {
						currentsize += strlen(buf);
						SetProgress(1000*currentsize/totalsize,"Sending mail %d%% of %dK",100*currentsize/totalsize,totalsize/1024+1);
					}
				}
			}
			state = MS_SENDDOT;
		case MS_SENDDOT:
			if (StreamedNetworkSend(worksocket,".") != NETWORK_OK)
				goto doerror;
			state = MS_RECEIVEDOT;
			break;
		case MS_RECEIVEDOT:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			t = atoi(string);
			if (t != 250) {
				smtprejected = TRUE;
				NetworkMessageBox(MB_ICONERROR,"SMTP host failed to accept message");
				goto abort;
			}
			goto abort;
doerror:
		case MS_ERROR:
			smtprejected = TRUE;
			NetworkMessageBox(MB_ICONERROR,"WINSOCK error, aborting");
abort:
		case MS_ABORT:
			state = MS_IGNORE;
			StreamedNetworkSend(worksocket,"QUIT");
			StreamedNetworkDisconnect(worksocket);
			fclose(workfil);
			if (isaoneshot)
				PostQuitMessage(0);
			else
				if (!smtprejected)
					SendMessage(hWndMain,WM_USENDSUCCESS,0,0);
				else
					SendMessage(hWndMain,WM_USENDFAILURE,0,0);
			break;
		case MS_IGNORE:
			if (!(flags & FL_RECEIVE))
				break;
			flags &= ~FL_RECEIVE;
			if (StreamedNetworkReceive(worksocket,string,10000) != NETWORK_OK)
				goto doerror;
			StreamedNetworkSend(worksocket,"QUIT");
			StreamedNetworkDisconnect(worksocket);
			break;
	}
}
// ==============================================================
// main routine for mail send
int SendMail(int count, char **myrcpt, char *file, int oneshot, int update)
{
	// set mode flag
	isaoneshot = oneshot;

	smtprejected = 0;
	updatesize = update;

	// make sure file is there
	if (!(workfil = fopen(file,"r"))) {
		ExtendedMessageBox("Send Mail Error",MB_ICONERROR,"Mail file %s can't be opened",file);
		return 0;
	}
	else if (oneshot) {
		currentsize = 0;
		fseek(workfil,0,2);
		fgetpos(workfil,&totalsize);
		fseek(workfil,0,0);
		SetProgress(1000*currentsize/totalsize,"Sending mail %d%% of %dK",100*currentsize/totalsize,totalsize/1024+1);
	}
	// initialize state machine address
	NetworkCallback = callback;

	// select recepient and server
	rcpt = myrcpt;
	rcptcount = count;
	if (!usesmtp) 
		server = smtpserver;
	else
		server = pop3server;

	// Initiate the connection
	switch(StreamedNetworkConnect(server,SMTPPort,&worksocket)) {
		case NETWORK_OK:
			flags = 0;
			break;
		case NETWORK_NOHOST:
			NoHost(server);
			break;
	}
}