//
// MODTOOL
//
// Copyright (c) !998 David Lindauer (LADSOFT)
//
// see license.txt for licensing info
//
// ==============================================================
//
// mailfile.cpp
//
// primitives for dealing with mail files
//
#define STRICT
#include <windows.h>

#include <stdio.h>
#include <time.h>
#include <memory.h>
#include <stdarg.h>

#include "mailfile.h"
#include "analyze.h"

static char *mozillastatsstring = "X-Mozilla-Status:";

// ==============================================================
//
// litlate routine to copy a string
char *litlate(char *text)
{
	char *rv = (char *)malloc(strlen(text)+1);
	if (!rv)
		return 0;
	strcpy(rv,text);
	return rv;
}
//************************************************************************
//************************************************************************
//************************************************************************
//
// this first set of functions is for handling the network interaction
// with mail files
//
// ==============================================================
// write the 'From - date' banner to the mail file

int Banner(FILE *fil)
{
	time_t itime;
	struct tm *t1;
	char string[80];

	time(&itime);
	t1 = localtime(&itime);
	strftime(string,80,"From - %a %b %d %H:%M:%S %Y\n",t1);
	return fputs(string,fil);
}

// ==============================================================
//
// append new mail to a file
int AppendNewToMailFile(char *mailfile, char*tempfile)
{
	// open input file
	FILE *in = fopen(tempfile,"r");
	if (!in)
		return FILE_NONE;

	// open output file
	FILE *out = fopen(mailfile,"a");
	if (!out) {
		fclose(in);
		return FILE_NONE;
	}

	// put the banner
	if (Banner(out) == EOF) {
		fclose(in);
		fclose(out);
		return FILE_WRITEERR;
	}

	int count,rv;
	char **headers, *next;

	// read in the headers and put in a mozilla status line to make
	// netscape happy
	switch (rv = ReadHeaders(in,&count,&headers,&next,0)) {
		case FILE_OK:
			// this inserts the mozilla status line
			rv = FindHeader(0,count,headers,mozillastatsstring);
			if (rv != -1)
				DeleteHeader(rv,&count,headers);
			InsertHeader(&count,headers,"%s 0000\n",mozillastatsstring);
			break;
		default:
			return rv;
	}

	// write the headers back out
	if ((rv = WriteHeaders(out,count,headers)) != FILE_OK) {
		return rv;
	}

	// write the next non-header line
	if (fputs(next,out) == EOF) {
		fclose(in);
		fclose(out);
		return FILE_WRITEERR;
	}
	
	// copy rest of file one line at a time
	while (!feof(in)) {
		char buffer[512];
		buffer[0] = 0;
		fgets(buffer,512,in);
		if (ferror(in)) {
			fclose(in);
			fclose(out);
			return FILE_READERR;
		}
		if (fputs(buffer,out) == EOF) {
			fclose(in);
			fclose(out);
			return FILE_WRITEERR;
		}
	}
	fclose(in);
	fclose(out);
	return FILE_OK;
}
// ==============================================================
//
// read a string from a mail file and translate it in such a way
// that it can go over the network
//
// any line that starts with '.' gets another '.' for transparency
//
int GetMailString(char *buf, FILE *fil)
{
	int len;
	buf[0] = 0;

	// read the file
	fgets(buf,508,fil);
	len = strlen(buf);
	if (!len) return 1;

	// get rid of terminator if any
	if (buf[len-1] == '\n')
		buf[--len] = 0;

	// add a prefix dot for transparency, if required
	if (buf[0] == '.') {
		while (len) {
			buf[len] = buf[len-1];
			len--;
		}
		buf[0] = '.';
	}
	return 0;

}
//************************************************************************
//************************************************************************
//************************************************************************
//
// this next set of functions is for handling the internal processing of
// mail files
//
// ==============================================================
//
// get a line from a mail file
//
// note we DO leave the \n in the text!!!
//
int ReadMailfileLine(FILE *fil, int possibleheader, char **buf)
{
	static char buffer[10000],string[10000];
	static int cached = 0;
	buffer[0] = 0;

	if (!cached) {
		/* read the line */
		fgets(buffer,10000,fil);
		if (ferror(fil))
			return FILE_READERR;
	}
	else {
		strcpy(buffer,string);
		cached = 0;
	}

	// check for header
	if (possibleheader && strchr(buffer,':')) {
		
		while (TRUE) {
			// multiple line search ALWAYS leaves a string in 'string'
			cached = 1;
			// if there are multiple lines to this header
			// cat them together, leaveing '\n' intact
			// so they get written back properly later
			// when we write the file out again
			fgets(string,10000,fil);
			if (ferror(fil))
				return FILE_READERR;
			if (string[0] == ' ' || string[0] == '\t')
				strcat(buffer,string);
			else
				break;
		}
		// litlate it and exit
		*buf = litlate(buffer);
		if (!*buf)
			return FILE_NOMALLOC;
		return FILE_READHEADER;
	}
	else {
		// Ignore from headers
		if (!memicmp(buffer,"From - ",7))
			return FILE_READFROM;

		// return a non-header
		*buf = litlate(buffer);
		if (!*buf)
			return FILE_NOMALLOC;
		return FILE_OK;
	}
}
// ==============================================================
//
// write a line to a mail file
//
int WriteMailFileLine(FILE *fil, char *string)
{
	if (fputs(string,fil) == EOF)
		return FILE_WRITEERR;
	else
		free(string);
	return FILE_OK;
}
// ==============================================================
//
// read in all headers
//
// don't call this unless you are sure the next is
// a header!!!
//
// this implementation will NOT allow multiprocessing!
//
int ReadHeaders(FILE *fil, int *count, char ***headptr,char **nextline, int *retpos)
{
	int mycount = 0;
	int temp;
	char *str;
	static char *arglist[500];
	if (retpos)
		*retpos = 0;
	while (TRUE) {
		long pos;
		fgetpos(fil,&pos);
		switch(temp = ReadMailfileLine(fil,TRUE,&str)) {
			case FILE_READERR:
			case FILE_WRITEERR:
				return temp;
			case FILE_OK:
				*nextline = str;
				*headptr=arglist;
				*count = mycount;
				return temp;
			case FILE_READHEADER:
				if (retpos && stristr(str,mozillastatsstring))
					*retpos = pos;
				arglist[mycount++] = str;
				break;
			case FILE_READFROM:
				// Unless someone makes a file by hand
				// and muffs it we won't get here.
				return FILE_READERR;
		}
	}
}			
// ==============================================================
//
// write all headers
//
int WriteHeaders(FILE *fil, int count, char **header)
{
	int i;
	for (i=0; i < count; i ++) {
		if (fputs(header[i],fil) == EOF)
			return FILE_WRITEERR;
		free(header[i]);
	}
	return FILE_OK;

}
// ==============================================================
//
// delete all headers from array
//
void FreeHeaders(int count, char **headers)
{
	for (int i= 0; i < count; i++)
		free(headers[i]);
}
// ==============================================================
//
// Insert a new header in the list
//
int InsertHeader(int *count, char **headers, char *fmt,...)
{
	va_list argptr;
	char string[256];
	char *rv;
	va_start( argptr, fmt);
  	vsprintf(string, fmt, argptr);
	va_end(argptr);
	if (!(rv = litlate(string)))
		return FILE_NOMALLOC;
	headers[(*count)++] = rv;
	return FILE_OK;
}
// ==============================================================
//
// Delete a header from the list
//
void DeleteHeader(int pos, int *count, char **headers)
{
	int i;
	free(headers[pos]);
	for (i= pos; i < *count; i++)
		headers[i] = headers[i+1];
	(*count)--;
}
// ==============================================================
//
// Find a header in the list
//
int FindHeader(int pos,int count, char **headers,char *string)
{
	for (int i = pos; i < count; i++)
		if (!memicmp(headers[i],string,strlen(string)))
			return i;

	return -1;
}