/*
 * client SMTP program, use sm io.
 *
 * based on proxy.c 
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: smtpc2.c,v 1.59 2005/10/18 17:31:50 ca Exp $")
#include "sm/assert.h"
#include "sm/error.h"
#include "sm/str.h"
#include "sm/test.h"
#include "sm/io.h"
#include "sm/ctype.h"
#if 0
#include "smi-net.h"
#endif
#include "sm/sysexits.h"

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "st.h"
#include "common.h"	/* HACK for debugging, otherwise structs are hidden */

extern sm_stream_T SmStThrIO;

#define SM_IOBUFSIZE (1024*1024)
#define SM_LINE_LEN (4*1024)
static	uchar databuf[SM_IOBUFSIZE];

#ifndef INADDR_NONE
# define INADDR_NONE 0xffffffff
#endif

#define MAXTC	65536L		/* max. number of total connections */
#define SC_MAXADDRS	256		/* max. number of addresses */
#define REQUEST_TIMEOUT 5
#define SEC2USEC(s) ((s)*1000000LL)

static char *maildom = "local.host";
static char *rcptdom = "local.host";
static char *prog;			/* Program name   */
static struct sockaddr_in rmt_addr;	/* Remote address */
static unsigned int ta_per_thrd;
static unsigned int sessions;
static unsigned int sess_cnt = 0;
static int debug = 0;
static int stoponerror = 0;
static unsigned long count = 0;
static int busy = 0;
static int concurrent = 0;
static int waitdata = 0;
static int sequence = -1;
static int seqfirst = 0;
static int postfix = 0;
static int rset = 1;
static unsigned int datalen = 0;
static char *cmdfile = NULL;
static char *datafile = NULL;
static int flushdata = 0;
static unsigned int rcpts = 0;
static long unsigned int ta_total = 0;
static long unsigned int ta_cnt = 0;
static char *from[SC_MAXADDRS], *rcpt[SC_MAXADDRS];
static unsigned int fromaddrs = 0;
static unsigned int rcptaddrs = 0;
static unsigned int request_timeout = REQUEST_TIMEOUT;
static int showerror = 0;
static int usesize = 0;
static char *mailarg = NULL;

#define SCE_ST_ABORT_SE	0x0100
#define SCE_ST_ABORT_TA	0x0200

#define SCE_CODE(code)	((code) & 0x7f00)
#define SCE_VAL(code)	((code) & 0x00ff)
#define SCE_IS_ABORT_SE(code, counter)	\
	((SCE_CODE(code) == SCE_ST_ABORT_SE) && \
	 (SCE_VAL(code) == 0 || SCE_VAL(code) == (counter)))
#define SCE_IS_ABORT_TA(code, counter)	\
	((SCE_CODE(code) == SCE_ST_ABORT_TA) && \
	 (SCE_VAL(code) == 0 || SCE_VAL(code) == (counter)))

#define STAGE_CONNECT	0
#define STAGE_HELO	1
#define STAGE_MAIL	2
#define STAGE_RCPT	3
#define STAGE_DATA	4
#define STAGE_DOT	5
#define STAGES		6

static int codes[STAGES];
static int waits[STAGES];

static void read_address(const char *str, struct sockaddr_in *sin);
static void *handle_request(void *arg);
static void print_sys_error(const char *msg);


static void
sce_usage(const char *prg)
{
	sm_io_fprintf(smioerr,
		"Usage: %s [options] -r host:port\n"
		"-2          flush data before final dot\n"
		"-a stage=code   set action reply code for stage to code\n"
		"   stage:\n"
		"     c: new session (connect)\n"
		"     h: HELO\n"
		"     m: MAIL\n"
		"     r: RCPT\n"
		"     d: DATA\n"
		"     b: Body\n"
		"   code:    action\n"
		"   %06#x   abort session [drop connection]\n"
		"   %06#x   abort transaction [move to next if there is one]\n"
		"   lower 2 bytes can be used as counter\n"
		"-B filename use content of filename for commands to send\n"
		"-C n        show counter (print if counter %% n == 0)\n"
		"-d n        set debug level\n"
		"-D filename use content of filename for DATA\n"
		"-E          show SMTP dialogue errors\n"
		"-F domain   use domain for sender addresses [%s]\n"
		"-f address  from address\n"
		"-l n        data length\n"
		"-M arg      set arg for MAIL\n"
		"-m n        number of messages to send\n"
		"-N          do not use RSET between transactions\n"
		"-n n        number of recipients per transaction\n"
		"-p n        use n in addresses as identifier\n"
		"-O timeout  I/O timeout [%d].\n"
		"-R address  recipient address (can be specified multiple times)\n"
		"-s n        n total sessions\n"
		"-S          stop on errors\n"
		"-t n        concurrent threads\n"
		"-T n        transactions per thread\n"
		"-w n        wait for n seconds after DATA\n"
		"-W stage=delay   wait for delay seconds before issuing stage command\n"
		"-q n        send n messages with 1..n as body.\n"
		"-Q m        use m as first element in sequence: m..n.\n"
		"-Y domain   use domain for recipient addresses [%s]\n"
		"-Z          use SIZE= for MAIL\n"
		, prg
		, SCE_ST_ABORT_SE
		, SCE_ST_ABORT_TA
		, maildom
		, request_timeout
		, rcptdom
		);
}

static unsigned int
getstage(char arg)
{
	unsigned int stage;

	stage = -1;
	switch (optarg[0])
	{
	  case 'c':
		stage = STAGE_CONNECT;
		break;
	  case 'h':
	  case 'e':
		stage = STAGE_HELO;
		break;
	  case 'm':
		stage = STAGE_MAIL;
		break;
	  case 'r':
		stage = STAGE_RCPT;
		break;
	  case 'd':
		stage = STAGE_DATA;
		break;
	  case 'b':
		stage = STAGE_DOT;
		break;
	  default:
		sce_usage(prog);
		exit(EX_USAGE);
	}
	return stage;
}

int
main(int argc, char *argv[])
{
	extern char    *optarg;
	int             opt, n, threads;
	int             raddr;
	unsigned int    u, off, stage;
	long int        tc;
	size_t		j;
	ssize_t		bytesread;

	prog = argv[0];
	raddr = 0;
	sessions = ta_per_thrd = threads = 1;
	rcpt[0] = from[0] = NULL;
	bytesread = 0;

	/* Parse arguments */
	while ((opt = getopt(argc, argv,
			"2a:B:C:c:D:d:EF:f:hl:M:m:Nn:O:p:r:R:s:St:T:W:w:Q:q:Y:Z")) != -1)
	{
		switch (opt)
		{
		  case '2':
			flushdata = 1;
			break;
		  case 'a':
			if (optarg != NULL && ISALPHA(optarg[0])
			    && optarg[1] == '=')
				off = 2;
			else
			{
				sce_usage(prog);
				break;
			}

			u = (unsigned int) strtoul(optarg + off, NULL, 0);
			stage = getstage(optarg[0]);
			SM_ASSERT(stage < STAGES);
			codes[stage] = u;
			break;
		  case 'B':
			cmdfile = strdup(optarg);
			if (cmdfile == NULL)
			{
				sm_io_fprintf(smioerr, "strdup(%s)=NULL\n"
					, optarg);
				exit(1);
			}
			break;
		  case 'C':
			count = (unsigned long) strtoul(optarg, NULL, 0);
			break;
		  case 'T':
		  case 'c':
			ta_per_thrd = atoi(optarg);
			if (ta_per_thrd < 1)
			{
				sm_io_fprintf(smioerr,
					"%s: invalid number of connections: %s\n",
					prog, optarg);
				exit(1);
			}
			break;
		  case 'd':
			debug = atoi(optarg);
			if (debug < 0)
			{
				sm_io_fprintf(smioerr,
					"%s: invalid number for debug: %s\n",
					prog, optarg);
				exit(1);
			}
			break;
		  case 'D':
			datafile = strdup(optarg);
			if (datafile == NULL)
			{
				sm_io_fprintf(smioerr, "strdup(%s)=NULL\n"
					, optarg);
				exit(1);
			}
			break;
		  case 'E':
			++showerror;
			break;
		  case 'f':
			if (fromaddrs >= SC_MAXADDRS - 1)
			{
				sm_io_fprintf(smioerr,
					"%s: too many addresses=%d, max=%d\n",
					prog, fromaddrs, SC_MAXADDRS);
				exit(1);
			}
			from[fromaddrs++] = optarg;
			break;
		  case 'F':
			maildom = strdup(optarg);
			if (maildom == NULL)
			{
				sm_io_fprintf(smioerr,
					"%s: failed to strdup() %s\n"
					, prog, optarg);
				exit(1);
			}
			break;
		  case 'M':
			if (optarg == NULL)
			{
				sm_io_fprintf(smioerr,
					"%s: missing arg for -%c\n",
					prog, opt);
				exit(1);
			}
			mailarg = strdup(optarg);
			if (mailarg == NULL)
			{
				sm_io_fprintf(smioerr,
					"%s: strdup(%s) failed\n",
					prog, optarg);
				exit(1);
			}
			break;
		  case 'm':
			ta_total = (unsigned long) atol(optarg);
			break;
		  case 'l':
			datalen = (unsigned int) atoi(optarg);
			break;
		  case 'N':
			rset = 0;
			break;
		  case 'n':
			rcpts = (unsigned int) atoi(optarg);
			if (rcpts < 1)
			{
				sm_io_fprintf(smioerr,
					"%s: invalid number of rcpts: %s\n",
					prog, optarg);
				exit(1);
			}
			break;
		  case 'O':
			request_timeout  = atoi(optarg);
			if (request_timeout <= 0)
			{
				sm_io_fprintf(smioerr,
					"%s: invalid timeout: %s\n",
					prog, optarg);
				exit(1);
			}
			break;
		  case 'p':
			postfix = atoi(optarg);
			break;
		  case 'r':
			read_address(optarg, &rmt_addr);
			if (rmt_addr.sin_addr.s_addr == INADDR_ANY)
			{
				sm_io_fprintf(smioerr,
					"%s: invalid remote address: %s\n",
					prog, optarg);
				exit(1);
			}
			raddr = 1;
			break;
		  case 'R':
			if (rcptaddrs >= SC_MAXADDRS - 1)
			{
				sm_io_fprintf(smioerr,
					"%s: too many addresses=%d, max=%d\n",
					prog, rcptaddrs, SC_MAXADDRS);
				exit(1);
			}
			rcpt[rcptaddrs++] = optarg;
			break;
		  case 's':
			sessions = (unsigned int) atoi(optarg);
			if (sessions < 1)
			{
				sm_io_fprintf(smioerr,
					"%s: invalid number of sessions: %s\n",
					prog, optarg);
				exit(1);
			}
			break;
		  case 'S':
			stoponerror++;
			break;
		  case 't':
			threads = atoi(optarg);
			if (threads < 1)
			{
				sm_io_fprintf(smioerr,
					"%s: invalid number of threads: %s\n",
					prog, optarg);
				exit(1);
			}
			break;
		  case 'q':
			sequence = atoi(optarg);
			if (sequence < 1)
			{
				sm_io_fprintf(smioerr,
					"%s: invalid number for sequence: %s\n"
					, prog, optarg);
				exit(1);
			}
			break;
		  case 'Q':
			seqfirst = atoi(optarg);
			if (seqfirst < 0)
			{
				sm_io_fprintf(smioerr,
					"%s: invalid number for sequence begin: %s\n"
					, prog, optarg);
				exit(1);
			}
			break;
		  case 'W':
			if (optarg != NULL && ISALPHA(optarg[0])
			    && optarg[1] == '=')
				off = 2;
			else
			{
				sce_usage(prog);
				break;
			}

			u = (unsigned int) strtoul(optarg + off, NULL, 0);
			stage = getstage(optarg[0]);
			SM_ASSERT(stage < STAGES);
			waits[stage] = u;
			break;
		  case 'w':
			waitdata = atoi(optarg);
			break;
		  case 'Y':
			rcptdom = strdup(optarg);
			if (rcptdom == NULL)
			{
				sm_io_fprintf(smioerr,
					"%s: failed to strdup() %s\n"
					, prog, optarg);
				exit(1);
			}
			break;
		  case 'Z':
			usesize = true;
			break;
		  case 'h':
		  case '?':
			sce_usage(prog);
			exit(1);
		}
	}
	if (!raddr)
	{
		sm_io_fprintf(smioerr, "%s: remote address required\n", prog);
		exit(1);
	}

	/* number of recipients not set? */
	if (rcpts == 0)
	{
		if (rcptaddrs > 1)
			rcpts = rcptaddrs;
		else
			rcpts = 1;
	}

	if (bytesread == 0)
	{
#define MAIL_HEADER "From: me\r\nTo: you\r\nSubject: test\r\n\r\n"
		j = strlcpy((char *)databuf, MAIL_HEADER, sizeof(databuf));
		for (j = strlen(MAIL_HEADER); j < sizeof(databuf); j++)
			databuf[j] = ' ' + (j % 64);
		for (j = 80; j < sizeof(databuf) - 1; j += 80)
		{
			databuf[j] = '\r';
			databuf[j + 1] = '\n';
		}
	}

	tc = (long) sessions * (long) ta_per_thrd;
	if (debug)
		sm_io_fprintf(smioerr, "%s: starting client [%d]\n", prog,
			threads);

	/* Initialize the ST library */
	if (st_init() < 0)
	{
		print_sys_error("st_init");
		exit(1);
	}
	for (n = 0; n < threads; n++)
	{
		if (debug)
			sm_io_fprintf(smioerr, "%s: starting client %d/%d\n",
				prog, n, threads);
		if (st_thread_create(handle_request, (void *) n, 0, 0) == NULL)
		{
			print_sys_error("st_thread_create");
			exit(1);
		}
	}

	/* wait for them... */
	st_sleep(1);
	while (busy > 0)
		st_sleep(1);
	/* XXX how? */

	sm_io_fprintf(smioerr, "%s: total=%lu (should be %lu/%lu)\n",
			prog, ta_cnt, tc, ta_total);
	return 0;
}

static void
read_address(const char *str, struct sockaddr_in * sin)
{
	char            host[128], *p;
	struct hostent *hp;
	short           port;

	strlcpy(host, str, sizeof(host));
	if ((p = strchr(host, ':')) == NULL)
	{
		sm_io_fprintf(smioerr, "%s: invalid address: %s\n", prog, host);
		exit(1);
	}
	*p++ = '\0';
	port = (short) atoi(p);
	if (port < 1)
	{
		sm_io_fprintf(smioerr, "%s: invalid port: %s\n", prog, p);
		exit(1);
	}
	memset(sin, 0, sizeof(struct sockaddr_in));
	sin->sin_family = AF_INET;
	sin->sin_port = htons(port);
	if (host[0] == '\0')
	{
		sin->sin_addr.s_addr = INADDR_ANY;
		return;
	}
	sin->sin_addr.s_addr = inet_addr(host);
	if (sin->sin_addr.s_addr == INADDR_NONE)
	{
		/* not dotted-decimal */
		if ((hp = gethostbyname(host)) == NULL)
		{
			sm_io_fprintf(smioerr, "%s: can't resolve address: %s\n", prog, host);
			exit(1);
		}
		memcpy(&sin->sin_addr, hp->h_addr, hp->h_length);
	}
}

/* before changing these, check the macros! */
#define SMTP_OK		0
#define SMTP_AN		1	/* SMTP reply type isn't 2 or 3 */
#define SMTP_SSD	2	/* 421 */
#define SMTP_RD		3	/* read error */
#define SMTP_WR		4	/* write error */
#define SMTP_IO_ERR(r)	((r) >= SMTP_RD)
#define SMTP_FATAL(r)	((r) >= SMTP_SSD)

static int
smtpwrite(int l, sm_file_T *fp, int tid, int i)
{
	int wr;
	sm_ret_T ret;
	ssize_t b;

	while (l > 0)
	{
		wr = SM_MIN(l, (int) sizeof(databuf));
		ret = sm_io_write(fp, databuf, wr, &b);
		if ((int) b != wr || ret != SM_SUCCESS)
		{
			sm_io_fprintf(smioerr,
				"[%d] data write error i=%d, n=%d, r=%d, ret=%x\n"
			      , tid, i, l, (int) b, ret);
			return SMTP_WR;
		}
		l -= wr;
	}
	return SMTP_OK;
}

static int
smtpdata(int l, sm_file_T *fp, int tid, int i)
{
	int r;

	if (datafile != NULL)
	{
		sm_file_T *dfp;
		ssize_t bytesread;

		r = sm_io_open(SmStStdio, datafile, SM_IO_RDONLY, &dfp,
			SM_IO_WHAT_END);
		if (r != SM_SUCCESS)
		{
			sm_io_fprintf(smioerr, "sm_io_open(%s)=%r\n"
				, datafile, r);
			return r;
		}
		do
		{
			r = sm_io_read(dfp, databuf, sizeof(databuf),
				&bytesread);
			if (r == SM_IO_EOF)
			{
				r = SMTP_OK;
				break;
			}
			if (sm_is_err(r))
				break;
			r = smtpwrite(bytesread, fp, tid, i);
			if (flushdata)
				(void) sm_io_flush(fp);
		} while (r == SMTP_OK);
		sm_io_close(dfp);
	}
	else
		r = smtpwrite(l, fp, tid, i);
	return r;
}

static int
smtpcommand(char *str, int l, sm_file_T *fp, sm_str_P out, int tid, unsigned int i, int stage)
{
	sm_ret_T        ret;
	ssize_t         b;

	if (stage >= 0 && stage <= STAGES && waits[stage] > 0)
		sleep(waits[stage]);
	if (debug > 3)
	{
		sm_io_fprintf(smioerr, "[%d] send: ", tid);
		sm_io_write(smioerr, (uchar *) str, l, &b);
		sm_io_flush(smioerr);
	}
	ret = sm_io_write(fp, (uchar *) str, l, &b);
	if (b != l)
	{
		sm_io_fprintf(smioerr, "[%d] write error i=%u, n=%d, r=%d, ret=%x\n",
			      tid, i, l, (int) b, ret);
		if (stoponerror)
			sessions = 0;
		return SMTP_RD;
	}
	ret = sm_io_flush(fp);
	if (sm_is_error(ret))
	{
		sm_io_fprintf(smioerr, "[%d] flush error i=%u, n=%d, ret=%x\n",
			      tid, i, (int) b, ret);
		if (stoponerror)
			sessions = 0;
		return SMTP_WR;
	}
	do
	{
		time_t before, after;

		sm_str_clr(out);
#if DEBUG
		before = st_time();
#else
		before = 0;
#endif
		ret = sm_fgetline0(fp, out);
#if DEBUG
		after = st_time();
#else
		after = 0;
#endif

		if (debug > 3)
		{
			sm_io_fprintf(smioerr, "[%d] rcvd [len=%d, res=%x]: ", tid,
				      sm_str_getlen(out), ret);
			sm_io_write(smioerr, sm_str_getdata(out),
				sm_str_getlen(out), &b);
			sm_io_flush(smioerr);
		}
		if (sm_is_error(ret))
		{
			sm_io_fprintf(smioerr,
				"[%d] cmd=%s, error=read, i=%u, n=%d, ret=%x, after-before=%ld\n",
				tid, str, i, (int) b, ret,
				(long) (after - before));
			if (stoponerror)
				sessions = 0;
			return SMTP_WR;
		}
		if (sm_str_getlen(out) == 0 ||
		    (sm_str_rd_elem(out, 0) != '2' && sm_str_rd_elem(out, 0) != '3'))
			return SMTP_AN;
	} while (!sm_is_error(ret) && sm_str_getlen(out) > 4 && sm_str_rd_elem(out, 3) == '-');

	if (sm_str_getlen(out) > 3
	    && sm_str_rd_elem(out, 0) == '4'
	    && sm_str_rd_elem(out, 1) == '2'
	    && sm_str_rd_elem(out, 2) == '1')
		return SMTP_SSD;

	if (sm_str_getlen(out) > 3
	    && sm_str_rd_elem(out, 0) != '2'
	    && sm_str_rd_elem(out, 0) != '3')
		return SMTP_AN;
	/* check reply code... */

	return SMTP_OK;
}

static int
smtpcmds(sm_file_T *fp, int tid, int i, sm_str_P out)
{
	int r;
	sm_file_T *cfp;

	r = sm_io_open(SmStStdio, cmdfile, SM_IO_RDONLY, &cfp, SM_IO_WHAT_END);
	if (r != SM_SUCCESS)
	{
		sm_io_fprintf(smioerr, "sm_io_open(%s)=%r\n", cmdfile, r);
		return r;
	}
	do
	{
		sm_str_clr(out);
		r = sm_fgetline(cfp, out);
		if (sm_is_err(r))
			break;
		r = smtpcommand((char *)sm_str_getdata(out), sm_str_getlen(out),
				fp, out, tid, 0, -1);
		if (r != SMTP_OK && showerror)
			sm_io_fprintf(smioerr, "text=%@T\n", out);
	} while (!SMTP_IO_ERR(r) && !SMTP_FATAL(r));
	sm_io_close(cfp);
	return r;
}

static void *
handle_request(void *arg)
{
	st_netfd_t      rmt_nfd;
	int             sock, n, i, tid, r, rok, myseq;
	unsigned int    u, j;
	ssize_t         b;
	sm_file_T      *fp;
	sm_str_P        out;
	sm_ret_T        ret;
	char		*s;
	char            buf[SM_LINE_LEN];
	char            sbuf[64];

	++busy;
	i = 0;
	tid = (int) arg;
	if (debug)
		sm_io_fprintf(smioerr, "client[%d]: transactions=%d\n",
			tid, ta_per_thrd);

	out = sm_str_new(NULL, SM_LINE_LEN, SM_IOBUFSIZE);
	if (out == NULL)
	{
		sm_snprintf(buf, sizeof(buf), "[%d] str new failed i=%d",
			tid, i);
		print_sys_error(buf);
		if (stoponerror)
			sessions = 0;
		goto done;
	}

	/*
	**  Only run a certain number of sessions;
	**  this is global variable but we can manipulate it without locking.
	*/

	while (sess_cnt < sessions)
	{
		++sess_cnt;

		myseq = sequence;
		if (debug)
			sm_io_fprintf(smioerr,
				"client[%d]: myseq=%d, sequence=%d, seqfirst=%d\n"
				, tid, myseq, sequence, seqfirst);
		if (myseq == seqfirst)
			goto done;
		if (sequence > seqfirst)
			--sequence;

		/* Connect to remote host */
		if ((sock = socket(PF_INET, SOCK_STREAM, 0)) < 0)
		{
			sm_snprintf(buf, sizeof(buf), "[%d] socket i=%d",
				tid, i);
			print_sys_error(buf);
			if (stoponerror)
				sessions = 0;
			goto done;
		}
		ret = sm_io_open(&SmStThrIO, (void *) &sock, SM_IO_RDWR, &fp,
				NULL);
		if (ret != SM_SUCCESS)
		{
			sm_snprintf(buf, sizeof(buf),
				"[%d] sm_io_open()=%d, i=%d", tid, ret, i);
			print_sys_error(buf);
			close(sock);
			if (stoponerror)
				sessions = 0;
			goto done;
		}
		sm_io_clrblocking(fp);
		rmt_nfd = (st_netfd_t) f_cookie(*fp);

		r = request_timeout;
		ret = sm_io_setinfo(fp, SM_IO_WHAT_TIMEOUT, &r);
		if (ret != SM_SUCCESS)
		{
			sm_snprintf(buf, sizeof(buf),
				"[%d] set timeout()=%d, i=%d", tid, ret, i);
			print_sys_error(buf);
			sm_io_close(fp);
			if (stoponerror)
				sessions = 0;
			goto done;
		}
		ret = sm_io_setinfo(fp, SM_IO_DOUBLE, NULL);
		if (ret != SM_SUCCESS)
		{
			sm_snprintf(buf, sizeof(buf),
				"[%d] set double()=%d, i=%d", tid, ret, i);
			print_sys_error(buf);
			sm_io_close(fp);
			if (stoponerror)
				sessions = 0;
			goto done;
		}
		rmt_nfd = (st_netfd_t) f_cookie(*fp);
		if (st_connect(rmt_nfd, (struct sockaddr *) & rmt_addr,
			       sizeof(rmt_addr), SEC2USEC(request_timeout)) < 0)
		{
			sm_snprintf(buf, sizeof(buf), "[%d] connect i=%d",
				tid, i);
			print_sys_error(buf);
			sm_io_close(fp);
			if (stoponerror)
				sessions = 0;
			goto done;
		}
		++concurrent;
		if (debug > 2)
			sm_io_fprintf(smioerr,
				"client[%d]: connected, i=%d, ta=%d, conc=%d\n",
				tid, i, ta_per_thrd, concurrent);

		do
		{
			sm_str_clr(out);
			ret = sm_fgetline0(fp, out);

			if (debug > 3)
			{
				sm_io_fprintf(smioerr,
					"[%d] greet [len=%d, res=%x]: ", tid,
					sm_str_getlen(out), ret);
				sm_io_write(smioerr, sm_str_getdata(out),
					sm_str_getlen(out), &b);
				sm_io_flush(smioerr);
			}
			if (sm_is_error(ret))
			{
				sm_io_fprintf(smioerr,
					"[%d] read greet i=%d, ret=%x\n",
					tid, i, ret);
				if (stoponerror)
					sessions = 0;
				goto fail;
			}
		} while (!sm_is_error(ret) && sm_str_getlen(out) > 4 &&
			 sm_str_rd_elem(out, 3) == '-');
		if (!sm_is_error(ret)
		    && sm_str_getlen(out) > 2
		    && sm_str_rd_elem(out, 0) == '4'
		    && sm_str_rd_elem(out, 1) == '2'
		    && sm_str_rd_elem(out, 2) == '1')
			goto fail;
		if (sm_is_error(ret) || sm_str_getlen(out) <= 0
		    || sm_str_rd_elem(out, 0) != '2')
			goto fail;

		if (SCE_IS_ABORT_SE(codes[STAGE_CONNECT], sess_cnt))
			goto fail;

		if (cmdfile != NULL)
		{
			r = smtpcmds(fp, tid, i, out);
			goto fail;
		}

		n = strlcpy(buf, "EHLO me.local\r\n", sizeof(buf));
		r = smtpcommand(buf, n, fp, out, tid, 0, STAGE_HELO);
		if (r == SMTP_SSD)
			goto fail;
		if (r != SMTP_OK)
		{
			if (showerror)
				sm_io_fprintf(smioerr,
					"EHLO=error, text=%@T, sessions=%d\n",
					out, sessions);
			goto fail;
		}

		if (SCE_IS_ABORT_SE(codes[STAGE_HELO], sess_cnt))
			goto fail;

		for (u = 0;
		     u < ta_per_thrd && myseq != seqfirst && sessions != 0;
		     u++)
		{
			sm_memzero(sbuf, sizeof(sbuf));
			s = sbuf;
			if (usesize && datalen > 0)
			{
				n = sm_snprintf(sbuf, sizeof(sbuf),
					" SIZE=%d", datalen);
			}
			if (mailarg != NULL)
			{
				/* ignore errors for now ... */
				strlcat(sbuf, " ", sizeof(sbuf));
				strlcat(sbuf, mailarg, sizeof(sbuf));
			}

			if (fromaddrs > 0)
				n = sm_snprintf(buf, sizeof(buf),
					"MAIL FROM:<%s>%s\r\n",
					from[u % fromaddrs], s);
			else if (myseq > 0)
				n = sm_snprintf(buf, sizeof(buf),
					"MAIL FROM:<nobody-%d-%d-%d_%d@%s>\r\n",
					myseq, i, tid, postfix, maildom);
			else
				n = sm_snprintf(buf, sizeof(buf),
					"MAIL FROM:<nobody-%d-%d-%d@%s>%s\r\n",
					u, tid, sessions, maildom, s);
			r = smtpcommand(buf, n, fp, out, tid, u, STAGE_MAIL);
			if (r == SMTP_SSD)
				goto fail;
			if (r != SMTP_OK)
			{
				if (showerror)
					sm_io_fprintf(smioerr,
						"MAIL=error, text=%@T, sessions=%d, ta=%d\n",
						out, sessions, u);
				break;
			}
			if (SCE_IS_ABORT_SE(codes[STAGE_MAIL], ta_cnt))
				goto fail;
			if (SCE_IS_ABORT_TA(codes[STAGE_MAIL], ta_cnt))
				break;

			rok = 0;
			for (j = 0; j < rcpts; j++)
			{
				if (rcptaddrs > 0)
					n = sm_snprintf(buf, sizeof(buf),
						"RCPT TO:<%s>\r\n",
						rcpt[j % rcptaddrs]);
				else if (myseq > 0)
					n = sm_snprintf(buf, sizeof(buf),
						"RCPT To:<nobody-%d-%d-%d-%d@%s>\r\n",
						myseq, i, j, tid, rcptdom);
				else
					n = sm_snprintf(buf, sizeof(buf),
						"RCPT TO:<nobody-%u-%u-%d-%d@%s>\r\n",
						u, j, tid, sessions, rcptdom);
				r = smtpcommand(buf, n, fp, out, tid, u,
						STAGE_RCPT);
				if (r == SMTP_SSD)
					goto fail;
				if (r != SMTP_OK)
				{
					if (showerror)
						sm_io_fprintf(smioerr,
							"RCPT=error, text=%@T, se=%d, ta=%u, rcpt=%u\n",
							out, sessions, u, j);
					if (SMTP_IO_ERR(r))
						break;
				}
				else
					++rok;
				if (SCE_IS_ABORT_SE(codes[STAGE_RCPT], ta_cnt))
					goto fail;
				if (SCE_IS_ABORT_TA(codes[STAGE_RCPT], ta_cnt))
					break;
			}
			if (SMTP_FATAL(r) || rok == 0)
				break;
			if (SCE_IS_ABORT_TA(codes[STAGE_RCPT], ta_cnt))
				break;

			n = strlcpy(buf, "DATA\r\n", sizeof(buf));
			r = smtpcommand(buf, n, fp, out, tid, u, STAGE_DATA);
			if (r == SMTP_SSD)
				goto fail;
			if (r != SMTP_OK)
			{
				if (showerror)
					sm_io_fprintf(smioerr,
						"DATA=error, text=%@T, se=%d, ta=%d, r=%d\n",
						out, sessions, u, r);
				break;
			}
			if (waitdata > 0)
				st_sleep(waitdata);
			if (SCE_IS_ABORT_SE(codes[STAGE_DATA], ta_cnt))
				goto fail;
			if (SCE_IS_ABORT_TA(codes[STAGE_DATA], ta_cnt))
				break;

			if (myseq > 0)
			{
				n = sm_snprintf(buf, sizeof(buf),
					"From: me+%d\r\nTo: you+%d\r\nSubject: test+%d\r\n\r\n%d\r\n",
					myseq, myseq, myseq, myseq);
				ret = sm_io_write(fp, (unsigned char *) buf,
						strlen(buf), &b);
				if ((int) b != strlen(buf) || ret != SM_SUCCESS)
				{
					sm_io_fprintf(smioerr,
						"[%d] data write error i=%d, n=%d, b=%d, ret=%x\n"
			      		, tid, i, strlen(buf), (int) b, ret);
					break;
				}
				if (datalen > 0 || datafile != NULL)
				{
					r = smtpdata(datalen, fp, tid, u);
					if (r != SMTP_OK)
						break;
				}
				n = strlcpy(buf, "\r\n.\r\n", sizeof(buf));
			}
			else if (datalen == 0 && datafile == NULL)
			{
				n = strlcpy(buf,
					"From: me\r\nTo: you\r\nSubject: test\r\n\r\nbody\r\n.\r\n",
					sizeof(buf));
			}
			else
			{
				r = smtpdata(datalen, fp, tid, u);
				if (r != SMTP_OK)
					break;
				n = strlcpy(buf, "\r\n.\r\n", sizeof(buf));
			}
			if (SCE_IS_ABORT_SE(codes[STAGE_DOT], ta_cnt))
				goto fail;
			if (SCE_IS_ABORT_TA(codes[STAGE_DOT], ta_cnt))
				break;

			r = smtpcommand(buf, n, fp, out, tid, u, STAGE_DOT);
			if (r == SMTP_SSD)
				goto fail;
			if (r != SMTP_OK)
			{
				if (showerror)
					sm_io_fprintf(smioerr,
						"DOT=error, text=%@T, se=%d, ta=%d\n",
						out, sessions, u);
				break;
			}

			/* another transaction? */
			if (u < ta_per_thrd - 1)
			{
				myseq = sequence;
				if (sequence > seqfirst)
					--sequence;
				if (rset)
				{
					n = strlcpy(buf, "RSET\r\n",
						sizeof(buf));
					r = smtpcommand(buf, n, fp, out, tid,
							u, -1);
				}
			}
			if (r != SMTP_OK)
				break;

			++ta_cnt;
			if (count > 0 && (ta_cnt % count) == 0)
				sm_io_fprintf(smioerr, "%ld\r", ta_cnt);
			if (ta_total > 0 && ta_cnt >= ta_total)
			{
				sessions = 0;
				break;
			}
		}
		n = strlcpy(buf, "QUIT\r\n", sizeof(buf));
		r = smtpcommand(buf, n, fp, out, tid, u, -1);
		/*
		if (r != SMTP_OK)
			;
		*/

		sm_io_close(fp);
		fp = NULL;
		--concurrent;
	}

  fail:
	if (fp != NULL)
	{
		sm_io_close(fp);
		fp = NULL;
		--concurrent;
	}

  done:
	SM_STR_FREE(out);
	--busy;
	return NULL;
}

static void 
print_sys_error(const char *msg)
{
	sm_io_fprintf(smioerr, "%s: %s: %s\n", prog, msg, strerror(errno));
}
