package Class::MethodMaker;

#
# $Id: MethodMaker.pm,v 1.1.1.2 1997/01/23 23:05:36 seibel Exp $
#

# Copyright (c) 1996 Organic Online. All rights reserved. This program is
# free software; you can redistribute it and/or modify it under the same
# terms as Perl itself.


=head1 NAME

B<Class::MethodMaker> - a module for creating generic methods

=head1 SYNOPSIS

use Class::MethodMaker
  new_with_init => 'new',
  get_set       => [ qw /foo bar baz / ];

=head1 DESCRIPTION

This module solves the problem of having to write a bazillion get/set
methods that are all the same. The argument to 'use' is a hash whose keys
are the names of types of generic methods generated by MethodMaker and
whose values tell method maker what methods to make. (More precisely, the
keys are the names of MethodMaker methods (methods that write methods)
and the values are the arguments to those methods.

=cut

use strict;
require 5.00307; # for the ->isa method.
use Carp;

use AutoLoader;
use vars '@ISA';
@ISA = qw ( AutoLoader );

use vars '$VERSION';
$VERSION = "0.92";
			
# Just to point out the existence of these variables

use vars
 '$TargetClass',    # The class we are making methods for.

 '%BooleanPos',     # A hash of the current index into the bit vector
                    # used in boolean for each class.

 '%BooleanFields',  # A hash of refs to arrays which store the names of
                    # the bit fileds for a given class

 '%StructPos',      # A hash of the current index into the arry used in
                    # struct for each class.

 '%StructFields';   # A hash of refs to arrays which store the names of
                    # the struct fields for a given class

sub ima_method_maker { 1 };

sub set_target_class {
  my ($class, $target) = @_;
  $TargetClass = $target;
}

sub get_target_class {
  my ($class) = @_;
  $TargetClass || $class->find_target_class;
}

sub find_target_class {
  # Find the class to add the methods to. I'm assuming that it would be
  # the first class in the caller() stack that's not a subsclass of
  # MethodMaker. If for some reason a sub-class of MethodMaker also
  # wanted to use MethodMaker it could redefine ima_method_maker to
  # return a false value and then $class would be set to it.
  my $class;
  my $i = 0;
  while (1) {
    $class = (caller($i))[0];
    $class->isa('Class::MethodMaker') or last;
    $i++;
  }
  $TargetClass = $class;
}

sub import {
  my ($class, @args) = @_;

  # Set a bit of syntactic sugar if desired which allows us to say things
  # like:
  #
  #   make methods
  #     get_set => [ qw / foo bar baz / ],
  #     list    => [ qw / a b c / ];

  if (defined $args[0] and $args[0] eq '-sugar') {
    shift @args;
    *methods:: = *Class::MethodMaker::;
  }
  
  @args and $class->make(@args);
}

sub make {
  my ($method_maker_class, @args) = @_;

  $method_maker_class->find_target_class; # sets $TargetClass

  # We have to initialize these before we run any of the
  # meta-methods. (At least the anon lists, so they get captured properly
  # in the closures.
  $BooleanPos{$TargetClass} ||= 0;
  $BooleanFields{$TargetClass} ||= [];
  $StructPos{$TargetClass} ||= 0;
  $StructFields{$TargetClass} ||= [];
  
  # make generic methods. The list passed to import should alternate
  # between the names of the meta-method to call to generate the methods
  # and either a scalar arg or a ARRAY ref to a list of args.

  # Each meta-method is responsible for calling install_methods() to get
  # it's methods installed.
  my ($meta_method, $arg);
  while (1) {
    $meta_method = shift @args or last;
    $arg = shift @args or
      croak "No arg for $meta_method in import of $method_maker_class.\n";

    my @args = ref($arg) ? @$arg : ($arg);
    $method_maker_class->$meta_method(@args);
  }
}

sub install_methods {
  my ($class, %methods) = @_;

  no strict 'refs';
#  print STDERR "CLASS: $class\n";
  $TargetClass || $class->find_target_class;
  my $package = $TargetClass . "::";
  
  my ($name, $code);
  while (($name, $code) = each %methods) {
    # add the method unless it's already defined (which should only
    # happen in the case of static methods, I think.)
    
    *{"$package$name"} = $code unless defined *{"$package$name"}{CODE};
  }
}

1;

__END__



## GENERIC METHODS ##

=head1 SUPPORTED METHOD TYPES

=head2 new

Creates a basic constructor.

Takes a single string or a reference to an array of strings as its
argument. For each string creates a method of the form:

    sub <string> {
      my ($class, @args) = @_;
      my $self = {};
      bless $self, $class;
    }

=cut

sub new {
  my ($class, @args) = @_;
  my %methods;
  foreach (@args) {
    $methods{$_} = sub {
      my ($class) = @_;
      my $self = {};
      bless $self, $class;
    };
  }
  $class->install_methods(%methods);
}

=head2 new_with_init

Creates a basic constructor which calls a method named init after
instatiating the object. The I<init>() method should be defined in the class
using MethodMaker.

Takes a single string or a reference to an array of strings as its
argument. For each string creates a method of the form listed below.

    sub <string> {
      my ($class, @args) = @_;
      my $self = {};
      bless $self, $class;
      $self->init(@args);
      $self;
    }

=cut

sub new_with_init {
  my ($class, @args) = @_;
  my %methods;
  foreach (@args) {
    my $field = $_;
    $methods{$field} = sub {
      my ($class, @args) = @_;
      my $self = {};
      bless $self, $class;
      $self->init(@args);
      $self;
    };
  }
  $class->install_methods(%methods);
}

=head2 new_hash_init

Creates a basic constructor which accepts a hash of slot-name/value pairs
with which to initialize the object. The slot-names are interpreted as
the names of methods that can be called on the object after it is created
and the values are the arguments to be passed to those methods.

Takes a single string or a reference to an array of strings as its
argument. For each string creates a method of the form listed below. Note
that this method can be called on an existing objec, which allows it to
be combined with new_with_init (see above) to provide some default
values. (Basically, declare a new_with_init method, say 'new' and a
new_hash_init method, for example, 'hash_init' and then in the init
method, you can call modify or add to the %args hash and then call
hash_init.)

    sub <string> {
      my ($class, %args) = @_;
      my $self = {};
      bless $self, $class;
      foreach (keys %args) {
	$self->$_($args{$_});
      }
      $self;
    }

=cut

sub new_hash_init {
  my ($class, @args) = @_;
  my %methods;
  foreach (@args) {
    $methods{$_} = sub {
      my ($class, %args) = @_;
      my $self = ref($class) ? $class : bless {}, $class;

      foreach (keys %args) {
	$self->$_($args{$_});
      }
      $self;
    };
  }
  $class->install_methods(%methods);
}

=head2 get_set

Takes a single string or a reference to an array of strings as its
argument. For each string, x creates two methods:

  sub x {
    my ($self, $new) = @_;
    defined $new and $self->{$name} = $new;
    $self->{$name};
  }

  sub clear_x
    my ($self) = @_;
    $self->{$name} = undef;
  }

This is your basic get/set method, and can be used for slots containing
any scalar value, including references to non-scalar data. Note, however,
that MethodMaker has meta-methods that define more useful sets of methods
for slots containing references to lists, hashes, and objects.

=cut

sub get_set {
  my ($class, @args) = @_;
  my %methods;
  foreach (@args) {
    my $name = $_;
    $methods{$name} = sub {
      my ($self, $new) = @_;
      defined $new and $self->{$name} = $new;
      $self->{$name};
    };
     
    $methods{"clear_$name"} = sub {
      my ($self) = @_;
      $self->{$name} = undef;
    };
  }
  $class->install_methods(%methods);
}

=head2 get_concat

Like get_set except sets don't clear out the original value, but instead
concatenate the new value to the existing one. Thus these slots are only
good for plain scalars. Also, like get_set, defines clear_foo method.

=cut

sub get_concat {
  my ($class, @args) = @_;
  my %methods;
  foreach (@args) {
    my $name = $_;
    $methods{$name} = sub {
      my ($self, $new) = @_;
      $self->{$name} ||= "";
      defined $new and $self->{$name} .= $new;
      $self->{$name};
    };

    $methods{"clear_$name"} = sub {
      my ($self) = @_;
      $self->{$name} = undef;
    };
  }
  $class->install_methods(%methods);
}

=head2 grouped_fields

Creates get/set methods like get_set but also defines a method which
returns a list of the slots in the group.

  grouped_fields methods
    some_group => [ qw / field1 field2 field3 / ];

Its argument list is parsed as a hash of group-name => field-list
pairs. Get-set methods are defined for all the fields and a method with
the name of the group is defined which returns the list of fields in the
group.

=cut

sub grouped_fields {
  my ($class, %args) = @_;
  my %methods;
  foreach (keys %args) {
    my @slots = @{$args{$_}};
    $class->get_set(@slots);
    $methods{$_} = sub { @slots };
  }
  $class->install_methods(%methods);
}

=head2 object

Creates methods for accessing a slot that contains an object of a given
class as well as methods to automatically pass method calls onto the
object stored in that slot.

    object => [
	       'Foo' => 'phooey',
	       'Bar' => [ qw / bar1 bar2 bar3 / ],
	       'Baz' => {
                         slot => 'foo',
                         comp_mthds => [ qw / bar baz / ]
                        },
              ];


This is a hairy one. The main argument should be a reference to an
array. The array should contain pairs of class => sub-argument
pairs. The sub-argument's are further parsed thusly:

If the sub-argument is a simple string or a reference to an array of
strings (as is the case for Foo and Bar above), for each string a get/set
method is created that can store an object of that class. (The get/set
method, if called with a reference to an object of the given class as the
first argument, stores it in the slot. If the slot isn't filled yet it
creates an object by calling the given class's new method. Any arguments
passed to the get/set method are passed on to new. In all cases the
object now stored in the slot is returned.

If the sub-argument is a ref to a hash (as with Baz, above) then the
hash should have two keys: slot and comp_mthds. The value indexed by
'slot' will be interpreted as the is in (a). The value or values (ref to
an array if plural) indexed by 'comp_mthds' are the names of methods
which should be "inherited" from the object stored in the slot. That is,
using the example above, a method, foo, is created in the class that
calls MethodMaker, which can get and set the value of a slot containing
an object of class Baz. Class Baz in turn defines two methods, 'bar', and
'baz'. Two more methods are created in the class using MethodMaker, named
'bar' and 'baz' which result in a call to the 'bar' and 'baz' methods,
through the Baz object stored in slot foo.

=cut

sub object {
  my ($class, @args) = @_;
  my %methods;

  while (@args) {
    my $class = shift @args;
    my $list = shift @args or die "No slot names for $class";
    my @list;

    my $ref = ref $list;
    if ($ref eq 'HASH') {
      my $name = $list->{'slot'};
      my $composites =  $list->{'forward'} || $list->{'comp_mthds'};
      @list = ($name);
      my @composites = ref($composites) eq 'ARRAY'
	? @$composites : ($composites);
      my $meth;
      foreach $meth (@composites) {
	$methods{$meth} =
	  sub {
	    my ($self, @args) = @_;
	    $self->$name()->$meth(@args);
	  };
      }
    } else {
      @list = ref($list) eq 'ARRAY' ? @$list : ($list);
    }

    foreach (@list) {
      my $name = $_;
      my $type = $class; # Hmmm. We have to do this for the closure to
                         # work. I.e. using $class in the closure dosen't
                         # work. Someday I'll actually understand scoping
                         # in Perl. [ Uh, is this true? 11/11/96 -PBS ]
      $methods{$name} = sub {
	my ($self, @args) = @_;
	if (ref $args[0] eq $class) { # This is sub-optimal. We should
                                      # really use isa from UNIVERSAL.pm
                                      # to catch sub-classes too.
	  $self->{$name} = $args[0];
	} else {
	  defined $self->{$name} or $self->{$name} = $type->new(@args);
	}
	$self->{$name};
      };

      $methods{"delete_$name"} = sub {
	my ($self) = @_;
	$self->{$name} = undef;
      };
    }
  }
  $class = $class; # Huh? Without this line the next line doesn't work!
  $class->install_methods(%methods);
}

sub forward {
  my ($class, %args) = @_;
  my %methods;

  foreach (keys %args) {
    my $slot = $_;
    my @methods = @{$args{$_}};
    foreach (@methods) {
      my $field = $_;
      $methods{$field} = sub {
	my ($self, @args) = @_;
	$self->$slot()->$field(@args);
      };
    }
  }
  $class->install_methods(%methods);
}


=head2 boolean

  boolean => [ qw / foo bar baz / ]

Creates methods for setting, checking and clearing flags. All flags
created with this meta-method are stored in a single vector for space
efficiency. The argument to boolean should be a string or a reference to
an array of strings. For each string x it defines several methods: x,
set_x, and clear x. x returns the value of the x-flag. If called with an
argument, it first sets the x-flag to the truth-value of the
argument. set_x is equivalent to x(1) and clear_x is equivalent to x(0).

Additionally, boolean defines three class method: I<bits>, which returns
the vector containing all of the bit fields (remember however that a
vector containing all 0 bits is still true), I<boolean_fields>, which returns
a list of all the flags by name, and I<bit_dump>, which returns a hash of
the flag-name/flag-value pairs.

=cut

sub boolean {
  my ($class, @args) = @_;
  my %methods;

  my $TargetClass = $class->get_target_class;

  my $boolean_fields =
    $BooleanFields{$TargetClass};

  $methods{'bits'} =
    sub {
      my ($self, $new) = @_;
      defined $new and $self->{'boolean'} = $new;
      $self->{'boolean'};
    };
  
  $methods{'bit_fields'} = sub { @$boolean_fields; };

  $methods{'bit_dump'} =
    sub {
      my ($self) = @_;
      map { ($_, $self->$_()) } @$boolean_fields;
    };
  
  foreach (@args) {
    my $field = $_;
    my $bfp = $BooleanPos{$TargetClass}++;
        # $boolean_pos a global declared at top of file. We need to make
        # a local copy because it will be captured in the closure and if
        # we capture the global version the changes to it will effect all
        # the closures. (Note also that it's value is reset with each
        # call to import_into_class.)
    push @$boolean_fields, $field;
        # $boolean_fields is also declared up above. It is used to store a
        # list of the names of all the bit fields.

    $methods{$field} =
      sub {
	my ($self, $on_off) = @_;
	defined $self->{'boolean'} or $self->{'boolean'} = "";
	if (defined $on_off) {
	  vec($self->{'boolean'}, $bfp, 1) = $on_off ? 1 : 0;
	}
	vec($self->{'boolean'}, $bfp, 1);
      };

    $methods{"set_$field"} =
      sub {
	my ($self) = @_;
	$self->$field(1);
      };

    $methods{"clear_$field"} =
      sub {
	my ($self) = @_;
	$self->$field(0);
      };
  }
  $class->install_methods(%methods);
}

=head2 struct

  struct => [ 'foo' => [ qw / foo bar baz / ] ];


XXX these docs aren't right yet.

Creates methods for setting, checking and clearing slots in a struct.
All the slots created with this meta-method are stored in a single array
for speed efficiency.  The argument to struct should be a string or a
reference to an array of strings. For each string x it defines two
methods: I<x> and I<clear_x>. x returns the value of the x-slot. If called with
an argument, it first sets the x-slot to the argument. clear_x sets the
slot to undef.

Additionally, struct defines three class method: I<struct>, which returns
the array containing all of the bit fields (remember however that a
vector containing all 0 bits is still true), I<boolean_fields>, which returns
a list of all the slots by name, and I<bit_dump>, which returns a hash of
the slot-name/slot-value pairs.

=cut

sub struct {
  my ($class, @args) = @_;
  my %methods;

  $class->get_target_class;

  my $struct_fields =
    $StructFields{$TargetClass};

  $methods{'struct_fields'} = sub { @$struct_fields; };

  $methods{'struct'} =
    sub {
      # For filling up the whole structure at once. The values must be
      # provided in the order they were declared.
      my ($self, @values) = @_;
      defined $self->{'struct'} or $self->{'struct'} = [];
      @values and @{$self->{'struct'}} = @values;
      @{$self->{'struct'}};
    };
  
  $methods{'struct_dump'} =
    sub {
      my ($self) = @_;
      map { ($_, $self->$_()) } @$struct_fields;
    };
  
  foreach (@args) {
    my $field = $_;
    my $sfp = $StructPos{$TargetClass}++;
        # $struct_pos a global declared at top of file. We need to make
        # a local copy because it will be captured in the closure and if
        # we capture the global version the changes to it will effect all
        # the closures. (Note also that its value is reset with each
        # call to import_into_class.)
    push @$struct_fields, $field;
        # $struct_fields is also declared up above. It is used to store a
        # list of the names of all the struct fields.

    $methods{$field} =
      sub {
	my ($self, $new) = @_;
	defined $self->{'struct'} or $self->{'struct'} = [];
	defined $new and $self->{'struct'}->[$sfp] = $new;
	$self->{'struct'}->[$sfp];
      };

    $methods{"clear_$field"} =
      sub {
	my ($self) = @_;
	defined $self->{'struct'} or $self->{'struct'} = [];
	$self->{'struct'}->[$sfp] = undef;
      };
  }
  $class->install_methods(%methods);
}


=head2 listed_attrib

  listed_attrib => [ qw / foo bar baz / ]

Like I<boolean>, I<listed_attrib> creates x, set_x, and clear_x
methods. However, it also defines a class method x_objects which returns
a list of the objects which presently have the x-flag set to
true. N.B. listed_attrib does not use the same space efficient
implementation as boolean, so boolean should be prefered unless the
x_objects method is actually needed.

=cut

sub listed_attrib {
  my ($class, @args) = @_;
  my %methods;

  foreach (@args) {
    my $field = $_;

    my %list = ();

    $methods{$field} =
      sub {
	my ($self, $on_off) = @_;
	if (defined $on_off) {
	  if ($on_off) {
	    $list{$self} = $self;
	  } else {
	    delete $list{$self};
	  }
	}
	$list{$self} ? 1 : 0;
      };

    $methods{"set_$field"} =
      sub {
	my ($self) = @_;
	$self->$field(1);
      };

    $methods{"clear_$field"} =
      sub {
	my ($self) = @_;
	$self->$field(0);
      };
    
    $methods{$field . "_objects"} =
      sub {
	values %list;
      };
  }
  $class->install_methods(%methods);
}

=head2 key_attrib

  key_attrib => [ qw / foo bar baz / ]

Creates get/set methods like get/set but also maintains a hash in which
each object is stored under the value of the field when the slot is
set. If an object has a slot set to a value which another object is
already set to the object currently set to that value has that slot set
to undef and the new object will be put into the hash under that
value. (I.e. only one object can have a given key. The method find_x is
defined which if called with any arguments returns a list of the objects
stored under those values in the hash. Called with no arguments, it
returns a reference to the hash.

=cut

sub key_attrib {
  my ($class, @args) = @_;
  my %methods;

  foreach (@args) {
    my $field = $_;
    my %list = ();

    $methods{$field} =
      sub {
	my ($self, $new) = @_;
	if (defined $new) {
	  # We need to set the value
	  if (defined $self->{$field}) {
	    # the object must be in the hash under its old value so
	    # that entry needs to be deleted
	    delete $list{$self->{$field}};
	  }
	  my $old;
	  if ($old = $list{$new}) {
	    # There's already an object stored under that value so we
	    # need to unset it's value
	    $old->{$field} = undef;
	  }

	  # Set our value to new
	  $self->{$field} = $new;

	  # Put ourself in the list under that value
	  $list{$new} = $self;
	}
	$self->{$field};
      };

    $methods{"clear_$field"} =
      sub {
	my ($self) = @_;
	delete $list{$self->{$field}};
	$self->{$field} = undef;
      };
    
    $methods{"find_$field"} =
      sub {
	my ($self, @args) = @_;
	if (scalar @args) {
	  return @list{@args};
	} else {
	  return \%list;
	}
      };
  }
  $class->install_methods(%methods);
}

=head2 key_with_create

  key_with_create => [ qw / foo bar baz / ]

Just like key_attrib except the find_x method is defined to call the new
method to create an object if there is no object already stored under any of the keys you give as arguments.

=cut

sub key_with_create {
  my ($class, @args) = @_;
  my %methods;

  foreach (@args) {
    my $field = $_;
    my %list = ();

    $methods{$field} =
      sub {
	my ($self, $new) = @_;
	if (defined $new) {
	  # We need to set the value
	  if (defined $self->{$field}) {
	    # the object must be in the hash under its old value so
	    # that entry needs to be deleted
	    delete $list{$self->{$field}};
	  }
	  my $old;
	  if ($old = $list{$new}) {
	    # There's already an object stored under that value so we
	    # need to unset it's value
	    $old->{$field} = undef;
	  }

	  # Set our value to new
	  $self->{$field} = $new;

	  # Put ourself in the list under that value
	  $list{$new} = $self;
	}
	$self->{$field};
      };
    
    $methods{"clear_$field"} =
      sub {
	my ($self) = @_;
	delete $list{$self->{$field}};
	$self->{$field} = undef;
      };
    
    $methods{"find_$field"} =
      sub {
	my ($class, @args) = @_;
	if (scalar @args) {
	  foreach (@args) {
	    $class->new->$field($_) unless defined $list{$_};
	  }
	  return @list{@args};
	} else {
	  return \%list;
	}
      };
  }
  $class->install_methods(%methods);
}

=head2 list

Creates several methods for dealing with slots containing list
data. Takes a string or a reference to an array of strings as its
argument and for each string, x, creates the methods: x, push_x, and
pop_x. The method x returns the list of values stored in the slot. In an
array context it returns them as an array and in a scalar context as a
reference to the array. If called with arguments, x will push them onto
the list. push_x and pop_x do about what you would expect.

=cut

sub list {
  my ($class, @args) = @_;
  my %methods;
  
  foreach (@args) {
    my $field = $_;

    $methods{$field} =
      sub {
	my ($self, @list) = @_;
	defined $self->{$field} or $self->{$field} = [];
	push @{$self->{$field}}, map { ref $_ eq 'ARRAY' ? @$_ : ($_) } @list;
	@{$self->{$field}}; # no it's not. That was exposing the
                            # implementation, plus you couldn't say
                            # scalar $obj->field to get the number of
                            # items in it.
      };

    $methods{"pop_$field"} =
      sub {
	my ($self) = @_;
	pop @{$self->{$field}}
      };

    $methods{"push_$field"} =
      sub {
	my ($self, @values) = @_;
	push @{$self->{$field}}, @values;
      };

    $methods{"shift_$field"} =
      sub {
	my ($self) = @_;
	shift @{$self->{$field}}
      };

    $methods{"unshift_$field"} =
      sub {
	my ($self, @values) = @_;
	unshift @{$self->{$field}}, @values;
      };

    $methods{"splice_$field"} =
      sub {
	my ($self, $offset, $len, @list) = @_;
	splice(@{$self->{$field}}, $offset, $len, @list);
      };

    $methods{"clear_$field"} =
      sub {
	my ($self) = @_;
	$self->{$field} = [];
      };

    $methods{"$ {field}_ref"} =
      sub {
	my ($self) = @_;
	$self->{$field};
      };
  }
  $class->install_methods(%methods);
}

=head2 hash

Creates a group of methods for dealing with hash data stored in a
slot. Takes a string or a reference to an array of strings and for each
string, x, creates: x, x_keys, x_values, and x_tally. Called with no
arguments x returns the hash stored in the slot, as a hash in an array
context or as a refernce in a scalar context. Called with one argument it
treats the argument as a key and returns the value stored under that key,
or as a list of keys (if it is a reference to a list) and returns the
list of values stored under those keys. Called with more than one
argument, treats them as a series of key/value pairs and adds them to the
hash. x_keys returns the keys of the hash, and x_values returns the list
of values. x_tally takes a list of arguments and for each scalar in the
list increments the value stored in the hash and returns a list of the
current (after the increment) values.

=cut

sub hash {
  my ($class, @args) = @_;
  my %methods;

  foreach (@args) {
    my $field = $_;

    $methods{$field} =
      sub {
	my ($self, @list) = @_;
	defined $self->{$field} or $self->{$field} = {};
	if (scalar @list == 1) {
	  my $key = shift @list;
	  if (ref $key) { # had better by an array ref
	    return @{$self->{$field}}{@$key};
	  } else {
	    return $self->{$field}->{$key};
	  }
	} else {
	  while (1) {
	    my $key = shift @list;
	    defined $key or last;
	    my $value = shift @list;
	    defined $value or carp "No value for key $key.";
	    $self->{$field}->{$key} = $value;
	  }
	  wantarray ? %{$self->{$field}} : $self->{$field};
	}
      };

    $methods{"$ {field}s"} = $methods{$field};

    $methods{$field . "_keys"} =
      sub {
	my ($self) = @_;
	keys %{$self->{$field}};
      };
    
    $methods{$field . "_values"} =
      sub {
	my ($self) = @_;
	values %{$self->{$field}};
      };

    $methods{$field . "_tally"} =
      sub {
	my ($self, @list) = @_;
	defined $self->{$field} or $self->{$field} = {};
	map { ++$self->{$field}->{$_} } @list;
      };

    $methods{"add_$field"} =
      sub {
	my ($self, $attrib, $value) = @_;
	$self->{$field}->{$attrib} = $value;
      };

    $methods{"clear_$field"} =
      sub {
	my ($self, $attrib) = @_;
	delete $ {$self->{$field}}{$attrib};
      };

    $methods{"add_$ {field}s"} =
      sub {
	my ($self, %attribs) = @_;
	my ($k, $v);
	while (($k, $v) = each %attribs) {
	  $self->{$field}->{$k} = $v;
	}
      };

    $methods{"clear_$ {field}s"} =
      sub {
	my ($self, @attribs) = @_;
	my $attrib;
	foreach $attrib (@attribs) {
	  delete $ {$self->{$field}}{$attrib};
	}
      };
  }
  $class->install_methods(%methods);
}

sub static_hash {
  my ($class, @args) = @_;
  my %methods;

  foreach (@args) {
    my $field = $_;
    my %hash;

    $methods{$field} =
      sub {
	my ($self, @list) = @_;
	if (scalar @list == 1) {
	  my $key = shift @list;
	  if (ref $key) { # had better by an array ref
	    return @hash{@$key};
	  } else {
	    return $hash{$key};
	  }
	} else {
	  while (1) {
	    my $key = shift @list;
	    defined $key or last;
	    my $value = shift @list;
	    defined $value or carp "No value for key $key.";
	    $hash{$key} = $value;
	  }
	  %hash;
	}
      };

    $methods{"$ {field}s"} = $methods{$field};

    $methods{$field . "_keys"} =
      sub {
	my ($self) = @_;
	keys %hash;
      };
    
    $methods{$field . "_values"} =
      sub {
	my ($self) = @_;
	values %hash;
      };

    $methods{$field . "_tally"} =
      sub {
	my ($self, @list) = @_;
	defined $self->{$field} or $self->{$field} = {};
	map { ++$hash{$_} } @list;
      };

    $methods{"add_$field"} =
      sub {
	my ($self, $attrib, $value) = @_;
	$hash{$attrib} = $value;
      };

    $methods{"clear_$field"} =
      sub {
	my ($self, $attrib) = @_;
	delete $hash{$attrib};
      };

    $methods{"add_$ {field}s"} =
      sub {
	my ($self, %attribs) = @_;
	my ($k, $v);
	while (($k, $v) = each %attribs) {
	  $hash{$k} = $v;
	}
      };

    $methods{"clear_$ {field}s"} =
      sub {
	my ($self, @attribs) = @_;
	my $attrib;
	foreach $attrib (@attribs) {
	  delete $hash{$attrib};
	}
      };
  }
  $class->install_methods(%methods);
}

=head2 code

  code => [ qw / foo bar baz / ]

Creates a slot that holds a code reference. Takes a string or a reference
to a list of string and for each string, x, creates a method B<x> which
if called with one argument which is a CODE reference, it installs that
code in the slot. Otherwise it runs the code stored in the slot with
whatever arguments (including none) were passed in.

=cut

sub code {
  my ($class, @args) = @_;
  my %methods;
  
  foreach (@args) {
    my $field = $_;
    
    $methods{$field} = sub {
      my ($self, @args) = @_;
      if (ref($args[0]) eq 'CODE') {
	# Set the function
	$self->{$field} = $args[0];
      } else {
	# Run the function on the given arguments
	&{$self->{$field}}(@args)
      }
    };
  }
  $class->install_methods(%methods);
}

=head2 method

  method => [ qw / foo bar baz / ]

Just like B<code>, except the code is called like a method, with $self as
it's first argument. Basically, you're creating a method which can be
different for each object. Which is sort of weird. But perhaps useful.

=cut

sub method {
  my ($class, @args) = @_;
  my %methods;

  foreach (@args) {
    my $field = $_;

    $methods{$field} = sub {
      my ($self, @args) = @_;
      if (ref($args[0]) eq 'CODE') {
	# Set the function
	$self->{$field} = $args[0];
      } else {
	# Run the function on the given arguments
	&{$self->{$field}}($self, @args)
      }
    };
  }
  $class->install_methods(%methods);
}

=head2 interface

  interface => [ qw / foo bar baz / ]

=cut

sub abstract {
  my ($class, @args) = @_;
  my %methods;
  
  $class->get_target_class;

  foreach (@args) {
    my $field = $_;
    $methods{$field} = sub {
      my ($self) = @_;
      my $calling_class = ref $self;
      die
	qq#Can't locate abstract method "$field" declared in #.
	qq#"$TargetClass", called from "$calling_class".\n#;
    };
  }
  $class->install_methods(%methods);
}


=head1 ADDDING NEW METHOD TYPES

MethodMaker is a class that can be inherited. A subclass can define new
method types by writing a method that returns a hash of
method_name/code-reference pairs.

For example a simple sub-class that defines a method type
upper_case_get_set might look like this:

  package Class::MethodMakerSubclass;

  use strict;
  use Class::MethodMaker;

  @Class::MethodMakerSubclass::ISA = qw ( Class::MethodMaker );

  sub upper_case_get_set {
    shift; # we don't need the class name
    my ($name) = @_;
    my %results;
    $results{$name} =
      sub {
	my ($self, $new) = @_;
	defined $new and $self->{$name} = uc $new;
	$self->{$name};
      };
    %results;
  }
  
  1;

=head1 VERSION

Class::MethodMaker v0.92

=cut

## EXPERIMENTAL META-METHODS

sub builtin_class {
  my ($class, $func, $arg) = @_;
  my @list = @$arg;
  my %results = ();
  my $field;
  
  $class->get_target_class;

  my $struct_fields =
    $StructFields{$TargetClass};

  # Cuz neither \&{"CORE::$func"} or $CORE::{$func} work ...  N.B. this
  # only works for core functions that take only one arg. But I can't
  # quite figure out how to pass in the list without it getting evaluated
  # in a scalar context. Hmmm.
  my $corefunc = eval "sub { scalar \@_ ? CORE::$func(shift) : CORE::$func }";

  $results{'new'} = sub {
    my ($class, @args) = @_;
    my $self = [];
    @$self = &$corefunc(@args);
    bless $self, $class;
  };

  $results{'fields'} = sub { @$struct_fields; };

  $results{'dump'} =
    sub {
      my ($self) = @_;
      map { ($_, $self->$_()) } @$struct_fields;
    };
  
  foreach $field (@list) {
    my $sfp = $StructPos{$TargetClass}++;
        # $struct_pos a global declared at top of file. We need to make
        # a local copy because it will be captured in the closure and if
        # we capture the global version the changes to it will effect all
        # the closures. (Note also that its value is reset with each
        # call to import_into_class.)
    push @$struct_fields, $field;
        # $struct_fields is also declared up above. It is used to store a
        # list of the names of all the struct fields.

    $results{$field} =
      sub {
	my ($self, $new) = @_;
	defined $new and $self->[$sfp] = $new;
	$self->[$sfp];
      };
  }
  $class->install_methods(%results);
}

sub method_maker {
  # This is crazy!!!
  my ($class, %args) = @_;
  my %methods;
  $class->set_target_class(caller);

  foreach (keys %args) {
    my $field = $_;
    my $sub = $args{$_};
    $methods{$field} = sub {
      my ($c, @a) = @_;
      my %m;

      foreach (@a) {
	my $f = $_;
	$m{$f} = $sub;
      }
      $c->install_methods(%m);
    }
  }
  $class->install_methods(%methods);
  $class->set_target_class(undef);
}


