package org.apache.torque.oid;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.List;

import org.apache.torque.TorqueException;
import org.apache.torque.adapter.Adapter;
import org.apache.torque.om.mapper.BigDecimalMapper;
import org.apache.torque.om.mapper.IntegerMapper;
import org.apache.torque.om.mapper.LongMapper;
import org.apache.torque.om.mapper.RecordMapper;
import org.apache.torque.om.mapper.StringMapper;
import org.apache.torque.sql.SqlBuilder;
import org.apache.torque.util.BasePeerImpl;

/**
 * This class serves as a common base class for the sequence-based and the
 * autoincrement-based id generators
 *
 * @author <a href="mailto:tv@apache.org">Thomas Vandahl</a>
 * @version $Id: AbstractIdGenerator.java 1448414 2013-02-20 21:06:35Z tfischer $
 */
public abstract class AbstractIdGenerator implements IdGenerator
{
    /** The adapter that knows the correct sql syntax */
    protected Adapter adapter;

    /**
     * The internal name of the Database that this Generator is connected to.
     */
    protected String databaseName = null;

    /**
     * Creates an IdGenerator which will work with the specified database.
     *
     * @param adapter the adapter that knows the correct sql syntax.
     * @param databaseName The name of the databaseName to find the correct
     *        schema.
     */
    public AbstractIdGenerator(final Adapter adapter, final String databaseName)
    {
        this.adapter = adapter;
        this.databaseName = databaseName;
    }

    /**
     * Returns the last ID used by this connection.
     *
     * @param connection The database connection to read the new id, not null.
     * @param keyInfo the name of the table for which the id is retrieved.
     *
     * @return An int with the new id.
     *
     * @throws TorqueException if a database error occurs.
     */
    public int getIdAsInt(Connection connection, Object keyInfo)
        throws TorqueException
    {
        return getId(connection, keyInfo, new IntegerMapper());
    }

    /**
     * Returns the last ID used by this connection.
     *
     * @param connection The database connection to read the new id, not null.
     * @param keyInfo the name of the table for which the id is retrieved.
     *
     * @return A long with the new id.
     *
     * @throws TorqueException if a database error occurs.
     */
    public long getIdAsLong(Connection connection, Object keyInfo)
        throws TorqueException
    {
        return getId(connection, keyInfo, new LongMapper());
    }

    /**
     * Returns the last ID used by this connection.
     *
     * @param connection The database connection to read the new id, not null.
     * @param keyInfo the name of the table for which the id is retrieved.
     *
     * @return A BigDecimal with the new id.
     *
     * @throws TorqueException if a database error occurs.
     */
    public BigDecimal getIdAsBigDecimal(Connection connection, Object keyInfo)
        throws TorqueException
    {
        return getId(connection, keyInfo, new BigDecimalMapper());
    }


    /**
     * Returns the last ID used by this connection.
     *
     * @param connection The database connection to read the new id, not null.
     * @param keyInfo the name of the table for which the id is retrieved.
     *
     * @return A String with the new id.
     *
     * @throws TorqueException if a database error occurs.
     */
    public String getIdAsString(Connection connection, Object keyInfo)
        throws TorqueException
    {
        return getId(connection, keyInfo, new StringMapper());
    }

    /**
     * A flag to determine the timing of the id generation
     *
     * @return a <code>boolean</code> value
     */
    public abstract boolean isPriorToInsert();

    /**
     * A flag to determine the timing of the id generation
     *
     * @return a <code>boolean</code> value
     */
    public abstract boolean isPostInsert();

    /**
     * A flag to determine whether a Connection is required to
     * generate an id.
     *
     * @return a <code>boolean</code> value
     */
    public abstract boolean isConnectionRequired();

    /**
     * Returns the last ID used by this connection.
     *
     * @param connection A Connection.
     * @param keyInfo an Object that contains additional info.
     * @param mapper The RecordMapper that maps from a ResultSet to the
     *        appropriate java object.
     *
     * @return The generated id.
     * @exception TorqueException if a database error occurs.
     */
    protected <T> T getId(
                Connection connection,
                Object keyInfo,
                RecordMapper<T> mapper)
            throws TorqueException
    {
        String tableName = SqlBuilder.getFullTableName(
                String.valueOf(keyInfo),
                databaseName);
        String idSql = adapter.getIDMethodSQL(tableName);

        BasePeerImpl<T> peer = new BasePeerImpl<T>(mapper, null, databaseName);
        List<T> result = peer.doSelect(idSql, connection);
        return result.get(0);
    }
}
