#ifndef SQUASHFS4_FS
#define SQUASHFS4_FS
/*
 * Squashfs
 *
 * Copyright (c) 2002, 2003, 2004, 2005, 2006, 2007, 2008
 * Phillip Lougher <phillip@lougher.demon.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * squashfs4_fs.h
 */

#define SQUASHFS4_CACHED_FRAGMENTS	CONFIG_SQUASHFS4_FRAGMENT_CACHE_SIZE
#define SQUASHFS4_MAJOR			4
#define SQUASHFS4_MINOR			0
#define SQUASHFS4_START			0

/* size of metadata (inode and directory) blocks */
#define SQUASHFS4_METADATA_SIZE		8192
#define SQUASHFS4_METADATA_LOG		13

/* default size of data blocks */
#define SQUASHFS4_FILE_SIZE		131072
#define SQUASHFS4_FILE_LOG		17

#define SQUASHFS4_FILE_MAX_SIZE		1048576
#define SQUASHFS4_FILE_MAX_LOG		20

/* Max number of uids and gids */
#define SQUASHFS4_IDS			65536

/* Max length of filename (not 255) */
#define SQUASHFS4_NAME_LEN		256

#define SQUASHFS4_INVALID_FRAG		(0xffffffffU)
#define SQUASHFS4_INVALID_BLK		(-1LL)

/* Filesystem flags */
#define SQUASHFS4_NOI			0
#define SQUASHFS4_NOD			1
#define SQUASHFS4_NOF			3
#define SQUASHFS4_NO_FRAG		4
#define SQUASHFS4_ALWAYS_FRAG		5
#define SQUASHFS4_DUPLICATE		6
#define SQUASHFS4_EXPORT			7

#define SQUASHFS4_BIT(flag, bit)		((flag >> bit) & 1)

#define SQUASHFS4_UNCOMPRESSED_INODES(flags)	SQUASHFS4_BIT(flags, \
						SQUASHFS4_NOI)

#define SQUASHFS4_UNCOMPRESSED_DATA(flags)	SQUASHFS4_BIT(flags, \
						SQUASHFS4_NOD)

#define SQUASHFS4_UNCOMPRESSED_FRAGMENTS(flags)	SQUASHFS4_BIT(flags, \
						SQUASHFS4_NOF)

#define SQUASHFS4_NO_FRAGMENTS(flags)		SQUASHFS4_BIT(flags, \
						SQUASHFS4_NO_FRAG)

#define SQUASHFS4_ALWAYS_FRAGMENTS(flags)	SQUASHFS4_BIT(flags, \
						SQUASHFS4_ALWAYS_FRAG)

#define SQUASHFS4_DUPLICATES(flags)		SQUASHFS4_BIT(flags, \
						SQUASHFS4_DUPLICATE)

#define SQUASHFS4_EXPORTABLE(flags)		SQUASHFS4_BIT(flags, \
						SQUASHFS4_EXPORT)

/* Max number of types and file types */
#define SQUASHFS4_DIR_TYPE		1
#define SQUASHFS4_REG_TYPE		2
#define SQUASHFS4_SYMLINK_TYPE		3
#define SQUASHFS4_BLKDEV_TYPE		4
#define SQUASHFS4_CHRDEV_TYPE		5
#define SQUASHFS4_FIFO_TYPE		6
#define SQUASHFS4_SOCKET_TYPE		7
#define SQUASHFS4_LDIR_TYPE		8
#define SQUASHFS4_LREG_TYPE		9
#define SQUASHFS4_LSYMLINK_TYPE		10
#define SQUASHFS4_LBLKDEV_TYPE		11
#define SQUASHFS4_LCHRDEV_TYPE		12
#define SQUASHFS4_LFIFO_TYPE		13
#define SQUASHFS4_LSOCKET_TYPE		14

/* Flag whether block is compressed or uncompressed, bit is set if block is
 * uncompressed */
#define SQUASHFS4_COMPRESSED_BIT		(1 << 15)

#define SQUASHFS4_COMPRESSED_SIZE(B)	(((B) & ~SQUASHFS4_COMPRESSED_BIT) ? \
		(B) & ~SQUASHFS4_COMPRESSED_BIT :  SQUASHFS4_COMPRESSED_BIT)

#define SQUASHFS4_COMPRESSED(B)		(!((B) & SQUASHFS4_COMPRESSED_BIT))

#define SQUASHFS4_COMPRESSED_BIT_BLOCK	(1 << 24)

#define SQUASHFS4_COMPRESSED_SIZE_BLOCK(B)	((B) & \
						~SQUASHFS4_COMPRESSED_BIT_BLOCK)

#define SQUASHFS4_COMPRESSED_BLOCK(B)	(!((B) & SQUASHFS4_COMPRESSED_BIT_BLOCK))

/*
 * Inode number ops.  Inodes consist of a compressed block number, and an
 * uncompressed offset within that block
 */
#define SQUASHFS4_INODE_BLK(A)		((unsigned int) ((A) >> 16))

#define SQUASHFS4_INODE_OFFSET(A)	((unsigned int) ((A) & 0xffff))

#define SQUASHFS4_MKINODE(A, B)		((long long)(((long long) (A)\
					<< 16) + (B)))

/* Translate between VFS mode and squashfs mode */
#define SQUASHFS4_MODE(A)		((A) & 0xfff)

/* fragment and fragment table defines */
#define SQUASHFS4_FRAGMENT_BYTES(A)	\
				((A) * sizeof(struct squashfs4_fragment_entry))

#define SQUASHFS4_FRAGMENT_INDEX(A)	(SQUASHFS4_FRAGMENT_BYTES(A) / \
					SQUASHFS4_METADATA_SIZE)

#define SQUASHFS4_FRAGMENT_INDEX_OFFSET(A)	(SQUASHFS4_FRAGMENT_BYTES(A) % \
						SQUASHFS4_METADATA_SIZE)

#define SQUASHFS4_FRAGMENT_INDEXES(A)	((SQUASHFS4_FRAGMENT_BYTES(A) + \
					SQUASHFS4_METADATA_SIZE - 1) / \
					SQUASHFS4_METADATA_SIZE)

#define SQUASHFS4_FRAGMENT_INDEX_BYTES(A)	(SQUASHFS4_FRAGMENT_INDEXES(A) *\
						sizeof(u64))

/* inode lookup table defines */
#define SQUASHFS4_LOOKUP_BYTES(A)	((A) * sizeof(u64))

#define SQUASHFS4_LOOKUP_BLOCK(A)	(SQUASHFS4_LOOKUP_BYTES(A) / \
					SQUASHFS4_METADATA_SIZE)

#define SQUASHFS4_LOOKUP_BLOCK_OFFSET(A)	(SQUASHFS4_LOOKUP_BYTES(A) % \
					SQUASHFS4_METADATA_SIZE)

#define SQUASHFS4_LOOKUP_BLOCKS(A)	((SQUASHFS4_LOOKUP_BYTES(A) + \
					SQUASHFS4_METADATA_SIZE - 1) / \
					SQUASHFS4_METADATA_SIZE)

#define SQUASHFS4_LOOKUP_BLOCK_BYTES(A)	(SQUASHFS4_LOOKUP_BLOCKS(A) *\
					sizeof(u64))

/* uid/gid lookup table defines */
#define SQUASHFS4_ID_BYTES(A)		((A) * sizeof(unsigned int))

#define SQUASHFS4_ID_BLOCK(A)		(SQUASHFS4_ID_BYTES(A) / \
					SQUASHFS4_METADATA_SIZE)

#define SQUASHFS4_ID_BLOCK_OFFSET(A)	(SQUASHFS4_ID_BYTES(A) % \
					SQUASHFS4_METADATA_SIZE)

#define SQUASHFS4_ID_BLOCKS(A)		((SQUASHFS4_ID_BYTES(A) + \
					SQUASHFS4_METADATA_SIZE - 1) / \
					SQUASHFS4_METADATA_SIZE)

#define SQUASHFS4_ID_BLOCK_BYTES(A)	(SQUASHFS4_ID_BLOCKS(A) *\
					sizeof(u64))

/* cached data constants for filesystem */
#define SQUASHFS4_CACHED_BLKS		8

#define SQUASHFS4_MAX_FILE_SIZE_LOG	64

#define SQUASHFS4_MAX_FILE_SIZE		(1LL << \
					(SQUASHFS4_MAX_FILE_SIZE_LOG - 2))

#define SQUASHFS4_MARKER_BYTE		0xff

/* meta index cache */
#define SQUASHFS4_META_INDEXES	(SQUASHFS4_METADATA_SIZE / sizeof(unsigned int))
#define SQUASHFS4_META_ENTRIES	127
#define SQUASHFS4_META_SLOTS	8

struct meta_entry_sq4 {
	u64			data_block;
	unsigned int		index_block;
	unsigned short		offset;
	unsigned short		pad;
};

struct meta_index_sq4 {
	unsigned int		inode_number;
	unsigned int		offset;
	unsigned short		entries;
	unsigned short		skip;
	unsigned short		locked;
	unsigned short		pad;
	struct meta_entry_sq4	meta_entry_sq4[SQUASHFS4_META_ENTRIES];
};


/*
 * definitions for structures on disk
 */
#define ZLIB_COMPRESSION	 1

struct squashfs4_super_block {
	__le32			s_magic;
	__le32			inodes;
	__le32			mkfs_time;
	__le32			block_size;
	__le32			fragments;
	__le16			compression;
	__le16			block_log;
	__le16			flags;
	__le16			no_ids;
	__le16			s_major;
	__le16			s_minor;
	__le64			root_inode;
	__le64			bytes_used;
	__le64			id_table_start;
	__le64			xattr_table_start;
	__le64			inode_table_start;
	__le64			directory_table_start;
	__le64			fragment_table_start;
	__le64			lookup_table_start;
};

struct squashfs4_dir_index {
	__le32			index;
	__le32			start_block;
	__le32			size;
	unsigned char		name[0];
};

struct squashfs4_base_inode {
	__le16			inode_type;
	__le16			mode;
	__le16			uid;
	__le16			guid;
	__le32			mtime;
	__le32	 		inode_number;
};

struct squashfs4_ipc_inode {
	__le16			inode_type;
	__le16			mode;
	__le16			uid;
	__le16			guid;
	__le32			mtime;
	__le32	 		inode_number;
	__le32			nlink;
};

struct squashfs4_dev_inode {
	__le16			inode_type;
	__le16			mode;
	__le16			uid;
	__le16			guid;
	__le32			mtime;
	__le32	 		inode_number;
	__le32			nlink;
	__le32			rdev;
};

struct squashfs4_symlink_inode {
	__le16			inode_type;
	__le16			mode;
	__le16			uid;
	__le16			guid;
	__le32			mtime;
	__le32	 		inode_number;
	__le32			nlink;
	__le32			symlink_size;
	char			symlink[0];
};

struct squashfs4_reg_inode {
	__le16			inode_type;
	__le16			mode;
	__le16			uid;
	__le16			guid;
	__le32			mtime;
	__le32	 		inode_number;
	__le32			start_block;
	__le32			fragment;
	__le32			offset;
	__le32			file_size;
	__le16			block_list[0];
};

struct squashfs4_lreg_inode {
	__le16			inode_type;
	__le16			mode;
	__le16			uid;
	__le16			guid;
	__le32			mtime;
	__le32	 		inode_number;
	__le64			start_block;
	__le64			file_size;
	__le64			sparse;
	__le32			nlink;
	__le32			fragment;
	__le32			offset;
	__le32			xattr;
	__le16			block_list[0];
};

struct squashfs4_dir_inode {
	__le16			inode_type;
	__le16			mode;
	__le16			uid;
	__le16			guid;
	__le32			mtime;
	__le32	 		inode_number;
	__le32			start_block;
	__le32			nlink;
	__le16			file_size;
	__le16			offset;
	__le32			parent_inode;
};

struct squashfs4_ldir_inode {
	__le16			inode_type;
	__le16			mode;
	__le16			uid;
	__le16			guid;
	__le32			mtime;
	__le32	 		inode_number;
	__le32			nlink;
	__le32			file_size;
	__le32			start_block;
	__le32			parent_inode;
	__le16			i_count;
	__le16			offset;
	__le32			xattr;
	struct squashfs4_dir_index	index[0];
};

union squashfs4_inode {
	struct squashfs4_base_inode		base;
	struct squashfs4_dev_inode		dev;
	struct squashfs4_symlink_inode		symlink;
	struct squashfs4_reg_inode		reg;
	struct squashfs4_lreg_inode		lreg;
	struct squashfs4_dir_inode		dir;
	struct squashfs4_ldir_inode		ldir;
	struct squashfs4_ipc_inode		ipc;
};

struct squashfs4_dir_entry {
	__le16			offset;
	__le16			inode_number;
	__le16			type;
	__le16			size;
	char			name[0];
};

struct squashfs4_dir_header {
	__le32			count;
	__le32			start_block;
	__le32			inode_number;
};

struct squashfs4_fragment_entry {
	__le64			start_block;
	__le32			size;
	unsigned int		unused;
};

#endif
