/*
 * Copyright (C) 2021 Codership Oy <info@codership.com>
 *
 * This file is part of wsrep-API.
 *
 * Wsrep-API is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * Wsrep-API is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with wsrep-API.  If not, see <https://www.gnu.org/licenses/>.
 */

/** @file wsrep_event_service.h
 *
 * This file defines interface for various unordered events generated by the
 * cluster or the provider.
 *
 * An event has a name and a payload, both are null-terminated strings.
 * (It is intended that payload is a JSON encoded structure).
 * The name serves to distinguish the events to pass them to respective
 * handlers.
 *
 * The provider which is capable of using the service interface v1 must
 * export the following functions.
 *
 * int wsrep_init_event_service_v1(wsrep_event_service_v1_t*)
 * void wsrep_deinit_event_service_v1()
 *
 * which can be probed by the application.
 *
 * The application must initialize the service via above init function
 * before the provider is initialized via wsrep->init(). The deinit
 * function must be called after the provider side resources have been
 * released via wsrep->free().
 */

#ifndef WSREP_EVENT_SERVICE_H
#define WSREP_EVENT_SERVICE_H

#include "wsrep_api.h"

#include <sys/types.h> /* posix size_t */

#ifdef __cplusplus
extern "C"
{
#endif /* __cplusplus */

/**
 * Type tag for application defined event processing context.
 *
 * Application may pass pointer to the context when initializing
 * the event service. This pointer is passed a first parameter for
 * each service call.
 */
typedef struct wsrep_event_context wsrep_event_context_t;


/**
 * Process an event
 *
 * @param name  event name
 * @param value JSON enconded event value
 *
 * @return void, it is up to the receiver to decide what to do about
 *         possible error.
 */
typedef void (*wsrep_event_cb_t)(wsrep_event_context_t*,
                                 const char* name, const char* value);


/**
 * Event service struct.
 *
 * A pointer to this struct must be passed to the call to
 * wsrep_init_event_service_v1.
 *
 * The application must provide implementation to all functions defined
 * in this struct.
 */
typedef struct wsrep_event_service_v1_st
{
    /* Event receiver callback */
    wsrep_event_cb_t event_cb;
    /* Pointer to application defined event context. */
    wsrep_event_context_t* context;
} wsrep_event_service_v1_t;

#ifdef __cplusplus
}
#endif /* __cplusplus */


#define WSREP_EVENT_SERVICE_INIT_FUNC_V1 "wsrep_init_event_service_v1"
#define WSREP_EVENT_SERVICE_DEINIT_FUNC_V1 "wsrep_deinit_event_service_v1"

#endif /* WSREP_EVENT_SERVICE_H */

