/*	$NetBSD: zynq_cemac.c,v 1.12 2025/10/15 01:21:39 thorpej Exp $	*/
/*-
 * Copyright (c) 2015  Genetec Corporation.  All rights reserved.
 * Written by Hashimoto Kenichi for Genetec Corporation.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: zynq_cemac.c,v 1.12 2025/10/15 01:21:39 thorpej Exp $");

#include <sys/param.h>

#include <sys/bus.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/intr.h>
#include <sys/systm.h>

#include <dev/cadence/if_cemacvar.h>

#include <net/if.h>
#include <net/if_media.h>
#include <net/if_ether.h>

#include <dev/fdt/fdtvar.h>

#include <dev/mii/mii.h>
#include <dev/mii/miivar.h>

static const struct device_compatible_entry compat_data[] = {
	{ .compat = "cdns,zynq-gem" },
	DEVICE_COMPAT_EOL
};

static int
cemac_get_phyid(const int phandle)
{
	bus_addr_t addr;
	int phy_phandle;

	phy_phandle = fdtbus_get_phandle(phandle, "phy");
	if (phy_phandle == -1)
		phy_phandle = fdtbus_get_phandle(phandle, "phy-handle");
	if (phy_phandle == -1)
		return MII_PHY_ANY;

	if (fdtbus_get_reg(phy_phandle, 0, &addr, NULL) != 0)
		return MII_PHY_ANY;

	return (int)addr;
}

static int
cemac_match(device_t parent, cfdata_t cfdata, void *aux)
{
	struct fdt_attach_args * const faa = aux;

	return of_compatible_match(faa->faa_phandle, compat_data);
}

static void
cemac_attach(device_t parent, device_t self, void *aux)
{
	struct fdt_attach_args * const faa = aux;
	struct cemac_softc *sc = device_private(self);
	const int phandle = faa->faa_phandle;
	char intrstr[128];
	bus_addr_t addr;
	bus_size_t size;
	int error;

	if (fdtbus_get_reg(phandle, 0, &addr, &size) != 0) {
		aprint_error(": couldn't get registers\n");
		return;
	}

	error = bus_space_map(faa->faa_bst, addr, size, 0, &sc->sc_ioh);
	if (error) {
		aprint_error(": failed to map register %#lx@%#lx: %d\n",
		    size, addr, error);
		return;
	}

	if (!fdtbus_intr_str(phandle, 0, intrstr, sizeof(intrstr))) {
		aprint_error(": failed to decode interrupt\n");
		return;
	}

	if (fdtbus_intr_establish(phandle, 0, IPL_NET, 0, cemac_intr,
				  device_private(self)) == NULL) {
		aprint_error(": failed to establish interrupt on %s\n",
		    intrstr);
		return;
	}

	sc->sc_dev = self;
	sc->sc_iot = faa->faa_bst;
	sc->sc_dmat = faa->faa_dmat;
	sc->sc_phyno = cemac_get_phyid(phandle);
	sc->cemac_flags = CEMAC_FLAG_GEM;

	cemac_attach_common(sc);
	aprint_normal_dev(self, "interrupting on %s\n", intrstr);
}


CFATTACH_DECL_NEW(cemac, sizeof(struct cemac_softc),
    cemac_match, cemac_attach, NULL, NULL);
