/*	$NetBSD: bcm283x_platform.c,v 1.52 2025/10/03 14:05:11 thorpej Exp $	*/

/*-
 * Copyright (c) 2017 Jared D. McNeill <jmcneill@invisible.ca>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: bcm283x_platform.c,v 1.52 2025/10/03 14:05:11 thorpej Exp $");

#include "opt_arm_debug.h"
#include "opt_bcm283x.h"
#include "opt_cpuoptions.h"
#include "opt_ddb.h"
#include "opt_evbarm_boardtype.h"
#include "opt_kgdb.h"
#include "opt_fdt.h"
#include "opt_rpi.h"
#include "opt_vcprop.h"

#include "sdhc.h"
#include "bcmsdhost.h"
#include "bcmdwctwo.h"
#include "bcmspi.h"
#include "bsciic.h"
#include "plcom.h"
#include "com.h"
#include "genfb.h"
#include "ukbd.h"

#include <sys/param.h>
#include <sys/bus.h>
#include <sys/cpu.h>
#include <sys/device.h>
#include <sys/endian.h>
#include <sys/kmem.h>
#include <sys/termios.h>

#include <net/if_ether.h>

#include <prop/proplib.h>

#include <dev/fdt/fdtvar.h>
#include <dev/fdt/fdt_console.h>
#include <dev/fdt/fdt_platform.h>

#include <uvm/uvm_extern.h>

#include <machine/bootconfig.h>

#include <arm/armreg.h>
#include <arm/cpufunc.h>

#include <libfdt.h>

#include <arm/broadcom/bcm2835reg.h>
#include <arm/broadcom/bcm2835var.h>
#include <arm/broadcom/bcm283x_platform.h>
#include <arm/broadcom/bcm2835_intr.h>
#include <arm/broadcom/bcm2835_mbox.h>
#include <arm/broadcom/bcm2835_pmwdogvar.h>

#include <evbarm/dev/plcomreg.h>
#include <evbarm/dev/plcomvar.h>
#include <evbarm/fdt/machdep.h>

#include <dev/ic/ns16550reg.h>
#include <dev/ic/comreg.h>

#include <evbarm/rpi/vcio.h>
#include <evbarm/rpi/vcpm.h>
#include <evbarm/rpi/vcprop.h>

#include <arm/fdt/arm_fdtvar.h>

#include <arm/cortex/gtmr_var.h>

#if NGENFB > 0
#include <dev/videomode/videomode.h>
#include <dev/videomode/edidvar.h>
#include <dev/wscons/wsconsio.h>
#endif

#if NUKBD > 0
#include <dev/usb/ukbdvar.h>
#endif

#ifdef DDB
#include <machine/db_machdep.h>
#include <ddb/db_sym.h>
#include <ddb/db_extern.h>
#endif

#define RPI_CPU_MAX	4

void bcm2835_platform_early_putchar(char c);
void bcm2835_aux_platform_early_putchar(char c);
void bcm2836_platform_early_putchar(char c);
void bcm2837_platform_early_putchar(char c);
void bcm2711_platform_early_putchar(char c);

extern void bcmgenfb_set_console_dev(device_t dev);
void bcmgenfb_set_ioctl(int(*)(void *, void *, u_long, void *, int, struct lwp *));
extern void bcmgenfb_ddb_trap_callback(int where);
static int rpi_ioctl(void *, void *, u_long, void *, int, lwp_t *);

extern struct bus_space arm_generic_bs_tag;
extern struct bus_space arm_generic_a4x_bs_tag;

/* Prototypes for all the bus_space structure functions */
bs_protos(arm_generic);
bs_protos(arm_generic_a4x);
bs_protos(bcm2835);
bs_protos(bcm2835_a4x);
bs_protos(bcm2836);
bs_protos(bcm2836_a4x);
bs_protos(bcm2711);
bs_protos(bcm2711_a4x);

struct bus_space bcm2835_bs_tag;
struct bus_space bcm2835_a4x_bs_tag;
struct bus_space bcm2836_bs_tag;
struct bus_space bcm2836_a4x_bs_tag;
struct bus_space bcm2711_bs_tag;
struct bus_space bcm2711_a4x_bs_tag;

static paddr_t bcm2835_bus_to_phys(bus_addr_t);
static paddr_t bcm2836_bus_to_phys(bus_addr_t);
static paddr_t bcm2711_bus_to_phys(bus_addr_t);

#ifdef VERBOSE_INIT_ARM
#define VPRINTF(...)	printf(__VA_ARGS__)
#else
#define VPRINTF(...)	__nothing
#endif

static paddr_t
bcm2835_bus_to_phys(bus_addr_t ba)
{

	/* Attempt to find the PA device mapping */
	if (ba >= BCM283X_PERIPHERALS_BASE_BUS &&
	    ba < BCM283X_PERIPHERALS_BASE_BUS + BCM283X_PERIPHERALS_SIZE)
		return BCM2835_PERIPHERALS_BUS_TO_PHYS(ba);

	return ba & ~BCM2835_BUSADDR_CACHE_MASK;
}

static paddr_t
bcm2836_bus_to_phys(bus_addr_t ba)
{

	/* Attempt to find the PA device mapping */
	if (ba >= BCM283X_PERIPHERALS_BASE_BUS &&
	    ba < BCM283X_PERIPHERALS_BASE_BUS + BCM283X_PERIPHERALS_SIZE)
		return BCM2836_PERIPHERALS_BUS_TO_PHYS(ba);

	if (ba >= BCM2836_ARM_LOCAL_BASE &&
	    ba < BCM2836_ARM_LOCAL_BASE + BCM2836_ARM_LOCAL_SIZE)
		return ba;

	return ba & ~BCM2835_BUSADDR_CACHE_MASK;
}

static paddr_t
bcm2711_bus_to_phys(bus_addr_t ba)
{

	/* Attempt to find the PA device mapping */
	if (ba >= BCM283X_PERIPHERALS_BASE_BUS &&
	    ba < BCM283X_PERIPHERALS_BASE_BUS + BCM283X_PERIPHERALS_SIZE)
		return BCM2711_PERIPHERALS_BUS_TO_PHYS(ba);

	if (ba >= BCM2711_SCB_BASE_BUS &&
	    ba < BCM2711_SCB_BASE_BUS + BCM2711_SCB_SIZE)
		return BCM2711_SCB_BUS_TO_PHYS(ba);

	if (ba >= BCM2711_ARM_LOCAL_BASE_BUS &&
	    ba < BCM2711_ARM_LOCAL_BASE_BUS + BCM2711_ARM_LOCAL_SIZE)
		return BCM2711_ARM_LOCAL_BUS_TO_PHYS(ba);

	return ba & ~BCM2835_BUSADDR_CACHE_MASK;
}

int
bcm2835_bs_map(void *t, bus_addr_t ba, bus_size_t size, int flag,
    bus_space_handle_t *bshp)
{
	const paddr_t pa = bcm2835_bus_to_phys(ba);

	return bus_space_map(&arm_generic_bs_tag, pa, size, flag, bshp);
}

paddr_t
bcm2835_bs_mmap(void *t, bus_addr_t ba, off_t offset, int prot, int flags)
{
	const paddr_t pa = bcm2835_bus_to_phys(ba);

	return bus_space_mmap(&arm_generic_bs_tag, pa, offset, prot, flags);
}

paddr_t
bcm2835_a4x_bs_mmap(void *t, bus_addr_t ba, off_t offset, int prot, int flags)
{

	return bcm2835_bs_mmap(t, ba, 4 * offset, prot, flags);
}

int
bcm2836_bs_map(void *t, bus_addr_t ba, bus_size_t size, int flag,
    bus_space_handle_t *bshp)
{
	const paddr_t pa = bcm2836_bus_to_phys(ba);

	return bus_space_map(&arm_generic_bs_tag, pa, size, flag, bshp);
}

paddr_t
bcm2836_bs_mmap(void *t, bus_addr_t ba, off_t offset, int prot, int flags)
{
	const paddr_t pa = bcm2836_bus_to_phys(ba);

	return bus_space_mmap(&arm_generic_bs_tag, pa, offset, prot, flags);
}

paddr_t
bcm2836_a4x_bs_mmap(void *t, bus_addr_t ba, off_t offset, int prot, int flags)
{

	return bcm2836_bs_mmap(t, ba, 4 * offset, prot, flags);
}

int
bcm2711_bs_map(void *t, bus_addr_t ba, bus_size_t size, int flag,
    bus_space_handle_t *bshp)
{
	const paddr_t pa = bcm2711_bus_to_phys(ba);

	return bus_space_map(&arm_generic_bs_tag, pa, size, flag, bshp);
}

paddr_t
bcm2711_bs_mmap(void *t, bus_addr_t ba, off_t offset, int prot, int flags)
{
	const paddr_t pa = bcm2711_bus_to_phys(ba);

	return bus_space_mmap(&arm_generic_bs_tag, pa, offset, prot, flags);
}

paddr_t
bcm2711_a4x_bs_mmap(void *t, bus_addr_t ba, off_t offset, int prot, int flags)
{

	return bcm2711_bs_mmap(t, ba, 4 * offset, prot, flags);
}

#if defined(SOC_BCM2835)
static const struct pmap_devmap *
bcm2835_platform_devmap(void)
{
	static const struct pmap_devmap devmap[] = {
		DEVMAP_ENTRY(BCM2835_PERIPHERALS_VBASE, BCM2835_PERIPHERALS_BASE,
		    BCM283X_PERIPHERALS_SIZE),	/* 16Mb */

		DEVMAP_ENTRY_END
	};

	return devmap;
}
#endif

#if defined(SOC_BCM2836)
static const struct pmap_devmap *
bcm2836_platform_devmap(void)
{
	static const struct pmap_devmap devmap[] = {
		DEVMAP_ENTRY(BCM2836_PERIPHERALS_VBASE, BCM2836_PERIPHERALS_BASE,
		    BCM283X_PERIPHERALS_SIZE),	/* 16Mb */
		DEVMAP_ENTRY(BCM2836_ARM_LOCAL_VBASE, BCM2836_ARM_LOCAL_BASE,
		    BCM2836_ARM_LOCAL_SIZE),
#if defined(MULTIPROCESSOR) && defined(__aarch64__)
		/* for fdt cpu spin-table */
		DEVMAP_ENTRY(BCM2836_ARM_SMP_VBASE, BCM2836_ARM_SMP_BASE,
		    BCM2836_ARM_SMP_SIZE),
#endif
		DEVMAP_ENTRY_END
	};

	return devmap;
}

static const struct pmap_devmap *
bcm2711_platform_devmap(void)
{
	static const struct pmap_devmap devmap[] = {
		DEVMAP_ENTRY(BCM2711_PERIPHERALS_VBASE, BCM2711_PERIPHERALS_BASE,
		    BCM283X_PERIPHERALS_SIZE),	/* 16Mb */
		DEVMAP_ENTRY(BCM2711_ARM_LOCAL_VBASE, BCM2711_ARM_LOCAL_BASE,
		    BCM2711_ARM_LOCAL_SIZE),
#if defined(MULTIPROCESSOR) && defined(__aarch64__)
		/* for fdt cpu spin-table */
		DEVMAP_ENTRY(BCM2711_ARM_SMP_VBASE, BCM2836_ARM_SMP_BASE,
		    BCM2836_ARM_SMP_SIZE),
#endif
		DEVMAP_ENTRY_END
	};

	return devmap;
}
#endif

/*
 * Macros to translate between physical and virtual for a subset of the
 * kernel address space.  *Not* for general use.
 */

#ifndef RPI_FB_WIDTH
#define RPI_FB_WIDTH	1280
#endif
#ifndef RPI_FB_HEIGHT
#define RPI_FB_HEIGHT	720
#endif

int uart_clk = BCM2835_UART0_CLK;
int core_clk;

static struct {
	struct vcprop_buffer_hdr	vb_hdr;
	struct vcprop_tag_clockrate	vbt_uartclockrate;
	struct vcprop_tag_clockrate	vbt_vpuclockrate;
	struct vcprop_tag end;
} vb_uart __cacheline_aligned = {
	.vb_hdr = {
		.vpb_len = htole32(sizeof(vb_uart)),
		.vpb_rcode = htole32(VCPROP_PROCESS_REQUEST),
	},
	.vbt_uartclockrate = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_CLOCKRATE),
			.vpt_len =
			    htole32(VCPROPTAG_LEN(vb_uart.vbt_uartclockrate)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
		.id = htole32(VCPROP_CLK_UART)
	},
	.vbt_vpuclockrate = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_CLOCKRATE),
			.vpt_len = htole32(VCPROPTAG_LEN(vb_uart.vbt_vpuclockrate)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
		.id = htole32(VCPROP_CLK_CORE)
	},
	.end = {
		.vpt_tag = htole32(VCPROPTAG_NULL)
	}
};

static struct {
	struct vcprop_buffer_hdr	vb_hdr;
	struct vcprop_tag_fwrev		vbt_fwrev;
	struct vcprop_tag_boardmodel	vbt_boardmodel;
	struct vcprop_tag_boardrev	vbt_boardrev;
	struct vcprop_tag_macaddr	vbt_macaddr;
	struct vcprop_tag_memory	vbt_memory;
	struct vcprop_tag_boardserial	vbt_serial;
	struct vcprop_tag_dmachan	vbt_dmachan;
	struct vcprop_tag_cmdline	vbt_cmdline;
	struct vcprop_tag_clockrate	vbt_emmcclockrate;
	struct vcprop_tag_clockrate	vbt_armclockrate;
	struct vcprop_tag_clockrate	vbt_vpuclockrate;
	struct vcprop_tag_clockrate	vbt_emmc2clockrate;
	struct vcprop_tag end;
} vb __cacheline_aligned = {
	.vb_hdr = {
		.vpb_len = htole32(sizeof(vb)),
		.vpb_rcode = htole32(VCPROP_PROCESS_REQUEST),
	},
	.vbt_fwrev = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_FIRMWAREREV),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_fwrev)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
	},
	.vbt_boardmodel = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_BOARDMODEL),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_boardmodel)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
	},
	.vbt_boardrev = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_BOARDREVISION),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_boardrev)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
	},
	.vbt_macaddr = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_MACADDRESS),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_macaddr)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
	},
	.vbt_memory = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_ARMMEMORY),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_memory)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
	},
	.vbt_serial = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_BOARDSERIAL),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_serial)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
	},
	.vbt_dmachan = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_DMACHAN),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_dmachan)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
	},
	.vbt_cmdline = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_CMDLINE),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_cmdline)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
	},
	.vbt_emmcclockrate = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_CLOCKRATE),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_emmcclockrate)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
		.id = htole32(VCPROP_CLK_EMMC)
	},
	.vbt_armclockrate = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_CLOCKRATE),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_armclockrate)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
		.id = htole32(VCPROP_CLK_ARM)
	},
	.vbt_vpuclockrate = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_CLOCKRATE),
			.vpt_len = htole32(VCPROPTAG_LEN(vb.vbt_vpuclockrate)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
		.id = htole32(VCPROP_CLK_CORE)
	},
	.vbt_emmc2clockrate = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_CLOCKRATE),
			.vpt_len =
			    htole32(VCPROPTAG_LEN(vb.vbt_emmc2clockrate)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST)
		},
		.id = htole32(VCPROP_CLK_EMMC2)
	},
	.end = {
		.vpt_tag = htole32(VCPROPTAG_NULL)
	}
};

#if NGENFB > 0
static struct {
	struct vcprop_buffer_hdr	vb_hdr;
	struct vcprop_tag_edidblock	vbt_edid;
	struct vcprop_tag end;
} vb_edid __cacheline_aligned = {
	.vb_hdr = {
		.vpb_len = htole32(sizeof(vb_edid)),
		.vpb_rcode = htole32(VCPROP_PROCESS_REQUEST),
	},
	.vbt_edid = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_EDID_BLOCK),
			.vpt_len = htole32(VCPROPTAG_LEN(vb_edid.vbt_edid)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST),
		},
		.blockno = htole32(0),
	},
	.end = {
		.vpt_tag = htole32(VCPROPTAG_NULL)
	}
};

static struct {
	struct vcprop_buffer_hdr	vb_hdr;
	struct vcprop_tag_fbres		vbt_res;
	struct vcprop_tag_fbres		vbt_vres;
	struct vcprop_tag_fbdepth	vbt_depth;
	struct vcprop_tag_fbalpha	vbt_alpha;
	struct vcprop_tag_allocbuf	vbt_allocbuf;
	struct vcprop_tag_blankscreen	vbt_blank;
	struct vcprop_tag_fbpitch	vbt_pitch;
	struct vcprop_tag end;
} vb_setfb __cacheline_aligned = {
	.vb_hdr = {
		.vpb_len = htole32(sizeof(vb_setfb)),
		.vpb_rcode = htole32(VCPROP_PROCESS_REQUEST),
	},
	.vbt_res = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_SET_FB_RES),
			.vpt_len = htole32(VCPROPTAG_LEN(vb_setfb.vbt_res)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST),
		},
		.width = htole32(0),
		.height = htole32(0),
	},
	.vbt_vres = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_SET_FB_VRES),
			.vpt_len = htole32(VCPROPTAG_LEN(vb_setfb.vbt_vres)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST),
		},
		.width = htole32(0),
		.height = htole32(0),
	},
	.vbt_depth = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_SET_FB_DEPTH),
			.vpt_len = htole32(VCPROPTAG_LEN(vb_setfb.vbt_depth)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST),
		},
		.bpp = htole32(32),
	},
	.vbt_alpha = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_SET_FB_ALPHA_MODE),
			.vpt_len = htole32(VCPROPTAG_LEN(vb_setfb.vbt_alpha)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST),
		},
		.state = htole32(VCPROP_ALPHA_IGNORED),
	},
	.vbt_allocbuf = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_ALLOCATE_BUFFER),
			.vpt_len =
			    htole32(VCPROPTAG_LEN(vb_setfb.vbt_allocbuf)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST),
		},
		.address = htole32(PAGE_SIZE),	/* alignment */
	},
	.vbt_blank = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_BLANK_SCREEN),
			.vpt_len = htole32(VCPROPTAG_LEN(vb_setfb.vbt_blank)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST),
		},
		.state = htole32(VCPROP_BLANK_OFF),
	},
	.vbt_pitch = {
		.tag = {
			.vpt_tag = htole32(VCPROPTAG_GET_FB_PITCH),
			.vpt_len = htole32(VCPROPTAG_LEN(vb_setfb.vbt_pitch)),
			.vpt_rcode = htole32(VCPROPTAG_REQUEST),
		},
	},
	.end = {
		.vpt_tag = htole32(VCPROPTAG_NULL),
	},
};

#endif

static int rpi_video_on = WSDISPLAYIO_VIDEO_ON;

#if defined(RPI_HWCURSOR)
#define CURSOR_BITMAP_SIZE	(64 * 8)
#define CURSOR_ARGB_SIZE	(64 * 64 * 4)
static uint32_t hcursor = 0;
static bus_addr_t pcursor = 0;
static uint32_t *cmem = NULL;
static int cursor_x = 0, cursor_y = 0, hot_x = 0, hot_y = 0, cursor_on = 0;
static uint32_t cursor_cmap[4];
static uint8_t cursor_mask[8 * 64], cursor_bitmap[8 * 64];
#endif

u_int
bcm283x_clk_get_rate_uart(void)
{

	if (vcprop_tag_success_p(&vb_uart.vbt_uartclockrate.tag))
		return le32toh(vb_uart.vbt_uartclockrate.rate);
	return 0;
}

u_int
bcm283x_clk_get_rate_vpu(void)
{

	if (vcprop_tag_success_p(&vb.vbt_vpuclockrate.tag) &&
	    vb.vbt_vpuclockrate.rate != 0) {
		return le32toh(vb.vbt_vpuclockrate.rate);
	}
	return 0;
}

u_int
bcm283x_clk_get_rate_emmc(void)
{

	if (vcprop_tag_success_p(&vb.vbt_emmcclockrate.tag) &&
	    vb.vbt_emmcclockrate.rate != 0) {
		return le32toh(vb.vbt_emmcclockrate.rate);
	}
	return 0;
}

u_int
bcm283x_clk_get_rate_emmc2(void)
{

	if (vcprop_tag_success_p(&vb.vbt_emmc2clockrate.tag) &&
	    vb.vbt_emmc2clockrate.rate != 0) {
		return le32toh(vb.vbt_emmc2clockrate.rate);
	}
	return 0;
}



static void
bcm283x_uartinit(bus_space_tag_t iot, bus_space_handle_t ioh)
{
	uint32_t res;

	bcm2835_mbox_write(iot, ioh, BCMMBOX_CHANARM2VC,
	    KERN_VTOPHYS((vaddr_t)&vb_uart));

	bcm2835_mbox_read(iot, ioh, BCMMBOX_CHANARM2VC, &res);

	/*
	 * RPI4 has Cortex A72 processors which do speculation, so
	 * we need to invalidate the cache for an updates done by
	 * the firmware
	 */
	cpu_dcache_inv_range((vaddr_t)&vb_uart, sizeof(vb_uart));

	if (vcprop_tag_success_p(&vb_uart.vbt_uartclockrate.tag))
		uart_clk = le32toh(vb_uart.vbt_uartclockrate.rate);
	if (vcprop_tag_success_p(&vb_uart.vbt_vpuclockrate.tag))
		core_clk = le32toh(vb_uart.vbt_vpuclockrate.rate);
}

#if defined(SOC_BCM2835)
static void
bcm2835_uartinit(void)
{
	const paddr_t pa = BCM2835_PERIPHERALS_BUS_TO_PHYS(BCM2835_ARMMBOX_BASE);
	const bus_space_tag_t iot = &bcm2835_bs_tag;
	const bus_space_handle_t ioh = BCM2835_IOPHYSTOVIRT(pa);

	bcm283x_uartinit(iot, ioh);
}
#endif

#if defined(SOC_BCM2836)
static void
bcm2836_uartinit(void)
{
	const paddr_t pa = BCM2836_PERIPHERALS_BUS_TO_PHYS(BCM2835_ARMMBOX_BASE);
	const bus_space_tag_t iot = &bcm2836_bs_tag;
	const bus_space_handle_t ioh = BCM2835_IOPHYSTOVIRT(pa);

	bcm283x_uartinit(iot, ioh);
}

static void
bcm2711_uartinit(void)
{
	const paddr_t pa = BCM2711_PERIPHERALS_BUS_TO_PHYS(BCM2835_ARMMBOX_BASE);
	const bus_space_tag_t iot = &bcm2711_bs_tag;
	const bus_space_handle_t ioh = BCM2711_IOPHYSTOVIRT(pa);

	bcm283x_uartinit(iot, ioh);
}
#endif

#define	BCM283x_MINIMUM_SPLIT	(128U * 1024 * 1024)

static size_t bcm283x_memorysize;

static void
bcm283x_bootparams(bus_space_tag_t iot, bus_space_handle_t ioh)
{
	uint32_t res;

	bcm2835_mbox_write(iot, ioh, BCMMBOX_CHANPM, (
#if (NSDHC > 0)
	    (1 << VCPM_POWER_SDCARD) |
#endif
#if (NPLCOM > 0)
	    (1 << VCPM_POWER_UART0) |
#endif
#if (NBCMDWCTWO > 0)
	    (1 << VCPM_POWER_USB) |
#endif
#if (NBSCIIC > 0)
	    (1 << VCPM_POWER_I2C0) | (1 << VCPM_POWER_I2C1) |
	/*  (1 << VCPM_POWER_I2C2) | */
#endif
#if (NBCMSPI > 0)
	    (1 << VCPM_POWER_SPI) |
#endif
	    0) << 4);

	bcm2835_mbox_write(iot, ioh, BCMMBOX_CHANARM2VC,
	    KERN_VTOPHYS((vaddr_t)&vb));

	bcm2835_mbox_read(iot, ioh, BCMMBOX_CHANARM2VC, &res);

	/*
	 * RPI4 has Cortex A72 processors which do speculation, so
	 * we need to invalidate the cache for an updates done by
	 * the firmware
	 */
	cpu_dcache_inv_range((vaddr_t)&vb, sizeof(vb));

	if (!vcprop_buffer_success_p(&vb.vb_hdr)) {
		bootconfig.dramblocks = 1;
		bootconfig.dram[0].address = 0x0;
		bootconfig.dram[0].pages = atop(BCM283x_MINIMUM_SPLIT);
		return;
	}

	struct vcprop_tag_memory *vptp_mem = &vb.vbt_memory;
	if (vcprop_tag_success_p(&vptp_mem->tag)) {
		size_t n = vcprop_tag_resplen(&vptp_mem->tag) /
		    sizeof(struct vcprop_memory);

		bcm283x_memorysize = 0;
		bootconfig.dramblocks = 0;

		for (int i = 0; i < n && i < DRAM_BLOCKS; i++) {
			bootconfig.dram[i].address =
			    le32toh(vptp_mem->mem[i].base);
			bootconfig.dram[i].pages =
			    atop(le32toh(vptp_mem->mem[i].size));
			bootconfig.dramblocks++;

			bcm283x_memorysize += le32toh(vptp_mem->mem[i].size);
		}
	}

	if (vcprop_tag_success_p(&vb.vbt_armclockrate.tag))
		curcpu()->ci_data.cpu_cc_freq =
		    le32toh(vb.vbt_armclockrate.rate);

#ifdef VERBOSE_INIT_ARM
	if (vcprop_tag_success_p(&vb.vbt_memory.tag))
		printf("%s: memory size  %zu\n", __func__,
		    bcm283x_memorysize);
	if (vcprop_tag_success_p(&vb.vbt_armclockrate.tag))
		printf("%s: arm clock    %d\n", __func__,
		    le32toh(vb.vbt_armclockrate.rate));
	if (vcprop_tag_success_p(&vb.vbt_vpuclockrate.tag))
		printf("%s: vpu clock    %d\n", __func__,
		    le32toh(vb.vbt_vpuclockrate.rate));
	if (vcprop_tag_success_p(&vb.vbt_emmcclockrate.tag))
		printf("%s: emmc clock   %d\n", __func__,
		    le32toh(vb.vbt_emmcclockrate.rate));
	if (vcprop_tag_success_p(&vb.vbt_emmc2clockrate.tag))
		printf("%s: emmc2 clock  %d\n", __func__,
		    le32toh(vb.vbt_emmcclockrate.rate));
	if (vcprop_tag_success_p(&vb.vbt_fwrev.tag))
		printf("%s: firmware rev %x\n", __func__,
		    le32toh(vb.vbt_fwrev.rev));
	if (vcprop_tag_success_p(&vb.vbt_boardmodel.tag))
		printf("%s: board model  %x\n", __func__,
		    le32toh(vb.vbt_boardmodel.model));
	if (vcprop_tag_success_p(&vb.vbt_macaddr.tag))
		printf("%s: mac-address  %" PRIx64 "\n", __func__,
		    le64toh(vb.vbt_macaddr.addr));
	if (vcprop_tag_success_p(&vb.vbt_boardrev.tag))
		printf("%s: board rev    %x\n", __func__,
		    le32toh(vb.vbt_boardrev.rev));
	if (vcprop_tag_success_p(&vb.vbt_serial.tag))
		printf("%s: board serial %" PRIx64 "\n", __func__,
		    le64toh(vb.vbt_serial.sn));
	if (vcprop_tag_success_p(&vb.vbt_dmachan.tag))
		printf("%s: DMA channel mask 0x%08x\n", __func__,
		    le32toh(vb.vbt_dmachan.mask));

	if (vcprop_tag_success_p(&vb.vbt_cmdline.tag))
		printf("%s: cmdline      %s\n", __func__,
		    vb.vbt_cmdline.cmdline);
#endif
}

#if defined(SOC_BCM2835)
static void
bcm2835_bootparams(void)
{
	const paddr_t pa = BCM2835_PERIPHERALS_BUS_TO_PHYS(BCM2835_ARMMBOX_BASE);
	const bus_space_tag_t iot = &bcm2835_bs_tag;
	const bus_space_handle_t ioh = BCM2835_IOPHYSTOVIRT(pa);

	bcm283x_bootparams(iot, ioh);
}
#endif

#if defined(SOC_BCM2836)
static void
bcm2836_bootparams(void)
{
	const paddr_t pa = BCM2836_PERIPHERALS_BUS_TO_PHYS(BCM2835_ARMMBOX_BASE);
	const bus_space_tag_t iot = &bcm2836_bs_tag;
	const bus_space_handle_t ioh = BCM2835_IOPHYSTOVIRT(pa);

	bcm283x_bootparams(iot, ioh);
}

static void
bcm2711_bootparams(void)
{
	const paddr_t pa = BCM2711_PERIPHERALS_BUS_TO_PHYS(BCM2835_ARMMBOX_BASE);
	const bus_space_tag_t iot = &bcm2711_bs_tag;
	const bus_space_handle_t ioh = BCM2711_IOPHYSTOVIRT(pa);

	bcm283x_bootparams(iot, ioh);
}

#if defined(MULTIPROCESSOR)
static int
cpu_enable_bcm2836(int phandle)
{
	bus_space_tag_t iot = &bcm2836_bs_tag;
	bus_space_handle_t ioh = BCM2836_ARM_LOCAL_VBASE;
	uint64_t mpidr;

	fdtbus_get_reg64(phandle, 0, &mpidr, NULL);

	const u_int cpuno = __SHIFTOUT(mpidr, MPIDR_AFF0);

	bus_space_write_4(iot, ioh, BCM2836_LOCAL_MAILBOX3_SETN(cpuno),
	    KERN_VTOPHYS((vaddr_t)cpu_mpstart));

	return 0;
}
ARM_CPU_METHOD(bcm2836, "brcm,bcm2836-smp", cpu_enable_bcm2836);
#endif

#endif	/* SOC_BCM2836 */

#if NGENFB > 0
static bool
rpi_fb_parse_mode(const char *s, uint32_t *pwidth, uint32_t *pheight)
{
	char *x;

	if (strncmp(s, "disable", 7) == 0)
		return false;

	x = strchr(s, 'x');
	if (x) {
		*pwidth = strtoul(s, NULL, 10);
		*pheight = strtoul(x + 1, NULL, 10);
	}

	return true;
}

#define RPI_EDIDSIZE 1024

static bool
rpi_fb_get_edid_mode(uint32_t *pwidth, uint32_t *pheight)
{
	struct edid_info ei;
	uint32_t res;
	int error;

	error = bcmmbox_request(BCMMBOX_CHANARM2VC, &vb_edid,
	    sizeof(vb_edid), &res);
	if (error) {
		printf("%s: mbox request failed (%d)\n", __func__, error);
		return false;
	}

	if (!vcprop_buffer_success_p(&vb_edid.vb_hdr) ||
	    !vcprop_tag_success_p(&vb_edid.vbt_edid.tag) ||
	    vb_edid.vbt_edid.status != 0)
		return false;

	uint8_t *edid_data = kmem_alloc(RPI_EDIDSIZE, KM_SLEEP);

	memset(edid_data, 0, RPI_EDIDSIZE);
	memcpy(edid_data, vb_edid.vbt_edid.data,
	    sizeof(vb_edid.vbt_edid.data));
	edid_parse(edid_data, &ei);
#ifdef VERBOSE_INIT_ARM
	edid_print(&ei);
#endif

	if (ei.edid_preferred_mode) {
		*pwidth = ei.edid_preferred_mode->hdisplay;
		*pheight = ei.edid_preferred_mode->vdisplay;
	}

	kmem_free(edid_data, RPI_EDIDSIZE);

	return true;
}

/*
 * Initialize framebuffer console.
 *
 * Some notes about boot parameters:
 *  - If "fb=disable" is present, ignore framebuffer completely.
 *  - If "fb=<width>x<height> is present, use the specified mode.
 *  - If "console=fb" is present, attach framebuffer to console.
 */
static bool
rpi_fb_init(prop_dictionary_t dict, void *aux)
{
	uint32_t width = 0, height = 0;
	uint32_t res;
	char *ptr;
	int integer;
	int error;
	bool is_bgr = true;

	if (get_bootconf_option(boot_args, "fb",
			      BOOTOPT_TYPE_STRING, &ptr)) {
		if (rpi_fb_parse_mode(ptr, &width, &height) == false)
			return false;
	}
	if (width == 0 || height == 0) {
		rpi_fb_get_edid_mode(&width, &height);
	}
	if (width == 0 || height == 0) {
		width = RPI_FB_WIDTH;
		height = RPI_FB_HEIGHT;
	}

	vb_setfb.vbt_res.width = htole32(width);
	vb_setfb.vbt_res.height = htole32(height);
	vb_setfb.vbt_vres.width = htole32(width);
	vb_setfb.vbt_vres.height = htole32(height);
	error = bcmmbox_request(BCMMBOX_CHANARM2VC, &vb_setfb,
	    sizeof(vb_setfb), &res);
	if (error) {
		printf("%s: mbox request failed (%d)\n", __func__, error);
		return false;
	}

	if (!vcprop_buffer_success_p(&vb_setfb.vb_hdr) ||
	    !vcprop_tag_success_p(&vb_setfb.vbt_res.tag) ||
	    !vcprop_tag_success_p(&vb_setfb.vbt_vres.tag) ||
	    !vcprop_tag_success_p(&vb_setfb.vbt_depth.tag) ||
	    !vcprop_tag_success_p(&vb_setfb.vbt_allocbuf.tag) ||
	    !vcprop_tag_success_p(&vb_setfb.vbt_blank.tag) ||
	    !vcprop_tag_success_p(&vb_setfb.vbt_pitch.tag)) {
		printf("%s: prop tag failed\n", __func__);
		return false;
	}

#ifdef VERBOSE_INIT_ARM
	printf("%s: addr = 0x%x size = %d\n", __func__,
	    le32toh(vb_setfb.vbt_allocbuf.address),
	    le32toh(vb_setfb.vbt_allocbuf.size));
	printf("%s: depth = %d\n", __func__, le32toh(vb_setfb.vbt_depth.bpp));
	printf("%s: pitch = %d\n", __func__,
	    le32toh(vb_setfb.vbt_pitch.linebytes));
	printf("%s: width = %d height = %d\n", __func__,
	    le32toh(vb_setfb.vbt_res.width), le32toh(vb_setfb.vbt_res.height));
	printf("%s: vwidth = %d vheight = %d\n", __func__,
	    le32toh(vb_setfb.vbt_vres.width),
	    le32toh(vb_setfb.vbt_vres.height));
#endif

	if (vb_setfb.vbt_allocbuf.address == 0 ||
	    vb_setfb.vbt_allocbuf.size == 0 ||
	    vb_setfb.vbt_res.width == 0 ||
	    vb_setfb.vbt_res.height == 0 ||
	    vb_setfb.vbt_vres.width == 0 ||
	    vb_setfb.vbt_vres.height == 0 ||
	    vb_setfb.vbt_pitch.linebytes == 0) {
		printf("%s: failed to set mode %ux%u\n", __func__,
		    width, height);
		return false;
	}

	prop_dictionary_set_uint32(dict, "width",
	    le32toh(vb_setfb.vbt_res.width));
	prop_dictionary_set_uint32(dict, "height",
	    le32toh(vb_setfb.vbt_res.height));
	prop_dictionary_set_uint8(dict, "depth",
	    le32toh(vb_setfb.vbt_depth.bpp));
	prop_dictionary_set_uint16(dict, "linebytes",
	    le32toh(vb_setfb.vbt_pitch.linebytes));
	prop_dictionary_set_uint32(dict, "address",
	    le32toh(vb_setfb.vbt_allocbuf.address));

	/*
	 * Old firmware uses BGR. New firmware uses RGB. The get and set
	 * pixel order mailbox properties don't seem to work. The firmware
	 * adds a kernel cmdline option bcm2708_fb.fbswap=<0|1>, so use it
	 * to determine pixel order. 0 means BGR, 1 means RGB.
	 *
	 * See https://github.com/raspberrypi/linux/issues/514
	 */
	if (get_bootconf_option(boot_args, "bcm2708_fb.fbswap",
				BOOTOPT_TYPE_INT, &integer)) {
		is_bgr = integer == 0;
	}
	prop_dictionary_set_bool(dict, "is_bgr", is_bgr);

	/* if "genfb.type=<n>" is passed in cmdline, override wsdisplay type */
	if (get_bootconf_option(boot_args, "genfb.type",
				BOOTOPT_TYPE_INT, &integer)) {
		prop_dictionary_set_uint32(dict, "wsdisplay_type", integer);
	}

#if defined(RPI_HWCURSOR)
	struct fdt_attach_args *faa = aux;
	bus_space_handle_t hc;

	hcursor = rpi_alloc_mem(CURSOR_ARGB_SIZE, PAGE_SIZE,
	    MEM_FLAG_L1_NONALLOCATING | MEM_FLAG_HINT_PERMALOCK);
	pcursor = rpi_lock_mem(hcursor);
#ifdef RPI_IOCTL_DEBUG
	printf("hcursor: %08x\n", hcursor);
	printf("pcursor: %08x\n", (uint32_t)pcursor);
	printf("fb: %08x\n", (uint32_t)vb_setfb.vbt_allocbuf.address);
#endif
	if (bus_space_map(faa->faa_bst, pcursor, CURSOR_ARGB_SIZE,
	    BUS_SPACE_MAP_LINEAR|BUS_SPACE_MAP_PREFETCHABLE, &hc) != 0) {
		printf("couldn't map cursor memory\n");
	} else {
		int i, j, k;

		cmem = bus_space_vaddr(faa->faa_bst, hc);
		k = 0;
		for (j = 0; j < 64; j++) {
			for (i = 0; i < 64; i++) {
				cmem[i + k] =
				 ((i & 8) ^ (j & 8)) ? 0xa0ff0000 : 0xa000ff00;
			}
			k += 64;
		}
		cpu_dcache_wb_range((vaddr_t)cmem, CURSOR_ARGB_SIZE);
		rpi_fb_initcursor(pcursor, 0, 0);
#ifdef RPI_IOCTL_DEBUG
		rpi_fb_movecursor(600, 400, 1);
#else
		rpi_fb_movecursor(cursor_x, cursor_y, cursor_on);
#endif
	}
#endif

	return true;
}


#if defined(RPI_HWCURSOR)
static int
rpi_fb_do_cursor(struct wsdisplay_cursor *cur)
{
	int pos = 0;
	int shape = 0;

	if (cur->which & WSDISPLAY_CURSOR_DOCUR) {
		if (cursor_on != cur->enable) {
			cursor_on = cur->enable;
			pos = 1;
		}
	}
	if (cur->which & WSDISPLAY_CURSOR_DOHOT) {

		hot_x = cur->hot.x;
		hot_y = cur->hot.y;
		pos = 1;
		shape = 1;
	}
	if (cur->which & WSDISPLAY_CURSOR_DOPOS) {

		cursor_x = cur->pos.x;
		cursor_y = cur->pos.y;
		pos = 1;
	}
	if (cur->which & WSDISPLAY_CURSOR_DOCMAP) {
		int i;
		uint32_t val;

		for (i = 0; i < uimin(cur->cmap.count, 3); i++) {
			val = (cur->cmap.red[i] << 16 ) |
			      (cur->cmap.green[i] << 8) |
			      (cur->cmap.blue[i] ) |
			      0xff000000;
			cursor_cmap[i + cur->cmap.index + 2] = val;
		}
		shape = 1;
	}
	if (cur->which & WSDISPLAY_CURSOR_DOSHAPE) {
		int err;

		err = copyin(cur->mask, cursor_mask, CURSOR_BITMAP_SIZE);
		err += copyin(cur->image, cursor_bitmap, CURSOR_BITMAP_SIZE);
		if (err != 0)
			return EFAULT;
		shape = 1;
	}
	if (shape) {
		int i, j, idx;
		uint8_t mask;

		for (i = 0; i < CURSOR_BITMAP_SIZE; i++) {
			mask = 0x01;
			for (j = 0; j < 8; j++) {
				idx = ((cursor_mask[i] & mask) ? 2 : 0) |
				    ((cursor_bitmap[i] & mask) ? 1 : 0);
				cmem[i * 8 + j] = cursor_cmap[idx];
				mask = mask << 1;
			}
		}
		/* just in case */
		cpu_dcache_wb_range((vaddr_t)cmem, CURSOR_ARGB_SIZE);
		rpi_fb_initcursor(pcursor, hot_x, hot_y);
	}
	if (pos) {
		rpi_fb_movecursor(cursor_x, cursor_y, cursor_on);
	}
	return 0;
}
#endif

static int
rpi_ioctl(void *v, void *vs, u_long cmd, void *data, int flag, lwp_t *l)
{

	switch (cmd) {
	case WSDISPLAYIO_SVIDEO:
		{
			int d = *(int *)data;
			if (d == rpi_video_on)
				return 0;
			rpi_video_on = d;
			rpi_fb_set_video(d);
#if defined(RPI_HWCURSOR)
			rpi_fb_movecursor(cursor_x, cursor_y,
			                  d ? cursor_on : 0);
#endif
		}
		return 0;
	case WSDISPLAYIO_GVIDEO:
		*(int *)data = rpi_video_on;
		return 0;
#if defined(RPI_HWCURSOR)
	case WSDISPLAYIO_GCURPOS:
		{
			struct wsdisplay_curpos *cp = (void *)data;

			cp->x = cursor_x;
			cp->y = cursor_y;
		}
		return 0;
	case WSDISPLAYIO_SCURPOS:
		{
			struct wsdisplay_curpos *cp = (void *)data;

			cursor_x = cp->x;
			cursor_y = cp->y;
			rpi_fb_movecursor(cursor_x, cursor_y, cursor_on);
		}
		return 0;
	case WSDISPLAYIO_GCURMAX:
		{
			struct wsdisplay_curpos *cp = (void *)data;

			cp->x = 64;
			cp->y = 64;
		}
		return 0;
	case WSDISPLAYIO_SCURSOR:
		{
			struct wsdisplay_cursor *cursor = (void *)data;

			return rpi_fb_do_cursor(cursor);
		}
#endif
	default:
		return EPASSTHROUGH;
	}
}

#endif

SYSCTL_SETUP(sysctl_machdep_rpi, "sysctl machdep subtree setup (rpi)")
{
	sysctl_createv(clog, 0, NULL, NULL,
	    CTLFLAG_PERMANENT, CTLTYPE_NODE, "machdep", NULL,
	    NULL, 0, NULL, 0, CTL_MACHDEP, CTL_EOL);

	sysctl_createv(clog, 0, NULL, NULL,
	    CTLFLAG_PERMANENT|CTLFLAG_READONLY,
	    CTLTYPE_INT, "firmware_revision", NULL, NULL, 0,
	    &vb.vbt_fwrev.rev, 0, CTL_MACHDEP, CTL_CREATE, CTL_EOL);

	sysctl_createv(clog, 0, NULL, NULL,
	    CTLFLAG_PERMANENT|CTLFLAG_READONLY,
	    CTLTYPE_INT, "board_model", NULL, NULL, 0,
	    &vb.vbt_boardmodel.model, 0, CTL_MACHDEP, CTL_CREATE, CTL_EOL);

	sysctl_createv(clog, 0, NULL, NULL,
	    CTLFLAG_PERMANENT|CTLFLAG_READONLY,
	    CTLTYPE_INT, "board_revision", NULL, NULL, 0,
	    &vb.vbt_boardrev.rev, 0, CTL_MACHDEP, CTL_CREATE, CTL_EOL);

	sysctl_createv(clog, 0, NULL, NULL,
	    CTLFLAG_PERMANENT|CTLFLAG_READONLY|CTLFLAG_HEX|CTLFLAG_PRIVATE,
	    CTLTYPE_QUAD, "serial", NULL, NULL, 0,
	    &vb.vbt_serial.sn, 0, CTL_MACHDEP, CTL_CREATE, CTL_EOL);
}

#if defined(SOC_BCM2835)
static void
bcm2835_platform_bootstrap(void)
{

	bcm2835_bs_tag = arm_generic_bs_tag;
	bcm2835_a4x_bs_tag = arm_generic_a4x_bs_tag;

	bcm2835_bs_tag.bs_map = bcm2835_bs_map;
	bcm2835_bs_tag.bs_mmap = bcm2835_bs_mmap;
	bcm2835_a4x_bs_tag.bs_map = bcm2835_bs_map;
	bcm2835_a4x_bs_tag.bs_mmap = bcm2835_a4x_bs_mmap;

	fdtbus_set_decoderegprop(false);

	bcm2835_uartinit();

	bcm2835_bootparams();
}
#endif

#if defined(SOC_BCM2836)
static void
bcm2836_platform_bootstrap(void)
{

	bcm2836_bs_tag = arm_generic_bs_tag;
	bcm2836_a4x_bs_tag = arm_generic_a4x_bs_tag;

	bcm2836_bs_tag.bs_map = bcm2836_bs_map;
	bcm2836_bs_tag.bs_mmap = bcm2836_bs_mmap;
	bcm2836_a4x_bs_tag.bs_map = bcm2836_bs_map;
	bcm2836_a4x_bs_tag.bs_mmap = bcm2836_a4x_bs_mmap;

	fdtbus_set_decoderegprop(false);

	bcm2836_uartinit();

	bcm2836_bootparams();

#ifdef MULTIPROCESSOR
	arm_cpu_max = RPI_CPU_MAX;
	arm_fdt_cpu_bootstrap();
#endif
}

static void
bcm2711_platform_bootstrap(void)
{

	bcm2711_bs_tag = arm_generic_bs_tag;
	bcm2711_a4x_bs_tag = arm_generic_a4x_bs_tag;

	bcm2711_bs_tag.bs_map = bcm2711_bs_map;
	bcm2711_bs_tag.bs_mmap = bcm2711_bs_mmap;
	bcm2711_a4x_bs_tag.bs_map = bcm2711_bs_map;
	bcm2711_a4x_bs_tag.bs_mmap = bcm2711_a4x_bs_mmap;

	fdtbus_set_decoderegprop(false);

	bcm2711_uartinit();

	bcm2711_bootparams();

#ifdef MULTIPROCESSOR
	arm_cpu_max = RPI_CPU_MAX;
	arm_fdt_cpu_bootstrap();
#endif
}
#endif

#if defined(SOC_BCM2835)
static void
bcm2835_platform_init_attach_args(struct fdt_attach_args *faa)
{

	faa->faa_bst = &bcm2835_bs_tag;
}
#endif

#if defined(SOC_BCM2836)
static void
bcm2836_platform_init_attach_args(struct fdt_attach_args *faa)
{

	faa->faa_bst = &bcm2836_bs_tag;
}

static void
bcm2711_platform_init_attach_args(struct fdt_attach_args *faa)
{

	faa->faa_bst = &bcm2711_bs_tag;
}
#endif


static void __noasan
bcm283x_platform_early_putchar(vaddr_t va, paddr_t pa, char c)
{
	volatile uint32_t *uartaddr =
	    cpu_earlydevice_va_p() ?
		(volatile uint32_t *)va :
		(volatile uint32_t *)pa;

	while ((le32toh(uartaddr[PL01XCOM_FR / 4]) & PL01X_FR_TXFF) != 0)
		continue;

	uartaddr[PL01XCOM_DR / 4] = htole32(c);

	while ((le32toh(uartaddr[PL01XCOM_FR / 4]) & PL01X_FR_TXFE) == 0)
		continue;
}

static void __noasan
bcm283x_aux_platform_early_putchar(vaddr_t va, paddr_t pa, char c)
{
	volatile uint32_t *uartaddr =
	    cpu_earlydevice_va_p() ?
		(volatile uint32_t *)va :
		(volatile uint32_t *)pa;

	while ((le32toh(uartaddr[com_lsr]) & LSR_TXRDY) == 0)
		continue;

	uartaddr[com_data] = htole32(c);
}

void __noasan
bcm2835_platform_early_putchar(char c)
{
	paddr_t pa = BCM2835_PERIPHERALS_BUS_TO_PHYS(BCM2835_UART0_BASE);
	vaddr_t va = BCM2835_IOPHYSTOVIRT(pa);

	bcm283x_platform_early_putchar(va, pa, c);
}

void __noasan
bcm2835_aux_platform_early_putchar(char c)
{
	paddr_t pa = BCM2835_PERIPHERALS_BUS_TO_PHYS(BCM2835_AUX_UART_BASE);
	vaddr_t va = BCM2835_IOPHYSTOVIRT(pa);

	bcm283x_aux_platform_early_putchar(va, pa, c);
}

void __noasan
bcm2836_platform_early_putchar(char c)
{
	paddr_t pa = BCM2836_PERIPHERALS_BUS_TO_PHYS(BCM2835_UART0_BASE);
	vaddr_t va = BCM2835_IOPHYSTOVIRT(pa);

	bcm283x_platform_early_putchar(va, pa, c);
}

void __noasan
bcm2837_platform_early_putchar(char c)
{
	paddr_t pa = BCM2836_PERIPHERALS_BUS_TO_PHYS(BCM2835_AUX_UART_BASE);
	vaddr_t va = BCM2835_IOPHYSTOVIRT(pa);

	bcm283x_aux_platform_early_putchar(va, pa, c);
}

void __noasan
bcm2711_platform_early_putchar(char c)
{
	paddr_t pa = BCM2711_PERIPHERALS_BUS_TO_PHYS(BCM2835_AUX_UART_BASE);
	vaddr_t va = BCM2711_IOPHYSTOVIRT(pa);

	bcm283x_aux_platform_early_putchar(va, pa, c);
}

#define	BCM283x_REF_FREQ	19200000

static void
bcm283x_platform_device_register(device_t dev, void *aux)
{
	prop_dictionary_t dict = device_properties(dev);

	if (device_is_a(dev, "bcmdmac") &&
	    vcprop_tag_success_p(&vb.vbt_dmachan.tag)) {
		prop_dictionary_set_uint32(dict,
		    "chanmask", le32toh(vb.vbt_dmachan.mask));
	}
#if NSDHC > 0
	if (booted_device == NULL &&
	    device_is_a(dev, "ld") &&
	    device_is_a(device_parent(dev), "sdmmc")) {
		booted_partition = 0;
		booted_device = dev;
	}
#endif
	if ((device_is_a(dev, "usmsc") ||
	     device_is_a(dev, "mue") ||
	     device_is_a(dev, "genet")) &&
	    vcprop_tag_success_p(&vb.vbt_macaddr.tag)) {
		const uint64_t addr = le64toh(vb.vbt_macaddr.addr);
		const uint8_t enaddr[ETHER_ADDR_LEN] = {
		     (addr >> 0)  & 0xff, (addr >> 8)  & 0xff,
		     (addr >> 16) & 0xff, (addr >> 24) & 0xff,
		     (addr >> 32) & 0xff, (addr >> 40) & 0xff
		};
		device_setprop_data(dev, "mac-address", enaddr,
		    sizeof(enaddr));
	}

#if NGENFB > 0
	if (device_is_a(dev, "genfb")) {
		char *ptr;

		bcmgenfb_set_console_dev(dev);
		bcmgenfb_set_ioctl(&rpi_ioctl);
#ifdef DDB
		db_trap_callback = bcmgenfb_ddb_trap_callback;
#endif
		if (rpi_fb_init(dict, aux) == false)
			return;
		if (get_bootconf_option(boot_args, "console",
		    BOOTOPT_TYPE_STRING, &ptr) && strncmp(ptr, "fb", 2) == 0) {
			device_setprop_bool(dev, "is_console", true);
#if NUKBD > 0
			/* allow ukbd to be the console keyboard */
			ukbd_cnattach();
#endif
		} else {
			/* XXX Needed? */
			device_setprop_bool(dev, "is_console", false);
		}
	}
#endif
}

static u_int
bcm283x_platform_uart_freq(void)
{

	/*
	 * We are safe to access stdout phandle - consinit did before
	 * calling fp_uart_freq
	 */
	const int phandle = fdtbus_get_stdout_phandle();

	static const struct device_compatible_entry aux_compat_data[] = {
		{ .compat = "brcm,bcm2835-aux-uart" },
		DEVICE_COMPAT_EOL
	};

	if (of_compatible_match(phandle, aux_compat_data))
		return core_clk * 2;

	return uart_clk;
}

#if defined(SOC_BCM2835)
static const struct fdt_platform bcm2835_platform = {
	.fp_devmap = bcm2835_platform_devmap,
	.fp_bootstrap = bcm2835_platform_bootstrap,
	.fp_init_attach_args = bcm2835_platform_init_attach_args,
	.fp_device_register = bcm283x_platform_device_register,
	.fp_reset = bcm2835_system_reset,
	.fp_delay = bcm2835_tmr_delay,
	.fp_uart_freq = bcm283x_platform_uart_freq,
};

FDT_PLATFORM(bcm2835, "brcm,bcm2835", &bcm2835_platform);
#endif

#if defined(SOC_BCM2836)

static const struct fdt_platform bcm2836_platform = {
	.fp_devmap = bcm2836_platform_devmap,
	.fp_bootstrap = bcm2836_platform_bootstrap,
	.fp_init_attach_args = bcm2836_platform_init_attach_args,
	.fp_device_register = bcm283x_platform_device_register,
	.fp_reset = bcm2835_system_reset,
	.fp_delay = gtmr_delay,
	.fp_uart_freq = bcm283x_platform_uart_freq,
	.fp_mpstart = arm_fdt_cpu_mpstart,
};

static const struct fdt_platform bcm2837_platform = {
	.fp_devmap = bcm2836_platform_devmap,
	.fp_bootstrap = bcm2836_platform_bootstrap,
	.fp_init_attach_args = bcm2836_platform_init_attach_args,
	.fp_device_register = bcm283x_platform_device_register,
	.fp_reset = bcm2835_system_reset,
	.fp_delay = gtmr_delay,
	.fp_uart_freq = bcm283x_platform_uart_freq,
	.fp_mpstart = arm_fdt_cpu_mpstart,
};

static const struct fdt_platform bcm2711_platform = {
	.fp_devmap = bcm2711_platform_devmap,
	.fp_bootstrap = bcm2711_platform_bootstrap,
	.fp_init_attach_args = bcm2711_platform_init_attach_args,
	.fp_device_register = bcm283x_platform_device_register,
	.fp_reset = bcm2835_system_reset,
	.fp_delay = gtmr_delay,
	.fp_uart_freq = bcm283x_platform_uart_freq,
	.fp_mpstart = arm_fdt_cpu_mpstart,
};

FDT_PLATFORM(bcm2836, "brcm,bcm2836", &bcm2836_platform);
FDT_PLATFORM(bcm2837, "brcm,bcm2837", &bcm2837_platform);
FDT_PLATFORM(bcm2711, "brcm,bcm2711", &bcm2711_platform);
#endif
