/**********************************************************
*   TCHBOBJ: Utility for 'Touching' Borland Object Files
*   ====================================================
*   This utility touches Borland object files that contain
*   autodependency information.  It is similar to the
*   classic TOUCH utility.
*
*   The problem is that the classic 'touch' utility is
*   inadequate for Borland object files.  'touch' will at
*   best alter the EXTERNAL timestamp of the object files,
*   not the INTERNAL timestamps of the dependent files.
*
*   This routine does just that -- updates all the
*   timestamps stored in Borland object files.
*
*   Written by: Moshe Rubin
**********************************************************/
#include <stdio.h>
#include <fcntl.h>
#include <io.h>
#include <stdlib.h>
#include <mem.h>
#include <dos.h>
#include <dir.h>
#include <string.h>

#define COMENT_RECORD_TYPE      0x88
#define DEPENDENCY_FILE_CLASS   0xE9
#define FALSE   0
#define TRUE    (!FALSE)

const int INCREMENT = 20;       // FileInfoList increment

typedef unsigned int    UINT;
typedef unsigned char   UCHAR;
typedef int BOOL;

/**********************************************************
*   Class FILEINFO
**********************************************************/
class FILEINFO
{
public:
   char *pPathName;
   struct ftime ft;

   FILEINFO () { pPathName = NULL; };
   virtual ~FILEINFO () {if (pPathName) free (pPathName);};

   void SetData (char *pNewPath, struct ftime NewFT) {
      pPathName = strdup (pNewPath);
      ft = NewFT;
   };
   struct ftime &GetFTime () { return ft; };
   char *GetPathName () { return pPathName; };
};
typedef FILEINFO *PFILEINFO;

/**********************************************************
*   Class FILEINFOLIST
**********************************************************/
class FILEINFOLIST
{
protected:
   PFILEINFO   pFI;
   int         nMaxElements;
   int         nUsedElements;

public:
   FILEINFOLIST () {
      pFI = new FILEINFO [INCREMENT];
      nMaxElements  = INCREMENT;
      nUsedElements = 0;
   };
   ~FILEINFOLIST () { delete [] pFI; };
   void    AddEntry (char *pPathName, struct ftime& ft);
   FILEINFO& GetEntry (int iIndex) { return pFI [iIndex]; };
   int     GetNumUsedElements () { return nUsedElements; };
};

/**********************************************************
*   FILEINFOLIST::AddEntry
*
*       Adds a new entry to FileInfoList.  Expands the
*       table if necessary.
**********************************************************/
void FILEINFOLIST::AddEntry (char *pPathName,
                             struct ftime& ft)
{
   if (nUsedElements == nMaxElements) {
      // We must expand the array by INCREMENT elements
      int i;
      PFILEINFO pNewArray;

      nMaxElements += INCREMENT;
      pNewArray = new FILEINFO [nMaxElements];

      for (i=0; i<nUsedElements; i++)
         pNewArray [i].SetData (pFI [i].GetPathName (),
                                pFI [i].GetFTime ());
      delete [] pFI;
      pFI = pNewArray;
   }
   pFI [nUsedElements++].SetData (pPathName, ft);
}

/**********************************************************
*   FileInfoList
**********************************************************/
FILEINFOLIST FileInfoList;

/**********************************************************
*   GetFileDataTime
*
*       Returns the current data/time file timestamp for
*       <szFileName>.
**********************************************************/
BOOL GetFileDateTime (char *szFileName, struct ftime& FTime)
{
   int     hFile;
   int     i;
   BOOL    Ret = FALSE;

   // Check if <szFileName> exists in <FileInfoList> cache.
   for (i=0; i<FileInfoList.GetNumUsedElements (); i++) {
      FILEINFO& FI = FileInfoList.GetEntry (i);

      if (stricmp (FI.GetPathName (), szFileName) == 0) {
          // Yes, exists in table.  return its <ft>
          FTime = FI.GetFTime ();
          Ret = TRUE;
          break;
      }
   }

   if (!Ret) {
      // File not found in list, try to open the real file.
      hFile = open (szFileName, O_RDONLY | O_BINARY);
      if (hFile > 0) {
         // Get the external timestamp
         getftime (hFile, &FTime);
         close (hFile);
         FileInfoList.AddEntry (szFileName, FTime);
         Ret = TRUE;
      }
   }
exit:
   return Ret;
}

/**********************************************************
*   TouchObjectFile
*
*       For the given file handle, this routine modifies
*       all COMENT records of class DEPENDENCY_FILE_CLASS.
*       These COMENT records are used by Borland (and I
*       assume only by them) to embed file dependencies in
*       the object file.  These records contain a dependent
*       file name together with the <struct ftime> file
*       stamp of the file at compilation time.
*
*       The COMENT/DEPENDENCY_FILE_CLASS record has the
*       following format:
*
*           BYTE (1)         RecordType
*           int  (2)         RecordLength (starting with
*                                          the next field)
*           BYTE (1)         Attribute
*           BYTE (1)         Class
*           struct ftime (4) FTime
*           BYTE (1)         StringLen
*           char (>0)        FileName [StringLen]  (not
                                            NUL-terminated)
*           BYTE (1)         CheckSum
*
*       This routine extracts the FileName, ascertains its
*       current FTime, updates the COMENT record with the
*       new FTime, and writes it back to the object file.
**********************************************************/
void TouchObjectFile (int hFile)
{
   UCHAR   uRecordType;
   static char    cRecord [MAXPATH];

   // Find all the dependency file COMENT records
   while (read (hFile, &uRecordType, 1)) {
      int iLen;

      // Get record length
      read (hFile, &iLen, 2);

      /***************
      *   We're only interested in COMENT records.  Since
      *   the DEPENDENCY_FILE_CLASS has different formats
      *   (ie Borland overloads this record type for other
      *   purposes), we initially are only interested in
      *   record that have a minimum of one letter in the
      *   FileName.  Given the record format above, we only
      *   want records whose RecordLength is nine (9) or
      *   greater.
      ***************/
      if ((uRecordType == COMENT_RECORD_TYPE) &&
             (iLen >= 9)) {
         UCHAR   cAttr;
         UCHAR   cClass;
         long    lAttrPos;
         long    lFileNameLenPos;
         long    lNextRecPos;
         char    cFileNameLen;

         lAttrPos = lseek (hFile, 0L, SEEK_CUR);
         read (hFile, &cAttr, 1);
         read (hFile, &cClass, 1);

         // Is this a dependency file record?
         if (cClass == DEPENDENCY_FILE_CLASS) {
            /***************
            *   We set the internal timestamp of the
            *   dependency file in this record to the
            *   timestamp of the file in the file system.
            ***************/
            struct ftime ObjFTime;

            // Get dependent file name (six bytes past Attr)
            lFileNameLenPos = lAttrPos + 6;
            lseek (hFile, lFileNameLenPos, SEEK_SET);
            read (hFile, &cFileNameLen, 1);
            memset (cRecord, 0, sizeof (cRecord));
            read (hFile, &cRecord, cFileNameLen);

            // Get the ftime of the "real" file
            if (GetFileDateTime (cRecord, ObjFTime)) {
               // The real file exists.  Insert its time
               // into the object file (two bytes past
               // Attribute)
               lseek (hFile, lAttrPos + 2, SEEK_SET);
               write (hFile,
                      &ObjFTime,
                      sizeof (struct ftime));
            }
         }

         // Seek to next record
         lNextRecPos = lAttrPos + iLen;
         lseek (hFile, lNextRecPos, SEEK_SET);
      }
      else {
         // We're not interested in this record, skip over it
         lseek (hFile, iLen, SEEK_CUR);
      }
   }
}

/**********************************************************
*   HandleFileTouching
*
*       Touches all object files, given the <pFileSpec>
*       file specification wildcard.
**********************************************************/
void HandleFileTouching (char *pFileSpec)
{
   int             hFile;
   struct find_t   ft;
   unsigned        uResult;

   // Open the object file(s)
   uResult = _dos_findfirst (pFileSpec,_A_NORMAL, &ft);
   while (uResult == 0) {
      char szExt [MAXEXT];

      fnsplit (ft.name, NULL, NULL, NULL, szExt);
      if (stricmp (szExt, ".OBJ") == 0) {
         // Object file, touch internal .OBJ timestamps
         hFile = open (ft.name, O_RDWR | O_BINARY);
         if (hFile > 0) {
            printf ("Touching %s\n", ft.name);
            TouchObjectFile (hFile);
            close (hFile);
         }
      }
      uResult = _dos_findnext (&ft);
   }
}

void Usage ()
{
   printf ("\nTCHBOBJ v1.00: Set date and time of "
           "specified Borland object files to \"now\"\n\n");
   printf ("Usage: TCHBOBJ <file list>\n\n");
   printf ("Examples: TCHBOBJ *.obj\n");
   printf ("          TCHBOBJ sound.obj a*.obj\n\n");
}

main (int argc, char *argv[])
{
   if (argc == 1)
      Usage ();
   else
   {
      while (--argc > 0)
         HandleFileTouching (*++argv);
   }
   return (0);
}
