// zoomview.cpp :
// Implements Zooming functions in a CScrollView window
//
// Written by Brad Pirtle,
// CS:72450,1156, Internet:pirtle@qlogic.com
// Copyright 1994, QuickLogic Corp., all rights reserved.
// Version 1.0

#include "stdafx.h"
#include "zoomview.h"
#include "resource.h"  // For CURSORS

BEGIN_MESSAGE_MAP(CZoomView, CZoomBase)
   //{{AFX_MSG_MAP(CZoomView)
   ON_WM_LBUTTONDOWN()
   ON_WM_LBUTTONUP()
   ON_WM_MOUSEMOVE()
   ON_WM_RBUTTONDOWN()
   ON_WM_SETCURSOR()
   //}}AFX_MSG_MAP
END_MESSAGE_MAP()

IMPLEMENT_DYNAMIC(CZoomView, CZoomBase)

/*----------------------------------------------------------
   FUNCTION: CZoomView
   PURPOSE : Constructor for the CZoomView class
----------------------------------------------------------*/
CZoomView::CZoomView()
: CZoomBase()
{
   // Init zoom mode to nothing
   m_zoomMode    = MODE_ZOOMOFF;
   m_bCaptured   = FALSE;
   m_ptDragRect.SetRectEmpty();

   // Load the zoom cursor
   m_hZoomCursor = ::LoadCursor(AfxGetInstanceHandle(),
      MAKEINTRESOURCE(IDC_ZOOMCURSOR));
} // CZoomView

/*----------------------------------------------------------
   FUNCTION: ~CZoomView
   PURPOSE : Destructor for the CZoomView class
----------------------------------------------------------*/
CZoomView::~CZoomView()
{
   // Clean up the cursors if they were loaded properly
   if (m_hZoomCursor) DestroyCursor(m_hZoomCursor);
} // ~CZoomView

/*----------------------------------------------------------
   FUNCTION: AssertValid
   PURPOSE : Make sure valid class
----------------------------------------------------------*/
#ifdef _DEBUG
void CZoomView::AssertValid() const
{
   // Bypass CScrollView because of MM_ANISOTROPIC map mode
   CView::AssertValid();
} // AssertValid
#endif //_DEBUG

////////////////////////////////////////////////////////////
// CZoomView custom members to implement zoom functionality
////////////////////////////////////////////////////////////

/*----------------------------------------------------------
   FUNCTION: SetZoomMode
   PURPOSE : Put the view into the specified zoom mode
----------------------------------------------------------*/
void CZoomView::SetZoomMode (
   ZoomMode_ zoomMode)
{
   ASSERT(m_nMapMode == MM_ANISOTROPIC);
   if (zoomMode != m_zoomMode) {
      m_zoomMode = zoomMode;
      // Force cursor change now
      OnSetCursor(NULL, HTCLIENT, 0);
   }
} // SetZoomMode


/*----------------------------------------------------------
   FUNCTION: DrawBox
   PURPOSE : Draw a box - XOR if want to erase
----------------------------------------------------------*/
void CZoomView::DrawBox (
   CDC   &dc,
   CRect &rect,
   BOOL  xor)
{
   CPen pen;
   // Save the device context
   dc.SaveDC();
   if (xor) {
      dc.SetROP2(R2_NOTXORPEN);
      // 0 width = 1 device unit
      pen.CreatePen(PS_DASH,  0, RGB(0, 0, 0));
   } else {
      // 0 width = 1 device unit
      pen.CreatePen(PS_SOLID, 0, RGB(0, 0, 0));
   }
   dc.SelectObject(&pen);
   // Draw the rect with lines (eliminate rect middle fill)
   dc.MoveTo(rect.left,  rect.top);
   dc.LineTo(rect.right, rect.top);
   dc.LineTo(rect.right, rect.bottom);
   dc.LineTo(rect.left,  rect.bottom);
   dc.LineTo(rect.left,  rect.top);
   // Clean up
   dc.RestoreDC(-1);
} // DrawBox

/*----------------------------------------------------------
   FUNCTION: DrawLine
   PURPOSE : Draw a line - XOR to erase
----------------------------------------------------------*/
void CZoomView::DrawLine (
   CDC &dc,
   const int &x1, // Logical units
   const int &y1,
   const int &x2,
   const int &y2,
   BOOL      xor)
{  
   CPen pen;
   // Save the device context
   dc.SaveDC();
   if (xor) {
      dc.SetROP2(R2_NOTXORPEN);
      // 0 width = 1 device unit
      pen.CreatePen(PS_DASH,  0, RGB(0, 0, 0));
   } else {
      // 0 width = 1 device unit
      pen.CreatePen(PS_SOLID, 0, RGB(0, 0, 0));
   }
   dc.SelectObject(&pen);

   // Draw the line
   dc.MoveTo(x1, y1);
   dc.LineTo(x2, y2);
   // Clean up
   dc.RestoreDC(-1);
} // DrawLine

/*----------------------------------------------------------
   FUNCTION: OnLButtonDown
   PURPOSE : Handle the left mouse click
----------------------------------------------------------*/
void CZoomView::OnLButtonDown(
   UINT nFlags,
   CPoint point)
{
   // Pass the message along
   CZoomBase::OnLButtonDown(nFlags, point);

   switch (m_zoomMode) {
      case MODE_ZOOMIN:
         // Capture the mouse for zooming in
         m_bCaptured = TRUE;
         SetCapture();
         // Save the mouse down point for XOR rect
         ViewDPtoLP(&point);
         m_ptDragRect.SetRect(point.x, point.y,
                              point.x, point.y);
         // Set the cursor to the cross hair
         ::SetCursor(::LoadCursor(NULL,
            MAKEINTRESOURCE(IDC_CROSS)));
         break;

      default:
         // Do nothing.
         break;
   }
} // OnLButtonDown

/*----------------------------------------------------------
   FUNCTION: OnMouseMove
   PURPOSE : Handle the mouse movement
----------------------------------------------------------*/
void CZoomView::OnMouseMove(UINT nFlags, CPoint point)
{
   // Pass the message along
   CZoomBase::OnMouseMove(nFlags, point);

   if (m_bCaptured) {
      // Get the Device Context
      CClientDC dc(this);
      OnPrepareDC(&dc);

      switch (m_zoomMode) {
         case MODE_ZOOMIN:
            // Draw the drag-rect
            // Erase last rect
            DrawBox(dc, m_ptDragRect);
            // Draw new rect
            dc.DPtoLP(&point);
            m_ptDragRect.BottomRight() = point;
            DrawBox(dc, m_ptDragRect);
            break;

         default:
            // Do nothing.
            break;
      }
   }
} // OnMouseMove

/*----------------------------------------------------------
   FUNCTION: OnLButtonUp
   PURPOSE : Handle the left mouse release
----------------------------------------------------------*/
void CZoomView::OnLButtonUp (
   UINT nFlags,
   CPoint point)
{
   // Pass the message along
   CZoomBase::OnLButtonUp(nFlags, point);

   switch (m_zoomMode) {
      case MODE_ZOOMIN:
         // Uncapture the mouse?
         if (m_bCaptured) {
            m_bCaptured = FALSE;
            ReleaseCapture();
            // Set back the cross cursor to the Z
            ::SetCursor(m_hZoomCursor);
            // Get the Device Context
            CClientDC dc(this);
            OnPrepareDC(&dc);
            // Erase the bounding box
            DrawBox(dc, m_ptDragRect);
            // Now Zoom in on logical rectangle
            DoZoomIn(m_ptDragRect);
         }
         break;

      case MODE_ZOOMOUT:
         ViewDPtoLP(&point);
         DoZoomOut(&point);
         break;

      default:
         // Do nothing.
         break;
   }
} // OnLButtonUp

/*----------------------------------------------------------
   FUNCTION: OnRButtonDown
   PURPOSE : Handle the right mouse click
             CANCELS CURRENT ZOOM MODE OR DRAG
----------------------------------------------------------*/
void CZoomView::OnRButtonDown(UINT nFlags, CPoint point)
{
   CZoomBase::OnRButtonDown(nFlags, point);

   // See if currently captured
   if (m_bCaptured) {
      // Maintain current mode, just stop current drag
      m_bCaptured = FALSE;
      ReleaseCapture();
      // Get the Device Context
      CClientDC dc(this);
      OnPrepareDC(&dc);

      switch (m_zoomMode) {
         case MODE_ZOOMIN:
            // Erase last rect
            DrawBox(dc, m_ptDragRect);
            break;

         default:
            // Do nothing.
            break;
      }
   } else {
      // Cancel current mode
      m_zoomMode = MODE_ZOOMOFF;
   }
} // OnRButtonDown

/*----------------------------------------------------------
   FUNCTION: OnSetCursor
   PURPOSE : Set the cursor depending on the zoom mode
----------------------------------------------------------*/
BOOL CZoomView::OnSetCursor (
   CWnd* pWnd,
   UINT nHitTest,
   UINT message)
{
   if (nHitTest != HTCLIENT)
     return CZoomBase::OnSetCursor(pWnd, nHitTest, message);

   switch (m_zoomMode) {
      case MODE_ZOOMOFF:
         ::SetCursor(::LoadCursor(NULL, IDC_ARROW));
         break;

      default:
         // All other zoom modes
         ::SetCursor(m_hZoomCursor);
         break;
   } // Zoom mode select

   return TRUE;
} // OnSetCursor

