// zoombase.cpp :
// Implements Zooming functions in a CScrollView window
// Written by Brad Pirtle
//            CS:72450,1156, Internet:pirtle@qlogic.com
// Copyright 1994, QuickLogic Corp., all rights reserved.
// Version 1.0

#include "stdafx.h"
#include "zoombase.h"

IMPLEMENT_DYNAMIC(CZoomBase, CScrollView)

////////////////////////////////////////////////////////////
// CZoomBase overridden default CScrollView members
////////////////////////////////////////////////////////////

/*----------------------------------------------------------
   FUNCTION: SetZoomSizes
   PURPOSE : Set up the CZoomBase class with the logical
             page size, and scrolling page/line units.
             This replaces CScrollView::SetScrollSizes.
----------------------------------------------------------*/
void CZoomBase::SetZoomSizes (
   SIZE sizeTotal,
   const SIZE& sizePage, // in logical units
   const SIZE& sizeLine) // in logical units
{
   // Set up the defaults
   ASSERT(sizeTotal.cx >= 0 && sizeTotal.cy >= 0);
   m_nMapMode    = MM_ANISOTROPIC; // For arbitrary scaling
   m_totalLog    = sizeTotal;
   // Setup default Viewport extent to be conversion of
   // Window extent into device units.
   //BLOCK for DC
   {
      CWindowDC dc(NULL);
      dc.SetMapMode(m_nMapMode);

      // total size
      m_totalDev = m_totalLog;
      dc.LPtoDP((LPPOINT)&m_totalDev);
   } // Release DC here

   // Save the origional Viewport Extent
   m_origTotalDev = m_totalDev;
   // Save the origional scrollbar info - for CalcBars
   m_origPageDev = sizePage;
   m_origLineDev = sizeLine;
   // Fugure out scroll bar info
   CalcBars();
   // Notify the class that the zoom scale was set
   NotifyZoom();
} // SetZoomSizes

/*----------------------------------------------------------
   FUNCTION: OnPrepareDC
   PURPOSE : Override of CScrollView for MM_ANISOTROPIC
----------------------------------------------------------*/
void CZoomBase::OnPrepareDC (
   CDC* pDC,
   CPrintInfo* pInfo)
{
#ifdef _DEBUG
   if (m_nMapMode != MM_ANISOTROPIC) {
      TRACE0("Error: must call SetZoomSizes()\n");
      ASSERT(FALSE);
      return;
   }
#endif //_DEBUG

   ASSERT_VALID(pDC);
   ASSERT(m_totalLog.cx >= 0 && m_totalLog.cy >= 0);
   ASSERT(m_totalDev.cx >= 0 && m_totalDev.cx >= 0);
   // Set the Mapping mode, the window and viewport extents
   pDC->SetMapMode(m_nMapMode);
   pDC->SetWindowExt(m_totalLog); // logical coordinates
   CPoint ptVpOrg;

   if (!pDC->IsPrinting()) {
      pDC->SetViewportExt(m_totalDev); // device coordinates

      // by default shift viewport origin in
      // negative direction of scroll
      ASSERT(pDC->GetWindowOrg() == CPoint(0,0));
      ptVpOrg = -GetDeviceScrollPosition();

      // Center full fit
      if (m_bCenter) {
         CRect rect;
         GetClientRect(&rect);

         // if client area is larger than total device size,
         // override scroll positions to place origin such
         // that output is centered in the window
         if (m_totalDev.cx < rect.Width())
            ptVpOrg.x = (rect.Width() - m_totalDev.cx) / 2;
         if (m_totalDev.cy < rect.Height())
            ptVpOrg.y = (rect.Height() - m_totalDev.cy) / 2;
      }
   } else {
      // Special case for printing
      CSize  printSize;
      printSize.cx = pDC->GetDeviceCaps(HORZRES);
      printSize.cy = pDC->GetDeviceCaps(VERTRES);
      // Maintain the origional ratio, setup origin shift
      PersistRatio(m_totalLog, printSize, ptVpOrg);
      // Zoom completely out
      pDC->SetViewportExt(printSize);
   }
   // Set the new origin
   pDC->SetViewportOrg(ptVpOrg);
   // For default Printing behavior
   CView::OnPrepareDC(pDC, pInfo);
} // OnPrepareDC

/*----------------------------------------------------------
   FUNCTION: CalcBars
   PURPOSE : Update the scrollbars - uses logical units
             Call when the Viewport changes size.
----------------------------------------------------------*/
void CZoomBase::CalcBars (void)
{
   {  // BLOCK for DC
      CWindowDC dc(NULL);
      dc.SetMapMode(m_nMapMode);

      // Calculate new device units for scrollbar
      // Start w/ origional logical units from SetScrollPos
      m_pageDev = m_origPageDev;
      dc.LPtoDP((LPPOINT)&m_pageDev);
      m_lineDev = m_origLineDev;
      dc.LPtoDP((LPPOINT)&m_lineDev);
   } // Free DC

   // Make sure of the range
   if (m_pageDev.cy < 0)  m_pageDev.cy = -m_pageDev.cy;
   if (m_lineDev.cy < 0)  m_lineDev.cy = -m_lineDev.cy;

   // If none specified - use one tenth
   ASSERT(m_totalDev.cx >= 0 && m_totalDev.cy >= 0);
   if (m_pageDev.cx == 0) m_pageDev.cx = m_totalDev.cx / 10;
   if (m_pageDev.cy == 0) m_pageDev.cy = m_totalDev.cy / 10;
   if (m_lineDev.cx == 0) m_lineDev.cx = m_pageDev.cx  / 10;
   if (m_lineDev.cy == 0) m_lineDev.cy = m_pageDev.cy  / 10;

   // Now update the scrollbars
   if (m_hWnd != NULL) {
      UpdateBars();
      Invalidate(TRUE); // Zoom scale changed, redraw all
   }
} // CalcBars

////////////////////////////////////////////////////////////
// CZoomBase custom members to implement zoom functionality
////////////////////////////////////////////////////////////

/*----------------------------------------------------------
   FUNCTION: DoZoomIn
   PURPOSE : Zoom the view in on a rect
----------------------------------------------------------*/
int  CZoomBase::DoZoomIn (
   CRect &rect)   // rect in logical coordinates
{
   ASSERT(m_nMapMode == MM_ANISOTROPIC);
   // Screen pixels apart for region zoom
   const int PICKMARGIN = 10;
   // Make sure that the rect is normalized
   CRect zoomRect = rect;
   NormalizeRect(zoomRect);

   // Get the center of rect
   CPoint ptCenter;
   ptCenter.x = ((zoomRect.left + zoomRect.right)  / 2);
   ptCenter.y = ((zoomRect.top  + zoomRect.bottom) / 2);

   // See if the rect is small enough for a point zoom
   // in Device coordinates)
   CRect rectDP = zoomRect;
   ViewLPtoDP((LPPOINT)&rectDP, 2);
   BOOL bPointZoom = (max(rectDP.Width(), rectDP.Height()) <
                     PICKMARGIN);
   if (bPointZoom) {
      // Just do normal point zoom
      return DoZoomIn(&ptCenter);
   }

   CRect clientRect;
   GetClientRect(&clientRect);

   // Calculate the new zoom scale.
   float scaleH = (float) (clientRect.right  + 1) /
                  (float) zoomRect.Width();
   float scaleV = (float) (clientRect.bottom + 1) /
                  (float) zoomRect.Height();
   // Keep the scale Isotropic
   m_zoomScale = min(scaleH, scaleV);

   // Modify the Viewport extent
   m_totalDev.cx = (int) ((float) m_origTotalDev.cx *
                                  m_zoomScale);
   m_totalDev.cy = (int) ((float) m_origTotalDev.cy *
                                  m_zoomScale);
   CalcBars();

   // Set the current center point.
   CenterOnLogicalPoint(ptCenter);
   // Notify the class that a new zoom scale was done
   NotifyZoom();
   return TRUE;
} // DoZoomIn (Rect)

/*----------------------------------------------------------
   FUNCTION: DoZoomIn
   PURPOSE : Zoom in on a point by the scale factor
----------------------------------------------------------*/
int  CZoomBase::DoZoomIn (
   CPoint *point,   // point in logical coordinates
   float  delta)    // scale factor
{
   CPoint ptCenter;

   ASSERT(m_nMapMode == MM_ANISOTROPIC);
   // Save the current center point.
   if (!point) {
      ptCenter = GetLogicalCenterPoint();
   } else {
      ptCenter = *point;
   }

   // Increase the zoom scale.
   m_zoomScale *= delta;

   // Modify the Viewport extent
   m_totalDev.cx = (int) ((float) m_origTotalDev.cx *
                                  m_zoomScale);
   m_totalDev.cy = (int) ((float) m_origTotalDev.cy *
                                  m_zoomScale);
   CalcBars();
   // Set the current center point.
   CenterOnLogicalPoint(ptCenter);
   // Notify the class that a new zoom scale was done
   NotifyZoom();
   return TRUE;
} // DoZoomIn (Pt)

/*----------------------------------------------------------
   FUNCTION: DoZoomOut
   PURPOSE : Zoom out on a point by a scale factor
----------------------------------------------------------*/
int  CZoomBase::DoZoomOut (
   CPoint *point,   // point in logical coordinates
   float  delta)    // scale factor
{
   CPoint ptCenter;

   ASSERT(m_nMapMode == MM_ANISOTROPIC);
   // Save the current center point.
   if (!point) {
      ptCenter = GetLogicalCenterPoint();
   } else {
      ptCenter = *point;
   }

   // Decrease the zoom scale.
   m_zoomScale /= delta;

   // Modify the Viewport extent
   m_totalDev.cx = (int) ((float) m_origTotalDev.cx *
                                  m_zoomScale);
   m_totalDev.cy = (int) ((float) m_origTotalDev.cy *
                                  m_zoomScale);
   CalcBars();
   // Set the current center point (logical coordinates.
   CenterOnLogicalPoint(ptCenter);
   // Notify the class that a new zoom scale was done
   NotifyZoom();
   return TRUE;
} // DoZoomOut

/*----------------------------------------------------------
   FUNCTION: DoZoomFull
   PURPOSE : Zoom the view to full state
----------------------------------------------------------*/
int  CZoomBase::DoZoomFull (void)
{
   ASSERT(m_nMapMode == MM_ANISOTROPIC);
   CRect  rc;
   CPoint pt;
   CSize  sizeSb;
   // Just set Viewport Extent to Client size for full fit
   GetTrueClientSize(m_totalDev, sizeSb);
   // Maintain origional ratio
   PersistRatio(m_totalLog, m_totalDev, pt);
   // Set the new zoom scale (could use cx or cy)
   m_zoomScale = (float) m_totalDev.cx / m_origTotalDev.cx;
   // Remove the scrollbars
   UpdateBars();
   // Complete redraw
   Invalidate(TRUE);
   // Notify the class that a new zoom scale was done
   NotifyZoom();
   return TRUE;
} // DoZoomInFull

/*----------------------------------------------------------
   FUNCTION: CenterOnLogicalPoint
   PURPOSE : Same as CScrollView::CenterOnPoint,
             but for logical coordinates
----------------------------------------------------------*/
void CZoomBase::CenterOnLogicalPoint(CPoint pt)
{
   // Convert the point to device coordinates
   ViewLPtoDP(&pt);
   // Account for scroll bar position
   ClientToDevice(pt);
   // Use CScrollView's function for device coordinates
   CScrollView::CenterOnPoint(pt);
} // CenterOnLogicalPoint

/*----------------------------------------------------------
   FUNCTION: GetLogicalCenterPoint
   PURPOSE : Get the center of screen in logical coordinates
----------------------------------------------------------*/
CPoint CZoomBase::GetLogicalCenterPoint (void)
{
   CPoint pt;
   CRect rect;
   // Get the center of screen
   GetClientRect(&rect);
   pt.x = (rect.Width()  / 2);
   pt.y = (rect.Height() / 2);

   // Convert the point to logical coordinates
   ViewDPtoLP(&pt);
   return pt;
} // GetLogicalCenterPoint

/*----------------------------------------------------------
   FUNCTION: ViewDPtoLP
   PURPOSE : Same as DPtoLP, using the ClientDC for the view
----------------------------------------------------------*/
void CZoomBase::ViewDPtoLP (
   LPPOINT lpPoints,
   int     nCount)
{
   // Convert to logical units
   // Called from View when no DC is available
   ASSERT(m_nMapMode > 0); // must be set
   CWindowDC dc(this);
   OnPrepareDC(&dc);
   dc.DPtoLP(lpPoints, nCount);
} // ViewDPtoLP

/*----------------------------------------------------------
   FUNCTION: ViewLPtoDP
   PURPOSE : Same as LPtoDP, using the ClientDC for the view
----------------------------------------------------------*/
void CZoomBase::ViewLPtoDP (
   LPPOINT lpPoints,
   int     nCount)
{
   // Convert to logical units
   // Called from View when no DC is available
   ASSERT(m_nMapMode > 0); // must be set
   CWindowDC dc(this);
   OnPrepareDC(&dc);
   dc.LPtoDP(lpPoints, nCount);
} // ViewLPtoDP

/*----------------------------------------------------------
   FUNCTION: ClientToDevice
   PURPOSE : Convert from Client coordinates to
             relative Device coordinates
----------------------------------------------------------*/
void CZoomBase::ClientToDevice (
   CPoint &point)
{
   // Need to account for scrollbar position
   CPoint scrollPt = GetDeviceScrollPosition();
   point.x += scrollPt.x;
   point.y += scrollPt.y;
} // ClientToDevice

/*----------------------------------------------------------
   FUNCTION: NormalizeRect
   PURPOSE : Normalize the rectangle
----------------------------------------------------------*/
void CZoomBase::NormalizeRect (
   CRect &rect)
{
   if (rect.left > rect.right) {
      int r = rect.right;
      rect.right = rect.left;
      rect.left = r;
   }
   if (rect.top > rect.bottom) {
      int b = rect.bottom;
      rect.bottom = rect.top;
      rect.top = b;
   }
} // NormalizeRect

/*----------------------------------------------------------
   FUNCTION: PersistRatio
   PURPOSE : Make a CSize maintain the given ratio
            (by shrinking if nescessary)
----------------------------------------------------------*/
void CZoomBase::PersistRatio (
   const CSize &orig,
   CSize       &dest,
   CPoint      &remainder)
{
   float ratio1 = (float) orig.cx / orig.cy;
   float ratio2 = (float) dest.cx / dest.cy;
   int   newSize;

   // Do nothing if they are the same
   if (ratio1 > ratio2) {
      // Shrink hieght
      newSize = (int)(dest.cx / ratio1);
      remainder.x = 0;
      remainder.y = dest.cy - newSize;
      dest.cy = newSize;
   } else if (ratio2 > ratio1) {
      // Shrink width
      newSize = (int)(dest.cy * ratio1);
      remainder.x = dest.cx - newSize;
      remainder.y = 0;
      dest.cx = newSize;
   }
} // PersistRatio

