///////////////////////////////////////////////////////////////////////////////
// File		: espy.cpp
// Purpose	: Implementation of the Espy debugger
// Author	: Karl Schorr
// Date		: 6 Feb. 1995
///////////////////////////////////////////////////////////////////////////////

#include <sys\timeb.h>
#include "afxole.h"
#include "espy.h"
#include "fstream.h"
#include "resource.h"

//--- consts ---
const int EXPAND_Y = 100;
const int EXCEPT_STR_TABLE[EXCEPTION_COUNT] =
{
	IDS_ARCHIVE,  IDS_FILE, IDS_MEMORY,
	IDS_NOT_SUPP, IDS_OLE,  IDS_RESOURCE
};
const CString 	THROW_LABEL_STR = "has thrown a ";	
const CString 	CATCH_LABEL_STR = "has caught a ";	

//--- vars ---
void (CALLBACK *OrgTermFunc)();
CFrameWnd DummyWnd; 

///////////////////////////////////////////////////////////////////////////////
// Definition of EspyDialog begins here.  This is the base class
// for the ThrowDialog and CatchDialog classes.
///////////////////////////////////////////////////////////////////////////////

BEGIN_MESSAGE_MAP(EspyDialog, CDialog)
  	ON_BN_CLICKED(IDC_THROW_OTHER, OnThrowOther)
  	ON_BN_CLICKED(IDC_REM_BREAK, OnRemoveBreak)
  	ON_BN_CLICKED(IDC_REM_ALL, OnRemoveAll)
  	ON_LBN_DBLCLK(IDC_LIST, OnListChoice)
END_MESSAGE_MAP()

BOOL ThrowDialog::AllBreakPts = TRUE;


EspyDialog::EspyDialog(const UINT TemplateID) :
	CDialog(TemplateID, &DummyWnd),
	m_DlgExpanded(FALSE)
{
} 


void EspyDialog::OnBizAsUsual()
{                              
	EndDialog(IDC_BIZ_AS_USUAL);                     
}


void EspyDialog::OnThrowOther()
{                 
	if (!m_DlgExpanded)
	{
		CListBox * pListBox = (CListBox *)GetDlgItem(IDC_LIST);
				
		// Remove any existing items from the list box
		pListBox -> ResetContent();
			        
		// Load the listbox with strings
		CString ListItem;
		for (int i = 0; i < EXCEPTION_COUNT; i++)
		{
			ListItem.LoadString(EXCEPT_STR_TABLE[i]);
			pListBox->AddString(ListItem);
		}                
			         
		// Resize the dialog to display the listbox
		CRect ClientRect;
		GetWindowRect(ClientRect);
		ClientRect.SetRect(ClientRect.TopLeft().x,
		                   ClientRect.TopLeft().y,
		                   ClientRect.BottomRight().x,
		                   ClientRect.BottomRight().y + EXPAND_Y);
		MoveWindow(ClientRect); 
		m_DlgExpanded = TRUE;
	}
}


void EspyDialog::OnRemoveBreak()
{ 
	EndDialog(IDC_REM_BREAK);                     
}


void EspyDialog::OnRemoveAll()
{                      
	EndDialog(IDC_REM_ALL);                     
}


void EspyDialog::OnListChoice()
{                
	EndDialog(((CListBox *)GetDlgItem(IDC_LIST))->GetCurSel());                     
}


///////////////////////////////////////////////////////////////////////////////
// Definition of ThrowDialog begins here
///////////////////////////////////////////////////////////////////////////////

BEGIN_MESSAGE_MAP(ThrowDialog, EspyDialog)
  ON_BN_CLICKED(IDC_THROW, OnThrowIt)
  ON_BN_CLICKED(IDC_DONT, OnBizAsUsual)
  ON_BN_CLICKED(IDC_LOG, OnLog)
  ON_COMMAND(IDCANCEL, OnBizAsUsual)
END_MESSAGE_MAP()


ThrowDialog::ThrowDialog(const CString & LocationStr,
                         const UINT      ExceptCode) :
	EspyDialog(IDD_THROW),
	m_LocationStr(LocationStr),
	m_ExceptCode(ExceptCode)
{                                  
}


BOOL ThrowDialog::OnInitDialog()
{               
	CString ExceptionStr;
	
	// Fill in the dialog's static controls
	if (m_ExceptCode <= MAX_EXCEPTION) 
		ExceptionStr.LoadString(EXCEPT_STR_TABLE[m_ExceptCode]); 
		
	GetDlgItem(IDC_LOC)->SetWindowText(m_LocationStr);
	GetDlgItem(IDC_EXCEPT)->SetWindowText(THROW_LABEL_STR + ExceptionStr);	
               
	return TRUE;                    
}


void ThrowDialog::OnThrowIt()
{                              
	EndDialog(IDC_THROW);                     
}


void ThrowDialog::OnLog()
{                
	ofstream LogFile("espy.log", ios::app);
	if (LogFile)
	{
		_timeb TimeBuf;
		_ftime(&TimeBuf);
		char * TimeLine = ctime(&TimeBuf.time);
		
		CString TimeStr, DateStr;
		
		_strtime(TimeStr.GetBuffer(9));
		TimeStr.ReleaseBuffer();

		_strdate(DateStr.GetBuffer(9));
		DateStr.ReleaseBuffer();

		CString ExceptionStr;
		ExceptionStr.LoadString(EXCEPT_STR_TABLE[m_ExceptCode]);

		LogFile	<< m_LocationStr << " threw a " << ExceptionStr 
		        << " at " << TimeStr << " on " << DateStr << "." << endl;
	}
}

///////////////////////////////////////////////////////////////////////////////
// Definition of CatchDialog begins here
///////////////////////////////////////////////////////////////////////////////

BEGIN_MESSAGE_MAP(CatchDialog, EspyDialog)
  ON_BN_CLICKED(IDC_HANDLE, OnBizAsUsual)
  ON_BN_CLICKED(IDC_RETHROW,  OnReThrow)
  ON_BN_CLICKED(IDC_LOG, OnLog)
END_MESSAGE_MAP()

BOOL CatchDialog::AllBreakPts = TRUE;


CatchDialog::CatchDialog(const CString    & LocationStr,
                         const CException * pException) :
	EspyDialog(IDD_CATCH),
	m_LocationStr(LocationStr),
	m_pException(pException)
{ 
}


BOOL CatchDialog::OnInitDialog()
{               
	// Fill in the dialog's static controls
	if (m_pException != NULL)
	{
		CRuntimeClass* pRtClass = m_pException->GetRuntimeClass();
		CString ExceptionStr = pRtClass->m_lpszClassName;
	
		GetDlgItem(IDC_LOC)->SetWindowText(m_LocationStr);
		GetDlgItem(IDC_EXCEPT)->SetWindowText(CATCH_LABEL_STR + ExceptionStr);	
	}           
	return TRUE;                    
}


void CatchDialog::OnReThrow()
{                              
	EndDialog(IDC_RETHROW);                     
}


void CatchDialog::OnLog()
{                
	ofstream LogFile("espy.log", ios::app);
	
	if (LogFile && (m_pException != NULL))
	{
		_timeb TimeBuf;
		_ftime(&TimeBuf);
		char * TimeLine = ctime(&TimeBuf.time);
		
		CString TimeStr, DateStr;
		
		_strtime(TimeStr.GetBuffer(9));
		TimeStr.ReleaseBuffer();

		_strdate(DateStr.GetBuffer(9));
		DateStr.ReleaseBuffer();

		CRuntimeClass* pRtClass = m_pException->GetRuntimeClass();
		CString ExceptionStr = pRtClass->m_lpszClassName;

		LogFile	<< m_LocationStr << " caught a " << ExceptionStr 
		        << " at " << TimeStr << " on " << DateStr << "." << endl;
	}
}

///////////////////////////////////////////////////////////////////////////////
// Global functions begin here
///////////////////////////////////////////////////////////////////////////////

void DoThrowDlg(const CString & LocationStr,
                const UINT      ExceptCode,
                int           & LocalBreakPt)
{
	ThrowDialog Dlg(LocationStr, ExceptCode);
		
	UINT UserChoice = Dlg.DoModal();
	
	if (UserChoice == IDC_THROW)
		UserChoice = ExceptCode;

	switch (UserChoice)
	{
	case IDC_REM_BREAK:
		LocalBreakPt = BREAKPOINT_CLEAR;
		break;
	case IDC_REM_ALL:
		ThrowDialog::AllBreakPts = BREAKPOINT_CLEAR;
		break;
	case EX_ARCHIVE:
		AfxThrowArchiveException(CArchiveException::generic);
		break;
	case EX_FILE:
		AfxThrowFileException(CFileException::generic);
		break;
	case EX_MEMORY:
		AfxThrowMemoryException();
		break;
	case EX_NOT_SUPP:
		AfxThrowNotSupportedException();
		break;
	case EX_RESOURCE:
		AfxThrowResourceException();
		break;
	case EX_OLE:
		AfxThrowOleException(OLE_ERROR_GENERIC);
		break;
	default:
		// Only IDC_BIZ_AS_USUAL should get here.
		break;
	} 
} 


void DoCatchDlg(const CString    & LocationStr,
                const CException * pException,
                int              & LocalBreakPt)
{
	CatchDialog Dlg(LocationStr, pException);
		
	UINT UserChoice = Dlg.DoModal();
	
	switch (UserChoice)
	{
	case IDC_RETHROW:
		THROW_LAST()
		break;
	case IDC_REM_BREAK:
		LocalBreakPt = BREAKPOINT_CLEAR;
		break;
	case IDC_REM_ALL:
		CatchDialog::AllBreakPts = BREAKPOINT_CLEAR;
		break;
	case IDS_ARCHIVE:
		AfxThrowArchiveException(CArchiveException::generic);
		break;
	case IDS_FILE:
		AfxThrowFileException(CFileException::generic);
		break;
	case IDS_MEMORY:
		AfxThrowMemoryException();
		break;
	case IDS_NOT_SUPP:
		AfxThrowNotSupportedException();
		break;
	case IDS_RESOURCE:
		AfxThrowResourceException();
		break;
	case IDS_OLE:
		AfxThrowOleException(OLE_ERROR_GENERIC);
		break;
	default:
		// Only IDCANCEL should get here.
		break;
	} 
} 
void EspyInit()
{
	OrgTermFunc = AfxSetTerminate(TerminateFunc);
	DummyWnd.Create(NULL,"");
} 

void CALLBACK TerminateFunc()
{
	AfxMessageBox("Uncaught Exception!");
	OrgTermFunc();
} 

///////////////////////////////////////////////////////////////////////////////
/* End of file */
