/* Library file for accessing physical memory beyond 1 MB
 *  from Windows. */
#include <windows.h>
#include <dos.h>
#include "phys_mem.h"

/*  AllocSetSelector
 * Parameters:
 *    dwAddress - linear address
 *    dwSize - size of region
 * Return:
 *    selector for linear address region, or 0 if failure
 * Description:
 *   Allocates and sets (initializes) a selector for a linear
 *   address region. Uses Windows functions AllocSelector,
 *   SetSelectorBase and SetSelectorLimit. Note: the last two
 *   were not documented untill Windows version 3.1. The
 *   Windows function FreeSelector must be called to free the
 *   selector after use.                                   */
WORD WINAPI AllocSetSelector( DWORD dwAddress, DWORD dwSize )
   {
   WORD wSelector;
   struct SREGS SegRegs;
   segread( &SegRegs );
   wSelector = AllocSelector( SegRegs.ds );
   if ( wSelector != 0 )
      {
      SetSelectorBase( wSelector, dwAddress );
      SetSelectorLimit( wSelector, dwSize );
      }
   return ( wSelector );
   }   /* function AllocSetSelector */

/*   LockLinearRegion
 * Parameters:
 *    dwAddress - linear address of memory to lock
 *    dwSize - size of region to lock
 * Return: 0 == success, nonzero == failure
 * Description:
 *    Locks the specified linear address range.
 *    DPMI function 0x600, Lock Linear Region   */
WORD WINAPI LockLinearRegion( DWORD dwAddress, DWORD dwSize )
   {
   union REGS Regs;
   /* ax = 0x600 */
   Regs.x.ax = 0x600;
   /* bx:cx = linear address of memory to lock */
   Regs.x.bx = HIWORD( dwAddress );
   Regs.x.cx = LOWORD( dwAddress );
   /* si:di = size of region to lock (bytes) */
   Regs.x.si = HIWORD( dwSize );
   Regs.x.di = LOWORD( dwSize );
   /* Call Int 0x031 Funciton 0x600 */
   int86( 0x31, &Regs, &Regs );
   return ( Regs.x.cflag );
   }   /* function LockLinearRegion */

/*  PhysicalAddressMapping
 * Parameters:
 *    dwAddress - physical address above 1 MB
 *    dwSize - size of region to map
 * Return: linear address or 0 if error
 * Description:
 *    Converts a physical address into a linear address.
 *    This function allows device drivers running under
 *    DPMI hosts which use paging to reach physical
 *    memory that is associated with their devices above
 *    the 1MB boundry.  A descriptor must be allocated
 *    and initialized to access the memory.
 *    DPMI function 0x800, Physical Address Mapping  */
DWORD WINAPI PhysicalAddressMapping( DWORD dwAddress,
      DWORD dwSize )
   {
   union REGS Regs;
   /* ax = 0x800 */
   Regs.x.ax = 0x800;
   /* bx:cx = pysical address of memory */
   Regs.x.bx = HIWORD( dwAddress );
   Regs.x.cx = LOWORD( dwAddress );
   /* si:di = size of region to map (bytes) */
   Regs.x.si = HIWORD( dwSize );
   Regs.x.di = LOWORD( dwSize );
   /* Call Int 0x031 Function 0x800 */
   int86( 0x031, &Regs, &Regs );
   if ( Regs.x.cflag != 0 )
      /* Error */
      return ( (DWORD)0 );
   /* bx:cx = linear address */
   return (DWORD)( MAKELONG( Regs.x.cx, Regs.x.bx ));
   }   /* function PhsyicalAddressMapping */

/*   PointerFromSelector
 * Parameters:
 *    wSelector - selector to get a pointer for
 * Return: pointer to memory associated with selector
 * Description:
 *    Generates a usable pointer to memory associated
 *    with a selector.      */
LPSTR WINAPI PointerFromSelector( WORD wSelector )
   {
   return ( (LPSTR)MAKELONG( 0, wSelector ));
   }   /* function PointerFromSelector */

/* UnlockLinearRegion
 * Parameters:
 *    dwAddress - linear address of memory to unlock
 *    dwSize - size of region to unlock
 * Return: non-zero if it fails
 * Description:
 *    Unlocks a linear address range that was previously
 *    lock using the LockLinearRegion function.
 *    DPMI function 0x601, Unlock Linear Region. */
WORD WINAPI UnlockLinearRegion( DWORD dwAddress,
      DWORD dwSize )
   {
   union REGS Regs;
   /* ax = 0x601 */
   Regs.x.ax = 0x601;
   /* bx:cx = linear address of memory to unlock */
   Regs.x.bx = HIWORD( dwAddress );
   Regs.x.cx = LOWORD( dwAddress );
   /* si:di = size of region to unlock (bytes) */
   Regs.x.si = HIWORD( dwSize );
   Regs.x.di = LOWORD( dwSize );
   /* Call Int 0x031 Funciton 0x601 */
   int86( 0x031, &Regs, &Regs );
   return ( Regs.x.cflag );
   }   /* function UnlockLinearRegion */
