CREATE EXTENSION http;

set http.timeout_msec = 10000;
SELECT http_set_curlopt('CURLOPT_TIMEOUT', '10');

-- Status code
SELECT status
FROM http_get('http://localhost:9080/status/202');

-- Headers
SELECT lower(field) AS field, value
FROM (
	SELECT (unnest(headers)).*
	FROM http_get('http://localhost:9080/response-headers?Abcde=abcde')
) a
WHERE field ILIKE 'Abcde';

-- GET
SELECT status,
content::json->'args'->>'foo' AS args,
content::json->>'url' AS url,
content::json->>'method' AS method
FROM http_get('http://localhost:9080/anything?foo=bar');

-- GET with data
SELECT status,
content::json->'args'->>'this' AS args,
content::json->>'url' AS url,
content::json->>'method' AS method
FROM http_get('http://localhost:9080/anything', jsonb_build_object('this', 'that'));

-- GET with data
SELECT status,
content::json->>'args' as args,
(content::json)->>'data' as data,
content::json->>'url' as url,
content::json->>'method' as method
FROM http(('GET', 'http://localhost:9080/anything', NULL, 'application/json', '{"search": "toto"}'));

-- DELETE
SELECT status,
content::json->'args'->>'foo' AS args,
content::json->>'url' AS url,
content::json->>'method' AS method
FROM http_delete('http://localhost:9080/anything?foo=bar');

-- DELETE with payload
SELECT status,
content::json->'args'->>'foo' AS args,
content::json->>'url' AS url,
content::json->>'method' AS method,
content::json->>'data' AS data
FROM http_delete('http://localhost:9080/anything?foo=bar', 'payload', 'text/plain');

-- PUT
SELECT status,
content::json->>'data' AS data,
content::json->'args'->>'foo' AS args,
content::json->>'url' AS url,
content::json->>'method' AS method
FROM http_put('http://localhost:9080/anything?foo=bar','payload','text/plain');

-- PATCH
SELECT status,
content::json->>'data' AS data,
content::json->'args'->>'foo' AS args,
content::json->>'url' AS url,
content::json->>'method' AS method
FROM http_patch('http://localhost:9080/anything?foo=bar','{"this":"that"}','application/json');

-- POST
SELECT status,
content::json->>'data' AS data,
content::json->'args'->>'foo' AS args,
content::json->>'url' AS url,
content::json->>'method' AS method
FROM http_post('http://localhost:9080/anything?foo=bar','payload','text/plain');

-- POST with json data
SELECT status,
content::json->'form'->>'this' AS args,
content::json->>'url' AS url,
content::json->>'method' AS method
FROM http_post('http://localhost:9080/anything', jsonb_build_object('this', 'that'));

-- POST with data
SELECT status,
content::json->'form'->>'key1' AS key1,
content::json->'form'->>'key2' AS key2,
content::json->>'url' AS url,
content::json->>'method' AS method
FROM http_post('http://localhost:9080/anything', 'key1=value1&key2=value2','application/x-www-form-urlencoded');

-- HEAD
SELECT lower(field) AS field, value
FROM (
	SELECT (unnest(headers)).*
	FROM http_head('http://localhost:9080/response-headers?Abcde=abcde')
) a
WHERE field ILIKE 'Abcde';

-- Follow redirect
SELECT status,
(content::json)->>'url' AS url
FROM http_get('http://localhost:9080/redirect-to?url=get');

-- Request image
WITH
  http AS (
    SELECT * FROM http_get('http://localhost:9080/image/png')
  ),
  headers AS (
    SELECT (unnest(headers)).* FROM http
  )
SELECT
  http.content_type,
  length(text_to_bytea(http.content)) AS length_binary
FROM http, headers
WHERE field ilike 'Content-Type';

-- Alter options and and reset them and throw errors
SELECT http_set_curlopt('CURLOPT_PROXY', '127.0.0.1');
-- Error because proxy is not there
DO $$
BEGIN
    SELECT status FROM http_get('http://localhost:9080/status/555');
EXCEPTION
    WHEN OTHERS THEN
        RAISE WARNING 'Failed to connect';
END;
$$;
-- Still an error
DO $$
BEGIN
    SELECT status FROM http_get('http://localhost:9080/status/555');
EXCEPTION
    WHEN OTHERS THEN
        RAISE WARNING 'Failed to connect';
END;
$$;
-- Reset options
SELECT http_reset_curlopt();
-- Now it should work
SELECT status FROM http_get('http://localhost:9080/status/555');

-- Alter the default timeout and then run a query that is longer than
-- the default (5s), but shorter than the new timeout
SELECT http_set_curlopt('CURLOPT_TIMEOUT_MS', '10000');
SELECT status FROM http_get('http://localhost:9080/delay/7');

-- Check that statement interruption works
SET statement_timeout = 200;
CREATE TEMPORARY TABLE timer AS
  SELECT now() AS start;
SELECT *
  FROM http_get('http://localhost:9080/delay/7');
SELECT round(extract(epoch FROM now() - start) * 10) AS m
  FROM timer;
DROP TABLE timer;



