/*
 * Copyright (c) 1995
 *	Ted Lemon (hereinafter referred to as the author)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* elf2ecoff.c

   This program converts an elf executable to an ECOFF executable.
   No symbol table is retained.   This is useful primarily in building
   net-bootable kernels for machines (e.g., DECstation and Alpha) which
   only support the ECOFF object file format. */

#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/exec_ecoff.h>
#include <sys/errno.h>
#include <string.h>
#include <limits.h>

main (int argc, char **argv, char **envp)
{
  struct ecoff_filehdr efh;
  struct ecoff_aouthdr eah;
  struct ecoff_scnhdr esec;
  int i;
  int status;

  /* Read the file header... */
  i = read (0, &efh, sizeof efh);
  if (i != sizeof efh)
    {
      fprintf (stderr, "efh: %s: %s.\n",
	       argv [1], i ? strerror (errno) : "End of file reached");
      exit (1);
    }

  printf ("ef_magic = %x\n", efh.ef_magic);
  printf ("ef_nsecs = %x\n", efh.ef_nsecs);
  printf ("ef_timestamp = %x\n", efh.ef_timestamp);
  printf ("ef_symptr = %x\n", efh.ef_symptr);
  printf ("ef_syms = %x\n", efh.ef_syms);
  printf ("ef_opthdr = %x\n", efh.ef_opthdr);
  printf ("ef_flags = %x\n", efh.ef_flags);

  if (efh.ef_opthdr)
    {
      /* Read the optional header... */
      i = read (0, &eah, sizeof eah);
      if (i != sizeof eah)
	{
	  fprintf (stderr, "eah: %s: %s.\n",
		   argv [1], i ? strerror (errno) : "End of file reached");
	  exit (1);
	}
      printf ("ea_magic = %x\n", eah.ea_magic);
      printf ("ea_vstamp = %x\n", eah.ea_vstamp);
      printf ("ea_tsize = %x\n", eah.ea_tsize);
      printf ("ea_dsize = %x\n", eah.ea_dsize);
      printf ("ea_bsize = %x\n", eah.ea_bsize);
      printf ("ea_entry = %x\n", eah.ea_entry);
      printf ("ea_text_start = %x\n", eah.ea_text_start);
      printf ("ea_data_start = %x\n", eah.ea_data_start);
      printf ("ea_bss_start = %x\n", eah.ea_bss_start);
      printf ("ea_gprmask = %x\n", eah.ea_gprmask);
      printf ("ea_gp_value = %x\n", eah.ea_gp_value);
    }

  for (i = 0; i < efh.ef_nsecs; i++)
    {
      /* Read the file header... */
      status = read (0, &esec, sizeof esec);
      if (status != sizeof esec)
	{
	  fprintf (stderr, "esec: %s: %s.\n",
		   argv [1], (status
			      ? strerror (errno)
			      : "End of file reached"));
	  exit (1);
	}
      printf ("es_name = \"%s\"\n", esec.es_name);
      printf ("es_physaddr = %x\n", esec.es_physaddr);
      printf ("es_virtaddr = %x\n", esec.es_virtaddr);
      printf ("es_size = %x\n", esec.es_size);
      printf ("es_raw_offset = %x\n", esec.es_raw_offset);
      printf ("es_reloc_offset = %x\n", esec.es_reloc_offset);
      printf ("es_line_offset = %x\n", esec.es_line_offset);
      printf ("es_nreloc = %x\n", esec.es_nreloc);
      printf ("es_nline = %x\n", esec.es_nline);
      printf ("es_flags = %x\n", esec.es_flags);
    }
}

