#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <time.h>
#include <zlib.h>

#include "dpi.h"
#include "io.h"
#include "sha1.h"

struct pack_stream {
	z_stream zs;
	SHA1_CTX sha;
	char *in_buf;
	size_t in_size;
	int fd;
};

enum git_type {
	git_type_none = 0,
	git_type_commit = 1,
	git_type_tree = 2,
	git_type_blob = 3,
	git_type_tag = 4,
	git_type_ofs_delta = 6,
	git_type_ref_delta = 7,
};

static const char *git_types[] = {
	"none", "commit", "tree", "blob",
	"tag", "unused", "ofs-delta", "ref-delta"
};

static int pkt_write(int fd, const char *buf, size_t len) {
	char sizebuf[5];
	int rc;
	rc = snprintf(sizebuf, 5, "%.4x", len + 4);
	if (rc < 0) return -1;
	rc = write_all(fd, sizebuf, 4);
	if (rc < 0) return -1;
	return write_all(fd, buf, len);
}

static int pkt_flush(int fd) {
	return write_all(fd, "0000", 4);
}

static int pkt_read_header(int fd, size_t *lenp) {
	char buf[5];
	int rc;
	rc = read_all(fd, buf, 4);
	if (rc < 0) return -1;
	buf[4] = '\0';
	*lenp = strtol(buf, NULL, 16);
	return 0;
}

static int pkt_read(int fd, char *buf, size_t *lenp) {
	size_t len;
	int rc;
	rc = pkt_read_header(fd, &len);
	if (rc < 0) return -1;
	if (len == 0) {
		*lenp = 0;
		return 0;
	}
	if (len < 4) {
		errno = EPROTO;
		return -1;
	}
	len -= 4;
	if (len > *lenp) {
		errno = EMSGSIZE;
		return -1;
	}
	*lenp = len;
	return read_all(fd, buf, len);
}

static void respond_errstr(const char *errstr, const char *fmt, ...) {
	va_list ap;
	int err = errno;
	va_start(ap, fmt);
	dpi_send_header(NULL, "text/plain");
	vprintf(fmt, ap);
	printf(": %s", errstr ? errstr : strerror(err));
	va_end(ap);
}

static void respond_err(const char *fmt, ...) {
	va_list ap;
	int err = errno;
	va_start(ap, fmt);
	dpi_send_header(NULL, "text/plain");
	vprintf(fmt, ap);
	printf(": %s", strerror(err));
	va_end(ap);
}

static void respond_errx(const char *fmt, ...) {
	va_list ap;
	int err = errno;
	va_start(ap, fmt);
	dpi_send_header(NULL, "text/plain");
	vprintf(fmt, ap);
	va_end(ap);
}

static void render_err(const char *fmt, ...) {
	va_list ap;
	int err = errno;
	va_start(ap, fmt);
	vprintf(fmt, ap);
	printf(": %s", strerror(err));
	va_end(ap);
}

static int parse_id(const char *url, char *id, size_t id_len) {
	if (!*url) {
		*id = '\0';
		return 0;
	}
	while (id_len > 0 && *url != '\0') {
		*id++ = *url++;
		id_len--;
	}
	if (*url) {
		errno = EMSGSIZE;
		return -1;
	}
	*id = '\0';
	return 0;
}

static int parse_path_id(const char *url, char *path, size_t path_len, char *id, size_t id_len) {
	if (!*url) {
		*path = '\0';
		*id = '\0';
		return 0;
	}
	while (path_len > 0 && *url != '\0' && *url != '?') {
		*path++ = *url++;
		path_len--;
	}
	if (path_len <= 0) {
		errno = EMSGSIZE;
		return -1;
	}
	*path = '\0';
	if (*url == '?') return parse_id(url + 1, id, id_len);
	*id = '\0';
	return 0;
}

static int parse_port_path_id(const char *url, char *port, size_t port_len, char *path, size_t path_len, char *id, size_t id_len) {
	while (port_len > 0 && *url != '\0' && *url != '/') {
		*port++ = *url++;
		port_len--;
	}
	if (port_len <= 0) {
		errno = EMSGSIZE;
		return -1;
	}
	*port = '\0';
	if (*url == '/') return parse_path_id(url, path, path_len, id, id_len);
	*path = '\0';
	return 0;
}

static int parse_url(const char *url, char *host, size_t host_len, char *port, size_t port_len, char *path, size_t path_len, char *id, size_t id_len) {
	char *hash;

	if (!strncmp(url, "git:", 4)) url += 4;
	while (*url == '/') url++;

	hash = strchr(url, '#');
	if (hash) *hash = '\0';

	while (host_len > 0 && *url != '\0' && *url != '/' && *url != ':') {
		*host++ = *url++;
		host_len--;
	}
	if (host_len <= 0) {
		errno = EMSGSIZE;
		return -1;
	}
	*host = '\0';
	if (*url == ':') return parse_port_path_id(url + 1, port, port_len, path, path_len, id, id_len);
	*port = '\0';
	if (*url == '/') return parse_path_id(url, path, path_len, id, id_len);
	*path = '\0';
	return 0;
}

static int putchar_htmlenc(char c) {
	switch (c) {
		case '<': return printf("&lt;");
		case '>': return printf("&gt;");
		case '"': return printf("&quot;");
		case '&': return printf("&amp;");
		default: return putchar(c);
	}
}

static int putchar_urlenc(char c) {
	if (isalnum(c)) return putchar(c);
	else return printf("%%%02x", c);
}

static void print_htmlenc(const char *str) {
	char c;
	while ((c = *str++)) putchar_htmlenc(c);
}

static void print_hex(char *bytes, size_t len) {
	while (len-- > 0) {
		printf("%02x", (unsigned char)*bytes++);
	}
}

static int print_hash_link(char id[20]) {
	printf("<code><a href=\"#");
	print_hex(id, 20);
	printf("\">");
	print_hex(id, 20);
	printf("</a></code>");
}

static void read_data(int s) {
	char buf[4096];
	size_t len = sizeof(buf);
	int rc;
	while (1) {
		rc = read_some(s, buf, &len);
		if (rc < 0 && errno == EPIPE) return;
		if (rc < 0) return warn("read response");
		rc = write_all(0, buf, len);
		if (rc < 0) return warn("write response");
	}
}

static void render_ref(char *buf, size_t len) {
	char *id, *ref;
	id = buf;
	buf[len] = '\0';
	ref = strchr(buf, ' ');
	if (ref) *ref++ = '\0';
	else ref = NULL;
	printf("<a href=\"?");
	print_htmlenc(id);
	printf("\">");
	print_htmlenc(id);
	printf("</a> ");
	print_htmlenc(ref);
	putchar('\n');
}

static void render_shallow(char *buf, size_t len) {
	char *type, *id;
	type = buf;
	buf[len] = '\0';
	id = strchr(buf, ' ');
	if (id) *id++ = '\0';
	else id = NULL;
	printf("<a href=\"?");
	print_htmlenc(id);
	printf("\">");
	print_htmlenc(id);
	printf("</a> ");
	print_htmlenc(type);
	putchar('\n');
}

static int pack_read_more(struct pack_stream *s) {
	int rc;
	size_t avail_in;
	if (s->zs.avail_in > 0) {
		errno = ENOBUFS;
		return -1;
	}
	avail_in = s->in_size;
	rc = read_some(s->fd, s->in_buf, &avail_in);
	if (rc < 0) return -1;
	s->zs.next_in = s->in_buf;
	s->zs.avail_in = avail_in;
	return 0;
}

static int pack_read_object_data(struct pack_stream *s, char *buf, size_t len) {
	/* read inflated object data */
	int rc;
	s->zs.next_out = buf;
	s->zs.avail_out = len;
	while (s->zs.avail_out > 0) {
		rc = inflate(&s->zs, Z_SYNC_FLUSH);
		switch (rc) {
		case Z_BUF_ERROR:
			rc = pack_read_more(s);
			if (rc < 0) return -1;
			break;
		case Z_STREAM_END:
			if (s->zs.avail_out == 0) break;
			errno = EPIPE;
			return -1;
		case Z_STREAM_ERROR: errno = EIO; return -1;
		case Z_MEM_ERROR: errno = ENOMEM; return -1;
		case Z_DATA_ERROR: errno = EBADMSG; return -1;
		case Z_OK: break;
		default: errno = ENOTSUP; return -1;
		}
	}
	SHA1Update(&s->sha, buf, len);
	return 0;
}

static int pack_read_mode(struct pack_stream *s) {
	int rc;
	unsigned char c;
	unsigned int mode = 0;
	while (1) {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) return -1;
		if (c == ' ') break;
		if (c < '0' || c >= '8') { errno = EPROTO; return -1; }
		mode <<= 3;
		mode |= (c - '0');
	}
	return mode;
}

static int pack_read_line(struct pack_stream *s, char *buf, size_t size) {
	int rc;
	char c;
	while (size-- > 0) {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) break;
		if (c == '\n') break;
		*buf++ = c;
	}
	*buf = '\0';
	if (rc < 0) return -1;
	if (size == 0) {
		errno = EMSGSIZE;
		return -1;
	}
	return 0;
}

static int pack_read_char(struct pack_stream *s, unsigned char *charp) {
	int rc;
	/* read uncompressed data instead of letting the zstream read it */
	if (s->zs.avail_in == 0) {
		rc = pack_read_more(s);
		if (rc < 0) return -1;
	}
	*charp = *s->zs.next_in++;
	s->zs.avail_in--;
	return 0;
}

static int pack_read_hash(struct pack_stream *s, unsigned char *hash) {
	int rc;
	size_t i;
	for (i = 0; i < 20; i++) {
		rc = pack_read_char(s, hash++);
		if (rc < 0) return -1;
	}
	return 0;
}

static int pack_read_inflated_offset_size(struct pack_stream *s, char cmd, size_t *offsetp, size_t *sizep) {
	int rc;
	char c;
	size_t offset = 0, size = 0;

	if (cmd & 1) {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) return -1;
		offset = c;
	}
	if (cmd & 2) {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) return -1;
		offset |= c << 8;
	}
	if (cmd & 4) {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) return -1;
		offset |= c << 16;
	}
	if (cmd & 8) {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) return -1;
		offset |= c << 24;
	}
	if (cmd & 16) {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) return -1;
		size = c;
	}
	if (cmd & 32) {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) return -1;
		size |= c << 8;
	}
	if (cmd & 64) {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) return -1;
		size |= c << 16;
	}
	if (size == 0) size = 0x10000;
	*offsetp = offset;
	*sizep = size;
	return 0;
}

static int pack_read_inflated_varint(struct pack_stream *s, size_t *valuep) {
	int rc;
	unsigned char c;
	size_t shift = 0;
	size_t value = 0;
	do {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) return -1;
		value += (c & 0x7f) << shift;
		shift += 7;
	} while (c & 0x80);
	*valuep = value;
	return 0;
}


static int pack_read_typed_varint(struct pack_stream *s, unsigned char *typep, size_t *sizep) {
	int rc;
	unsigned char c;
	size_t shift = 4;
	size_t value;

	rc = pack_read_char(s, &c);
	if (rc < 0) return -1;
	*typep = (c >> 4) & 7;
	value = c & 15;
	while (c & 0x80) {
		rc = pack_read_char(s, &c);
		if (rc < 0) return -1;
		value += (c & 0x7f) << shift;
		shift += 7;
	}
	*sizep = value;
	return 0;
}

static int render_string(struct pack_stream *s) {
	int rc;
	char c;
	while (1) {
		rc = pack_read_object_data(s, &c, 1);
		if (rc < 0) return -1;
		if (c == '\0') break;
		putchar_htmlenc(c);
	}
	return 0;
}

static void render_text(struct pack_stream *s, char *prefix, size_t prefix_len) {
	int rc;
	char c;
	int err;
	printf("<pre>");
	while (prefix_len-- > 0) {
		putchar_htmlenc(*prefix++);
	}
	while (!pack_read_object_data(s, &c, 1)) {
		putchar_htmlenc(c);
	}
	err = errno;
	printf("</pre>");
	if (err != EPIPE) return (void)printf("object decode: %s\n", strerror(err));
}

static const char *detect_image_type(const char buf[4]) {
	if (!memcmp(buf, "GIF8", 4)) return "image/gif";
	if (!memcmp(buf, "\x89PNG", 4)) return "image/png";
	if (!memcmp(buf, "\xff\xd8", 2)) return "image/jpeg";
	return NULL;
}

static void render_blob(struct pack_stream *s, size_t size) {
	int rc;
	char c;
	char header[4];
	const char *content_type;
	int err;

	if (size < 4) return render_text(s, NULL, 0);
	rc = pack_read_object_data(s, header, 4);
	if (rc < 0) return (void)printf("object: %s\n", strerror(errno));

	content_type = detect_image_type(header);
	if (!content_type) return render_text(s, header, 4);

	printf("<img src=\"data:%s,", content_type);
	putchar_urlenc(header[0]);
	putchar_urlenc(header[1]);
	putchar_urlenc(header[2]);
	putchar_urlenc(header[3]);
	while (!pack_read_object_data(s, &c, 1)) {
		putchar_urlenc(c);
	}
	err = errno;
	printf("\">");
	if (err != EPIPE) return (void)printf("blob decode: %s\n", strerror(err));
}

static void render_author_name(char *line) {
	char *old_tz;
	char tzbuf[32];
	char *tz, *timestr;
	time_t timet;
	struct tm tm;
	char timebuf[1024];
	char tz_plus, tz_hour, tz_min;
	tz = strrchr(line, ' ');
	if (!tz) return (void)print_htmlenc(line);
	for (timestr = tz-1; timestr > line && *timestr != ' '; timestr--);
	if (!timestr || *timestr != ' ') return (void)print_htmlenc(line);
	*timestr++ = '\0';
	*tz++ = '\0';
	print_htmlenc(line);
	printf(" <b>");
	timet = atof(timestr);
	old_tz = getenv("TZ");
	if (strlen(tz) < 5) tz = "+0000";
	tz_plus = *tz == '+' ? '-' : '+';
	tz_hour = (tz[1] - '0') * 10 + (tz[2] - '0');
	tz_min = (tz[3] - '0') * 10 + (tz[4] - '0');
	snprintf(tzbuf, sizeof(tzbuf), "GIT%c%02u:%02u", tz_plus, tz_hour, tz_min);
	setenv("TZ", tzbuf, 1);
	strftime(timebuf, sizeof(timebuf), "%c", localtime(&timet));
	setenv("TZ", old_tz, 1);
	print_htmlenc(timebuf);
	printf("</b> ");
	print_htmlenc(tz);
}

static void render_commit_tag_property(char *name, char *value) {
	print_htmlenc(name);
	if (!value) return;
	printf(" ");
	if (!strcmp(name, "parent")
	 || !strcmp(name, "tree")
	 || !strcmp(name, "object")) {
		printf("<a href=\"#");
		print_htmlenc(value);
		printf("\">");
		print_htmlenc(value);
		printf("</a>");
	} else if (!strcmp(name, "author")
	        || !strcmp(name, "tagger")
	        || !strcmp(name, "committer")) {
		render_author_name(value);
	} else {
		print_htmlenc(value);
	}
	printf("\n");
}

static void render_commit_tag(struct pack_stream *s) {
	char buf[1024];
	char *value;
	int rc;
	int err;
	printf("<pre>");
	while (!(rc = pack_read_line(s, buf, sizeof(buf))) && *buf) {
		value = strchr(buf, ' ');
		if (value) *value++ = '\0';
		render_commit_tag_property(buf, value);
	}
	err = errno;
	printf("</pre>");
	if (rc < 0) {
		if (err != EPIPE) printf("<div>object: %s</div>\n", strerror(err)); 
		return;
	}
	return render_text(s, NULL, 0);
}

static void render_commit(struct pack_stream *s) {
	render_commit_tag(s);
}

static void render_tag(struct pack_stream *s) {
	render_commit_tag(s);
}

static void render_unknown(struct pack_stream *s, size_t size) {
	render_blob(s, size);
}

static int render_tree_item(struct pack_stream *s) {
	int rc, mode;
	char id[20];

	mode = pack_read_mode(s);
	if (mode < 0) return -1;
	printf("<td><code>%o</code></td>", mode);

	printf("<td><code>");
	rc = render_string(s);
	printf("</code></td>");
	if (rc < 0) return -1;

	rc = pack_read_object_data(s, id, 20);
	if (rc < 0) return -1;
	printf("<td>");
	print_hash_link(id);
	printf("</td>");

	return 0;
}

static void render_tree(struct pack_stream *s) {
	int rc;
	int err;
	printf("<table>");
	do {
		printf("<tr>");
		rc = render_tree_item(s);
		err = errno;
		printf("</tr>\n");
	} while (rc == 0);
	printf("</table>");
	if (err != EPIPE) printf("<div>error: %s</div>\n", strerror(err));
}

static int pack_read_ref_delta_header(struct pack_stream *s, char id[20], size_t *src_len, size_t *target_len) {
	int rc;
	rc = pack_read_hash(s, id);
	if (rc < 0) return -1;
	rc = pack_read_inflated_varint(s, src_len);
	if (rc < 0) return -1;
	rc = pack_read_inflated_varint(s, target_len);
	if (rc < 0) return -1;
	return 0;
}

static int pack_read_delta_data(struct pack_stream *s, size_t *offset, size_t *size, char *buf, size_t *size_passthrough) {
	char c;
	int rc;
	rc = pack_read_object_data(s, &c, 1);
	if (rc < 0) return -1;
	if (c & 0x80) {
		*size_passthrough = 0;
		return pack_read_inflated_offset_size(s, c, offset, size);
	} else {
		rc = pack_read_object_data(s, buf, c);
		if (rc < 0) return -1;
		*size = 0;
		*offset = 0;
		*size_passthrough = c;
		return 0;
	}
}

static int render_ref_delta_part(struct pack_stream *s, size_t *obj_len) {
	char buf[256];
	size_t offset, size, size_passthrough, i;
	int rc;
	size_passthrough = sizeof(buf);
	size_t len_read;
	if (!*obj_len) {
		errno = EPIPE;
		return -1;
	}
	rc = pack_read_delta_data(s, &offset, &size, buf, &size_passthrough);
	if (rc < 0) return -1;
	len_read = size + size_passthrough;
	if (*obj_len < len_read) {
		errno = EPIPE;
		return -1;
	}
	*obj_len -= len_read;
	if (size > 0) {
		printf("-%zu+%zu", offset, size);
	}
	if (size_passthrough == 0) return 0;
	printf("<pre>");
	for (i = 0; i < size_passthrough; i++) {
		putchar_htmlenc(buf[i]);
	}
	printf("</pre>");
	return 0;
}

static void render_ref_delta(struct pack_stream *s, size_t size) {
	int rc;
	char id[20];
	size_t src_len, target_len;
	int err;

	rc = pack_read_ref_delta_header(s, id, &src_len, &target_len);
	if (rc < 0) return (void) printf("ref delta: %s", strerror(errno));
	printf("<div>ref delta ");
	print_hash_link(id);
	printf(" src %zu dest %zu</div>\n", src_len, target_len);

	do rc = render_ref_delta_part(s, &target_len);
	while (rc == 0);
	err = errno;
	if (err != EPIPE) printf("<div>ref: %s</div>\n", strerror(err));
}

static void render_packfile_object2(struct pack_stream *s, enum git_type type, size_t size) {
	switch (type) {
		case git_type_commit: return render_commit(s);
		case git_type_tree: return render_tree(s);
		case git_type_blob: return render_blob(s, size);
		case git_type_tag: return render_tag(s);
		case git_type_ref_delta: return render_ref_delta(s, size);
		default: return render_unknown(s, size);
	}
}

static void render_packfile_object(struct pack_stream *s, size_t object_i) {
	int rc;
	char c;
	unsigned char type;
	unsigned char id[20];
	size_t size;
	char size_str[32];
	rc = pack_read_typed_varint(s, &type, &size);
	if (rc < 0) return (void)printf("object type error: %s<br>\n", strerror(errno));
	rc = snprintf(size_str, sizeof(size_str), "%u", size);
	if (rc < 0) return (void)printf("object size error: %s<br>\n", strerror(errno));
	rc = inflateInit(&s->zs);
	if (rc != Z_OK) {
		printf("inflate error %zu", rc);
		return;
	}
	SHA1Init(&s->sha);
	SHA1Update(&s->sha, git_types[type], strlen(git_types[type]));
	SHA1Update(&s->sha, " ", 1);
	SHA1Update(&s->sha, size_str, strlen(size_str) + 1);
	printf("<div>%u. %s: %zu bytes</div>\n", object_i, git_types[type], size);
	printf("<blockquote>");
	render_packfile_object2(s, type, size);
	printf("</blockquote>");
	rc = inflateEnd(&s->zs);
	if (rc == Z_STREAM_ERROR) {
		printf("<div>inflate error: ");
		print_htmlenc(s->zs.msg);
		printf("</div>\n");
	}
	if (type == git_type_ref_delta) return;
	SHA1Final(id, &s->sha);
	printf("<div><a name=\"");
	print_hex(id, 20);
	printf("\" href=\"#");
	print_hex(id, 20);
	printf("\"><code>");
	print_hex(id, 20);
	printf("</code></a></div>");
}

static void render_packfile(int fd) {
	struct pack_stream s;
	char header[12];
	int rc;
	long num_objects;
	char in_buf[4096];
	size_t i;

	rc = read_all(fd, header, 12);
	if (rc < 0) return (void)printf("packfile error: %s\n", strerror(errno));
	if (memcmp(header, "PACK\0\0\0\2", 8)) {
		printf("unknown packfile data: <pre>");
		read_data(fd);
		printf("</pre>");
		return;
	}
	num_objects = (header[8] << 24) | (header[9] << 16) | (header[10] << 8) | header[11];
	printf("<div>%zu objects</div>\n<hr>\n", num_objects);

	s.fd = fd;
	s.in_buf = in_buf;
	s.in_size = sizeof(in_buf);
	s.zs.next_in = NULL;
	s.zs.avail_in = 0;
	s.zs.zalloc = Z_NULL;
	s.zs.zfree = Z_NULL;

	for (i = 1; i <= num_objects; i++) {
		render_packfile_object(&s, i);
		printf("<hr>");
	}
}

static char *str_append(char *buf, const char *buf_end, char *str) {
	while (*str && buf < buf_end) {
		*buf++ = *str++;
	}
	return buf;
}

static void read_response(int s, char *want_id) {
	char buf[4096];
	char out_buf[4096];
	char *out_bufp;
	char *out_buf_end = out_buf + sizeof(out_buf);
	size_t len;
	int rc;
	char *cap;
	char *cap_end;
	size_t i;
	char advertised_agent = 0;

	dpi_send_header(NULL, "text/html");

	/* read server's capabilities or error */
	len = sizeof(buf) - 1;
	rc = pkt_read(s, buf, &len);
	if (rc < 0) return render_err("pkt_read");
	if (len > 4 && !strncmp(buf, "ERR ", 4)) {
		buf[len] = '\0';
		printf("<h3>Error</h3>\n"
				"<pre>");
		print_htmlenc(buf + 4);
		printf("</pre>");
		return;
	}
	buf[len] = '\0';
	for (i = 0; i < len && buf[i]; i++);
	cap = i < len ? buf + i + 1 : NULL;
	if (cap) {
		printf("<code>");
		print_htmlenc(cap);
		printf("</code>\n");
	}
	while (cap) {
		cap_end = strchr(cap, ' ');
		if (cap_end) *cap_end++ = '\0';
		if (!strncmp(cap, "agent=", 6)) advertised_agent = 1;
		putchar(' ');
		cap = cap_end;
	}

	if (want_id) {
		out_bufp = str_append(out_buf, out_buf_end, "want ");
		out_bufp = str_append(out_bufp, out_buf_end, want_id);
		if (advertised_agent) out_bufp = str_append(out_bufp, out_buf_end, " agent=dillo-git");
		out_bufp = str_append(out_bufp, out_buf_end, " agent=dillo-git\n");
		rc = pkt_write(s, out_buf, out_bufp - out_buf);
		if (rc < 0) return render_err("write want");
		rc = pkt_write(s, "deepen 1", 8);
		if (rc < 0) return render_err("write deepen");
		rc = pkt_flush(s);
		if (rc < 0) return render_err("write flush");
		rc = pkt_write(s, "done", 4);
		if (rc < 0) return render_err("write done");
	}
	rc = shutdown(s, SHUT_WR);
	if (rc < 0) return render_err("shutdown");

	printf("<pre>");
	while (1) {
		len = sizeof(buf) - 1;
		rc = pkt_read(s, buf, &len);
		if (rc < 0) return (void)printf("error: %s\n", strerror(errno));
		if (len == 0) break;
		if (buf[len-1] == '\n') len--;
		if (len > 8 && !strncmp(buf, "shallow ", 8)) render_shallow(buf, len);
		else render_ref(buf, len);
	}

	if (!want_id) {
		printf("</pre>");
		return;
	}

	printf("\n");
	while (1) {
		len = sizeof(buf) - 1;
		rc = pkt_read(s, buf, &len);
		if (rc < 0) return (void)printf("error: %s\n", strerror(errno));
		if (len == 0) break;
		if (buf[len-1] == '\n') len--;
		if (len > 8 && !strncmp(buf, "shallow ", 8) && strcmp(buf + 8, want_id)) continue;
		render_shallow(buf, len);
	}

	len = sizeof(buf) - 1;
	rc = pkt_read(s, buf, &len);
	if (rc < 0) return (void)printf("error: %s\n", strerror(errno));
	if (len == 4 && !strncmp(buf, "NAK\n", 4)) {}
	else if (len > 0) {
		buf[len-1] = 0;
		print_htmlenc(buf);
	}
	printf("</pre>\n");
	render_packfile(s);
}

static void respond(const char *url) {
	int rc;
	const char *errstr = NULL;
	char host[1024];
	char port[1024];
	char path[4096];
	char buf[8192];
	char id[41];
	char *bufp;
	char *buf_end = buf + sizeof(buf);
	int s;

	rc = parse_url(url, host, sizeof(host), port, sizeof(port), path, sizeof(path), id, sizeof(id));
	if (rc < 0) return respond_err("parse_url");
	if (!*host) strcpy(host, "127.0.0.1");
	if (!*port) strcpy(port, "9418");
	s = tcp_connect(host, port, &errstr);
	if (s < 0) return respond_errstr(errstr, "tcp_connect");
	bufp = str_append(buf, buf_end, "git-upload-pack ");
	bufp = str_append(bufp, buf_end, path);
	if (bufp < buf_end) *bufp++ = '\0';
	bufp = str_append(bufp, buf_end, "host=");
	bufp = str_append(bufp, buf_end, host);
	if (bufp < buf_end) *bufp++ = '\0';
	rc = pkt_write(s, buf, bufp - buf);
	if (rc < 0) return respond_err("write request");
	return read_response(s, *id ? id : NULL);
}

int main() {
	char url[1024];
	setvbuf(stdout, NULL, _IONBF, BUFSIZ);
	dpi_read_request(url, sizeof(url));
	respond(url);
}
