# Copyright (C) 2012-2013 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
from gettext import dgettext

import gobject
import sugar_stats
from sugar_network import client
from sugar_network.toolkit import Option, enforce


_ = lambda x: dgettext('sugar-plugin-stats', x)

ORDER = 100
TITLE = _('Upload usage statistics to a Sugar Network server '
          '(potentially disclosure personal data)')

stats_step = Option(
        'step interval in seconds for RRD databases to keep usage statistics',
        default=60, type_cast=int, name='stats-step')

stats_rras = Option(
        'space separated list of RRAs for RRD databases to keep '
        'usage statistics',
        default=['RRA:AVERAGE:0.5:1:4320', 'RRA:AVERAGE:0.5:5:2016'],
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        name='stats-rras')

stats_timeout = Option(
        'if more than 0, upload collected statistics to '
        'a Sugar Network server with specified delay in seconds',
        default=60 * 60, type_cast=int, name='stats-timeout')

_UPLOAD_PAGE = 256

_logger = logging.getLogger('stats')
_rrd = None
_monitor = None


def init():
    pass


def start():
    global _rrd, _monitor

    _rrd = sugar_stats.Rrd(client.profile_path('stats'),
            stats_step.value, stats_rras.value)
    _monitor = sugar_stats.Collector(_rrd)

    _logger.info('Start collecting usage statistics')
    _monitor.start()


def online_schedule():
    if not stats_timeout.value:
        _logger.info('Uploading is disabled')

    ipc = client.IPCClient()
    enforce(ipc.get(cmd='inline'), 'No connection to server')
    dbs = []

    stats_info = ipc.get(['user', client.sugar_uid()], cmd='stats-info')
    for db in _rrd:
        first = stats_info['status'].get(db.name, db.first)
        dbs.append((db.name, first, db.last))

    _logger.info('Uploading usage statistics')
    gobject.idle_add(_upload, ipc, dbs, None, None)
    return stats_timeout.value


def _upload(ipc, dbs, items, name):
    if items is None:
        if not dbs:
            _logger.info('Usage statistics successfully uploaded')
            return
        name, first, last = dbs.pop()
        items = _rrd[name].get(first, last)

    values = []
    for timestamp, row in items:
        values.append((timestamp, row))
        if len(values) >= _UPLOAD_PAGE:
            break
    else:
        items = None
    if values:
        ipc.post(['user', client.sugar_uid()],
                {'name': name, 'values': values}, cmd='stats-upload')

    gobject.idle_add(_upload, ipc, dbs, items, name)
