# Copyright (C) 2010 Collabora Ltd. <http://www.collabora.co.uk/>
# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
from hashlib import sha1

import dbus
import gconf
import gobject
from dbus import PROPERTIES_IFACE
from telepathy.interfaces import ACCOUNT, ACCOUNT_MANAGER

from sugar.profile import get_profile

from .buddy import get_owner_instance
from .connection import Connection


ACCOUNT_MANAGER_SERVICE = 'org.freedesktop.Telepathy.AccountManager'
ACCOUNT_MANAGER_PATH = '/org/freedesktop/Telepathy/AccountManager'


_logger = logging.getLogger('plugins.telepathy.neighborhood')
_model = None


class Neighborhood(gobject.GObject):

    __gsignals__ = {
            'activity-added': (
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, [object]),
            'activity-removed': (
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, [object]),
            'buddy-added': (
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, [object]),
            'buddy-removed': (
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, [object]),
            'current-activity-changed': (
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, [object]),
            }

    def __init__(self):
        gobject.GObject.__init__(self)

        self._account_paths = []
        self._link_local_account = None
        self._server_account = None

        Connection.on_buddy_add = lambda account, buddy: \
                self.emit('buddy-added', buddy)
        Connection.on_buddy_remove = lambda account, buddy: \
                self.emit('buddy-removed', buddy)
        Connection.on_activity_add = lambda account, activity: \
                self.emit('activity-added', activity)
        Connection.on_activity_remove = lambda account, activity: \
                self.emit('activity-removed', activity)
        Connection.on_current_activity_change = lambda account, buddy: \
                self.emit('current-activity-changed', buddy)
        Connection.on_disconnect = self.__account_disconnected_cb
        Connection.on_connect = self.__account_connected_cb

        client = gconf.client_get_default()
        client.add_dir('/desktop/sugar/collaboration',
                       gconf.CLIENT_PRELOAD_NONE)
        client.notify_add('/desktop/sugar/collaboration/jabber_server',
                          self.__jabber_server_changed_cb)
        client.add_dir('/desktop/sugar/user/nick', gconf.CLIENT_PRELOAD_NONE)
        client.notify_add('/desktop/sugar/user/nick', self.__nick_changed_cb)

        account_manager = dbus.Interface(
                dbus.Bus().get_object(ACCOUNT_MANAGER_SERVICE,
                    ACCOUNT_MANAGER_PATH),
                ACCOUNT_MANAGER)
        account_manager.Get(ACCOUNT_MANAGER, 'ValidAccounts',
                dbus_interface=PROPERTIES_IFACE,
                reply_handler=self.__got_accounts_cb,
                error_handler=self.__error_handler_cb)

    @property
    def _account(self):
        for account in (self._server_account, self._link_local_account):
            if account is not None and account.ready:
                return account

    def __got_accounts_cb(self, account_paths):
        self._account_paths = account_paths

        self._link_local_account = self._ensure_link_local_account()

        conf = gconf.client_get_default()
        server = conf.get_string('/desktop/sugar/collaboration/jabber_server')
        if server:
            self._server_account = self._ensure_server_account()
        else:
            _logger.info('Jabber server is not set, no Gabble connection')

    def __error_handler_cb(self, error):
        raise RuntimeError(error)

    def __account_connected_cb(self, account):
        if account is self._server_account:
            _logger.debug('Connected to Gabble, disable Salut')
            self._link_local_account.disable()

    def __account_disconnected_cb(self, account):
        if account is self._server_account:
            _logger.debug('Disconnected from Gabble, enable Salut')
            self._link_local_account.enable()

    def _get_published_name(self):
        """Construct the published name based on the public key

        Limit the name to be only 8 characters maximum. The avahi
        service name has a 64 character limit. It consists of
        the room name, the published name and the host name.

        """
        public_key_hash = sha1(get_profile().pubkey).hexdigest()
        return public_key_hash[:8]

    def _ensure_link_local_account(self):
        for account_path in self._account_paths:
            if 'salut' in account_path:
                _logger.debug('Already have a Salut account')
                account = Connection(account_path)
                account.enable()
                return account

        _logger.debug('Still dont have a Salut account, creating one')

        client = gconf.client_get_default()
        nick = client.get_string('/desktop/sugar/user/nick')

        params = {
                'nickname': nick,
                'first-name': '',
                'last-name': '',
                'jid': self._get_jabber_account_id(),
                'published-name': self._get_published_name(),
                }

        properties = {
                ACCOUNT + '.Enabled': True,
                ACCOUNT + '.Nickname': nick,
                ACCOUNT + '.ConnectAutomatically': True,
                }

        bus = dbus.Bus()
        obj = bus.get_object(ACCOUNT_MANAGER_SERVICE, ACCOUNT_MANAGER_PATH)
        account_manager = dbus.Interface(obj, ACCOUNT_MANAGER)
        account_path = account_manager.CreateAccount('salut', 'local-xmpp',
                                                     'salut', params,
                                                     properties)
        return Connection(account_path)

    def _ensure_server_account(self):
        for account_path in self._account_paths:
            if 'gabble' in account_path:
                _logger.debug('Already have a Gabble account')
                account = Connection(account_path)
                account.enable()
                return account

        _logger.debug('Still dont have a Gabble account, creating one')

        client = gconf.client_get_default()
        nick = client.get_string('/desktop/sugar/user/nick')
        server = client.get_string('/desktop/sugar/collaboration'
                                   '/jabber_server')
        key_hash = get_profile().privkey_hash

        params = {
                'account': self._get_jabber_account_id(),
                'password': key_hash,
                'server': server,
                'resource': 'sugar',
                'require-encryption': True,
                'ignore-ssl-errors': True,
                'register': True,
                'old-ssl': True,
                'port': dbus.UInt32(5223),
                'alias': nick,
                }

        properties = {
                ACCOUNT + '.Enabled': True,
                ACCOUNT + '.Nickname': nick,
                ACCOUNT + '.ConnectAutomatically': True,
                }

        bus = dbus.Bus()
        obj = bus.get_object(ACCOUNT_MANAGER_SERVICE, ACCOUNT_MANAGER_PATH)
        account_manager = dbus.Interface(obj, ACCOUNT_MANAGER)
        account_path = account_manager.CreateAccount('gabble', 'jabber',
                                                     'jabber', params,
                                                     properties)
        return Connection(account_path)

    def _get_jabber_account_id(self):
        public_key_hash = sha1(get_profile().pubkey).hexdigest()
        client = gconf.client_get_default()
        server = client.get_string('/desktop/sugar/collaboration'
                                   '/jabber_server')
        return '%s@%s' % (public_key_hash, server)

    def __jabber_server_changed_cb(self, client, timestamp, entry, *extra):
        _logger.debug('Reflect on jabber server change')

        server = client.get_string(
            '/desktop/sugar/collaboration/jabber_server')

        if server:
            if self._server_account is None:
                self._server_account = self._ensure_server_account()
            else:
                self._server_account.enable()
        else:
            if self._server_account is not None:
                self._server_account.disable()
            return

        bus = dbus.Bus()
        account = bus.get_object(ACCOUNT_MANAGER_SERVICE,
                                 self._server_account.object_path)

        account_id = self._get_jabber_account_id()
        params_needing_reconnect = account.UpdateParameters(
            {'server': server,
             'account': account_id,
             'register': True},
            dbus.Array([], 's'), dbus_interface=ACCOUNT)
        if params_needing_reconnect:
            account.Reconnect()

        self._update_jid()

    def __nick_changed_cb(self, client, timestamp, entry, *extra):
        if self._server_account is None:
            return

        _logger.debug('Reflect on nickname change')

        nick = client.get_string('/desktop/sugar/user/nick')

        bus = dbus.Bus()
        server_obj = bus.get_object(ACCOUNT_MANAGER_SERVICE,
                                    self._server_account.object_path)
        server_obj.Set(ACCOUNT, 'Nickname', nick,
                       dbus_interface=PROPERTIES_IFACE)

        link_local_obj = bus.get_object(ACCOUNT_MANAGER_SERVICE,
                                        self._link_local_account.object_path)
        link_local_obj.Set(ACCOUNT, 'Nickname', nick,
                           dbus_interface=PROPERTIES_IFACE)
        params_needing_reconnect = link_local_obj.UpdateParameters(
            {'nickname': nick, 'published-name': self._get_published_name()},
            dbus.Array([], 's'), dbus_interface=ACCOUNT)
        if params_needing_reconnect:
            link_local_obj.Reconnect()

        self._update_jid()

    def _update_jid(self):
        bus = dbus.Bus()
        account = bus.get_object(ACCOUNT_MANAGER_SERVICE,
                                 self._link_local_account.object_path)

        account_id = self._get_jabber_account_id()
        params_needing_reconnect = account.UpdateParameters(
            {'jid': account_id}, dbus.Array([], 's'), dbus_interface=ACCOUNT)
        if params_needing_reconnect:
            account.Reconnect()

    def get_buddies(self):
        # TODO Needs to be removed after changing meshbox.py
        # Only meshbox.py needs this function and only for getting owner
        return [get_owner_instance()]

    def get_buddy_by_key(self, pubkey):
        if self._account is None:
            buddies = [get_owner_instance()]
        else:
            buddies = self._account.buddies.itervalues()
        for buddy in buddies:
            if buddy.pubkey == pubkey:
                return buddy

    def get_buddy_by_handle(self, contact_handle):
        if self._account is not None:
            return self._account.buddies.get(contact_handle)

    def get_activity(self, activity_id):
        for activity in self.get_activities():
            if activity.activity_id == activity_id:
                return activity

    def get_activity_by_room(self, room_handle):
        if self._account is not None:
            return self._account.activities.get(room_handle)

    def get_activities(self):
        if self._account is None:
            return []
        else:
            return self._account.activities.values()


def get_model():
    global _model
    if _model is None:
        _model = Neighborhood()
    return _model
