# Copyright (C) 2012-2013 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
from gettext import dgettext

import gobject
import sugar_stats
from sugar_network import client
from sugar_network.toolkit import Option
from jarabe import plugins


_ = lambda x: dgettext('sugar-plugin-stats', x)

ORDER = 100
TITLE = _('Upload usage statistics to a Sugar Network server '
          '(potentially disclosure personal information)')

stats_step = Option(
        'step interval in seconds for RRD databases to keep usage statistics',
        default=60, type_cast=int, name='stats-step')

stats_rras = Option(
        'space separated list of RRAs for RRD databases to keep '
        'usage statistics',
        default=['RRA:AVERAGE:0.5:1:4320', 'RRA:AVERAGE:0.5:5:2016'],
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        name='stats-rras')

stats_timeout = Option(
        'if more than 0, upload collected statistics to '
        'a Sugar Network server with specified delay in seconds',
        default=60 * 60, type_cast=int, name='stats-timeout')

_UPLOAD_PAGE = 256

_logger = logging.getLogger('plugins.stats')
_rrd = None
_monitor = None
_schedule = None


def init():
    pass


def start():
    global _rrd, _monitor, _schedule

    _rrd = sugar_stats.Rrd(client.profile_path('stats'),
            stats_step.value, stats_rras.value)
    _monitor = sugar_stats.Collector(_rrd)

    _logger.info('Start collecting usage statistics')
    _monitor.start()

    if not stats_timeout.value:
        _logger.info('Uploading disabled')
        return

    sn = plugins.get('sn')
    if sn is None:
        _logger.warning('sugar-plugin-sn is not enabled, no uploading')
        return

    sn.connection().connect('event', __event_cb)
    _schedule = plugins.Schedule('stats', stats_timeout.value, _upload)


def __event_cb(sender, event, data):
    if event != 'inline':
        return
    if data['state'] == 'online':
        _schedule.start()
    elif data['state'] == 'offline':
        _schedule.stop()


def _upload():
    _logger.info('Uploading usage statistics')
    conn = plugins.get('sn').connection()

    conn.get(cmd='logon')
    uid = conn.get(cmd='whoami')['guid']

    info = conn.get(['user', uid], cmd='stats-info')
    for db in _rrd:
        first = info['status'].get(db.name, db.first)
        items = db.get(first, db.last)
        chunk = []
        while True:
            try:
                chunk.append(next(items))
                if len(chunk) < _UPLOAD_PAGE:
                    continue
            except StopIteration:
                pass
            if not chunk:
                break
            conn.post(['user', uid],
                    {'name': db.name, 'values': chunk}, cmd='stats-upload')
            chunk = []
