# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import json
import shutil
import logging
import tempfile
from os.path import exists
from gettext import dgettext

import gtk
import gobject

from sugar import util

from sugar_network import client
from sugar_network.toolkit.pylru import lrucache
from sugar.bundle.activitybundle import ActivityBundle
from sugar.bundle.contentbundle import ContentBundle
from sugar.bundle.bundleversion import NormalizedVersion
from sugar.bundle.bundle import AlreadyInstalledException
from jarabe.model import mimeregistry
from jarabe.model.bundleregistry import BundleRegistry as _Origin
from jarabe.plugins.sn import SN_BROWSER_NAME, connection
from jarabe.journal.journalentrybundle import JournalEntryBundle


_ = lambda x: dgettext('sugar-plugin-sn', x)

_logger = logging.getLogger('plugins.sn.bundleregistry')
_stub_icon_path = None


def stub_icon():
    global _stub_icon_path
    if not _stub_icon_path:
        theme = gtk.icon_theme_get_default()
        _stub_icon_path = theme.lookup_icon('empty', 0, 0).get_filename()
    return _stub_icon_path


class BundleRegistry(_Origin):

    def __init__(self):
        self._bundles_map = {}
        self._online_cache = lrucache(64)
        self._only_orig_get_bundle = False

        _Origin.__init__(self)
        connection().connect('event', self.__Event_cb)

        response = connection().get(['context'], reply=['guid'], layer='clone')
        for props in response['result']:
            self._populate_bundle(props['guid'])

    def get_bundle(self, bundle_id):
        if bundle_id == SN_BROWSER_NAME:
            return _BrowserInfo()

        bundle = _Origin.get_bundle(self, bundle_id)
        if bundle is not None or self._only_orig_get_bundle or \
                bundle_id in ('org.laptop.JournalActivity', SN_BROWSER_NAME):
            return bundle

        if bundle_id in self._online_cache:
            return self._online_cache[bundle_id]

        try:
            props = connection().get(
                    ['context', bundle_id], reply=['guid', 'title'])
            bundle = _RemoteContext(props)
        except Exception:
            _logger.warning('Cannot fetch %r activity metadata', bundle_id)
            bundle = None

        # Keep even None budles to avoid needless retrying
        self._online_cache[bundle_id] = bundle

        return bundle

    def is_activity_protected(self, bundle_id):
        if isinstance(self._bundles_map.get(bundle_id), ContextInfo):
            return True
        else:
            return _Origin.is_activity_protected(self, bundle_id)

    def uninstall(self, bundle, force=False, delete_profile=False):
        if isinstance(bundle, ContextInfo):
            del self._bundles_map[bundle.get_bundle_id()]
        else:
            return _Origin.uninstall(self, bundle, force, delete_profile)

    def remove_bundle(self, bundle_path):
        for bundle in self._bundles:
            if bundle.get_path() == bundle_path:
                self._remove_bundle(bundle)
                return True
        return False

    def _remove_bundle(self, bundle):
        del self._bundles_map[bundle.get_bundle_id()]
        self._bundles.remove(bundle)
        self.emit('bundle-removed', bundle)

    def _add_bundle(self, bundle_path, install_mime_type=False):
        self._only_orig_get_bundle = True
        try:
            bundle = _Origin._add_bundle(self, bundle_path, install_mime_type)
            if bundle is not None:
                self._bundles_map[bundle.get_bundle_id()] = bundle
        finally:
            self._only_orig_get_bundle = False
        return bundle

    def _populate_bundle(self, bundle_id):
        if bundle_id in self._bundles_map:
            return None
        try:
            bundle = _LocalContext(bundle_id)
        except Exception:
            _logger.exception('Cannot find cloned path')
            return None
        self._bundles_map[bundle_id] = bundle
        self._bundles.append(bundle)
        return bundle

    def __Event_cb(self, sender, event, data):
        if data.get('resource') != 'context' or \
                event not in ('create', 'update'):
            return
        bundle_id = data['guid']
        bundle = self._bundles_map.get(bundle_id)
        if 'clone' in connection().get(['context', bundle_id, 'layer']):
            if bundle is None:
                bundle = self._populate_bundle(bundle_id)
                if bundle is not None:
                    self._set_bundle_favorite(bundle_id,
                            bundle.get_activity_version(), True)
                    self.emit('bundle-added', bundle)
        elif bundle is not None:
            self._remove_bundle(bundle)


class ContextInfo(object):
    pass


class _LocalContext(ActivityBundle, ContextInfo):

    def __init__(self, bundle_id):
        impl_path = connection().get(['context', bundle_id], cmd='path')
        ActivityBundle.__init__(self, impl_path)


class _RemoteContext(ContextInfo):

    def __init__(self, props):
        self.props = props
        self._tmp_icon = None

    def get_name(self):
        return self.props['title']

    def get_bundle_id(self):
        return self.props['guid']

    def get_icon(self):
        if self._tmp_icon is None:
            blob = None
            try:
                blob = connection().get(
                        ['context', self.get_bundle_id(), 'artifact_icon'])
            except Exception, error:
                _logger.debug('Fail to get icon for %r: %s',
                        self.get_bundle_id(), error)
            if not blob:
                self._tmp_icon = stub_icon()
            else:
                fd, path = tempfile.mkstemp(suffix='.svg')
                os.close(fd)
                self._tmp_icon = util.TempFilePath(path)
                with file(self._tmp_icon, 'w') as f:
                    f.write(blob)

        # Cast to `str` to avoid spreading `util.TempFilePath`
        return str(self._tmp_icon)

    def get_tags(self):
        return []

    def get_activity_version(self):
        return '0'

    def get_installation_time(self):
        return 0

    def get_command(self):
        # Doesn't matter for Sugar Network activities
        return ''

    def is_user_activity(self):
        return False

    def get_path(self):
        return '/'

    def get_mime_types(self):
        return []


class _BrowserInfo(object):

    def __init__(self):
        icon = gtk.icon_theme_get_default().lookup_icon('sugar-network', 0, 0)
        self._icon_path = icon.get_filename() if icon is not None else stub_icon()

    def get_name(self):
        return _('Sugar Network')

    def get_bundle_id(self):
        return SN_BROWSER_NAME

    def get_icon(self):
        return self._icon_path

    def get_tags(self):
        return []

    def get_activity_version(self):
        return '0'

    def get_installation_time(self):
        return 0

    def get_command(self):
        return ''

    def is_user_activity(self):
        return False

    def get_path(self):
        return '/'
