# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import threading
from gettext import gettext as _

import active_document as ad
enforce = ad.util.enforce
util = ad.util


_default = object()


host = util.Option(
        _('hostname to listen incomming connections'),
        default='0.0.0.0')

port = util.Option(
        _('port number to listen incomming connections'),
        default=8000, type_cast=int)

debug = util.Option(
        _('debug logging level; multiple argument'),
        default=0, type_cast=int, short_option='-D', action='count')

foreground = util.Option(
        _('do not send the process into the background'),
        default=False, type_cast=util.Option.bool_cast, short_option='-F',
        action='store_true')

logdir = util.Option(
        _('path to the directory to place log files'),
        default='/var/log/sugar-network')

rundir = util.Option(
        _('path to the directory to place pid files'),
        default='/var/run/sugar-network')

auth = util.Option(
        _('authenticate requests using Sugar credentials'),
        default=True, type_cast=util.Option.bool_cast, action='store_true')

trust_users = util.Option(
        _('switch off user credentials check; disabling this option will ' \
                'require OpenSSH-5.6 or later.'),
        default=False, type_cast=util.Option.bool_cast, action='store_true')

keyfile = util.Option(
        _('path to SSL certificate keyfile to serve requests via HTTPS'))

certfile = util.Option(
        _('path to SSL certificate file to serve requests via HTTPS'))

master = util.Option(
        _('should server be a master for node servers; ' \
                'makes sense only for synchronization between servers'),
        default=False, type_cast=util.Option.bool_cast, action='store_true')


def pop_str(name, kwargs, default=_default):
    if name in kwargs:
        return kwargs.pop(name)
    else:
        enforce(default is not _default, BadRequest,
                _('Argument "%s" should be specified'), name)
        return default


def pop_int(name, kwargs, default=_default):
    value = pop_str(name, kwargs, default)
    if value is not default:
        # pylint: disable-msg=E1103
        enforce(value.isdigit(), BadRequest,
                _('Argument "%s" should be an integer value'), name)
        value = int(value)
    return value


def pop_list(name, kwargs, default=_default):
    value = pop_str(name, kwargs, default)
    if value is not default:
        # pylint: disable-msg=E1103
        value = value.split(',')
    return value


class HTTPStatus(Exception):

    status = None
    headers = None
    result = None


class SeeOther(HTTPStatus):

    status = '303 See Other'
    headers = {}
    result = ''

    def __init__(self, url):
        HTTPStatus.__init__(self)
        self.headers['Location'] = url


class BadRequest(HTTPStatus):

    status = '400 Bad Request'


class Unauthorized(HTTPStatus):

    status = '401 Unauthorized'
    headers = {'WWW-Authenticate': 'Sugar'}


class Forbidden(HTTPStatus):

    status = '403 Forbidden'


class NotFound(HTTPStatus):

    status = '404 Not Found'


class Request(threading.local):

    environ = None
    method = None
    url = None
    path = None
    query = None
    content = None
    content_stream = None
    content_length = None

    def read(self, size=None):
        if self.content_stream is None:
            return ''
        result = self.content_stream.read(size or self.content_length)
        if not result:
            return ''
        self.content_length -= len(result)
        return result

    def __getitem__(self, key):
        enforce(self.environ, KeyError)
        return self.environ.get('HTTP_%s' % key.upper())

    def __contains__(self, key):
        return self.environ and (('HTTP_%s' % key.upper()) in self.environ)


class Responce(threading.local):

    status = None
    _headers = None

    def reset(self):
        self.status = '200 OK'
        self._headers = {}

    def update(self, values):
        self._headers.update(values)

    def items(self):
        return self._headers.items()

    def setdefault(self, key, value=None):
        self._headers.setdefault(key, value)

    def __getitem__(self, key):
        return self._headers.get(key)

    def __setitem__(self, key, value):
        self._headers[key] = value

    def __contains__(self, key):
        return key in self._headers


class Principal(threading.local):
    """Authenticated user."""

    user = None
    authenticated = set()


request = Request()
responce = Responce()
principal = Principal()
