/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.validation;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Guard;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.StateType;
import com.evelopers.unimod.core.stateworks.Transition;

/**
 * Validates structure (checks that every attainable composite state has initial state).
 * Also validate that only root state machine has associations with {@link com.evelopers.unimod.core.stateworks.EventProviderHandler}s
 * 
 * @author Maxim Mazin
 * @version Revision: 1
 */
class StructureValidator extends StructureVisitor {
    private List strucutreListeners = new ArrayList();

    /**
     * Adds <code>listener</code> to list of structure listeners.
     * @param listener listener to add
     * @return true (as List).
     * @see StructureListener
     */
    public boolean addStrucutreListener(StructureListener listener) {
        return strucutreListeners.add(listener);
    }

    /**
     * Removes <code>listener</code> from list of structure listeners.
     * @param listener listener to remove.
     * @return true if list of listeners contained <code>listener</code>.
     * @see StructureListener
     */
    public boolean removeStrucutreListener(StructureListener listener) {
        return strucutreListeners.remove(listener);
    }

    /**
     * Checks if every attainable composite state of <code>sm</code> has
     * initial state. Every structure listener will be notified about every
     * attainable composite state that has no initial substate.
     * @param sm <code>StateMachine</code> to validate.
     */
    public void validateStructure(StateMachine sm) {
        attainableStates.clear();
        initialStateLess.clear();
        redundantInitStates.clear();
        notRootMachinesWithEventProviders.clear();
        attainableStates.clear();
               
        visit(sm.getTop());

        addAttainableSuperstates();
        for (Iterator i = sm.getAllStates().iterator();i.hasNext();) {
            State state = (State) i.next();
            if (! attainableStates.contains(state)) {
                fireUnattainableState(sm, state);
            }
        }
        
        for (Iterator i = getNoInitState().iterator(); i.hasNext();) {
            fireNoInitState(sm, (State) i.next());
        }
        
        for (Iterator i = redundantInitStates.keySet().iterator(); i.hasNext();) {
            State state = (State) i.next();
            int initStateNumber = ((Integer) redundantInitStates.get(state)).intValue();
            fireRedundantInitStates(sm, state, initStateNumber);
        }

        for (Iterator i = notRootMachinesWithEventProviders.iterator(); i.hasNext();) {
            StateMachine s = (StateMachine) i.next();
            fireNotRootStateMacineHasEventProviders(s);
        }

        for (Iterator i = sm.getAllStates().iterator(); i.hasNext();) {
            State state = (State) i.next();
            if (StateType.INITIAL.equals(state.getType())) {
                int numberOfTransitions = 0;
                for (Iterator j = state.getOutgoingTransitions().iterator(); j.hasNext();) {
                    Transition transition = (Transition) j.next();
                    numberOfTransitions ++;
                    if (! Guard.TRUE.equals(transition.getGuard())) {
                        fireBadTransitionFromInitState(sm, transition);
                    }
                    if (! Event.NO_EVENT.equals(transition.getEvent())) {
                        fireBadTransitionFromInitState(sm, transition);
                    }
                }
                if (numberOfTransitions == 0) {
                    fireNoTransitionFromInitState(sm, state);
                } else if (numberOfTransitions > 1) {
                    fireRedundantTransitionsFromInitState(sm, state, numberOfTransitions);
                }
                for (Iterator j = state.getIncomingTransitions().iterator(); j.hasNext();) {
                    Transition transition = (Transition) j.next();
                    fireIncomingTransitionToInitState(sm, transition);
                }
            } else {
                for (Iterator j = state.getOutgoingTransitions().iterator(); j.hasNext();) {
                    Transition transition = (Transition) j.next();
                    if (Event.NO_EVENT.equals(transition.getEvent())) {
                        fireNoEventOnTransition(sm, transition);
                    }
                }
            }
            if (StateType.FINAL.equals(state.getType())) {
                for (Iterator j = state.getOutgoingTransitions().iterator(); j.hasNext();) {
                    Transition transition = (Transition) j.next();
                    fireOutgoingTransitionFromFinalState(sm, transition);
                }
            }
        }
    }

    /**
     * Super state is inderectly attainable if it has attainable substate
     */
    private void addAttainableSuperstates() {
        Set attainableSupertates = new HashSet();
        for (Iterator i = attainableStates.iterator(); i.hasNext();) {
            State state = (State) i.next();
            for (State cur = state.getSuperstate(); cur != null; cur = cur.getSuperstate()) {
                attainableSupertates.add(cur);
            }
        }
        attainableStates.addAll(attainableSupertates);
    }

    private void fireUnattainableState(StateMachine sm, State state) {
        for (Iterator i = strucutreListeners.iterator(); i.hasNext();) {
            StructureListener structureListener = (StructureListener) i.next();
            structureListener.unattainableState(sm, state);
        }
    }

    private void fireNoInitState(StateMachine sm, State state) {
        for (Iterator i = strucutreListeners.iterator(); i.hasNext();) {
            StructureListener structureListener = (StructureListener) i.next();
            structureListener.noInitState(sm, state);
        }
    }

    private void fireRedundantInitStates(StateMachine sm, State state, int initStateNumber) {
        for (Iterator i = strucutreListeners.iterator(); i.hasNext();) {
            StructureListener structureListener = (StructureListener) i.next();
            structureListener.redundantInitStates(sm, state, initStateNumber);
        }
    }

    private void fireNotRootStateMacineHasEventProviders(StateMachine sm) {
        for (Iterator i = strucutreListeners.iterator(); i.hasNext();) {
            StructureListener structureListener = (StructureListener) i.next();
            structureListener.notRootStateMachineHasEventProviders(sm);
        }
    }
 
    private void fireBadTransitionFromInitState(StateMachine sm, Transition transition) {
        for (Iterator i = strucutreListeners.iterator(); i.hasNext();) {
            StructureListener structureListener = (StructureListener) i.next();
            structureListener.badTransitionFromInitState(sm, transition);
        }
    }
    
    private void fireNoTransitionFromInitState(StateMachine sm, State initState) {
        for (Iterator i = strucutreListeners.iterator(); i.hasNext();) {
            StructureListener structureListener = (StructureListener) i.next();
            structureListener.noTransitionFromInitState(sm, initState);
        }
    }
    
    private void fireRedundantTransitionsFromInitState(StateMachine sm, State initState, int transitionsNumber) {
        for (Iterator i = strucutreListeners.iterator(); i.hasNext();) {
            StructureListener structureListener = (StructureListener) i.next();
            structureListener.redundantTransitionsFromInitState(sm, initState, transitionsNumber);
        }
    }

    private void fireIncomingTransitionToInitState(StateMachine sm, Transition transition) {
        for (Iterator i = strucutreListeners.iterator(); i.hasNext();) {
            StructureListener structureListener = (StructureListener) i.next();
            structureListener.incomingTransitionToInitState(sm, transition);
        }
    }

    private void fireOutgoingTransitionFromFinalState(StateMachine sm, Transition transition) {
        for (Iterator i = strucutreListeners.iterator(); i.hasNext();) {
            StructureListener structureListener = (StructureListener) i.next();
            structureListener.outgoingTransitionFromFinalState(sm, transition);
        }
    }
    
    private void fireNoEventOnTransition(StateMachine sm, Transition transition) {
        for (Iterator i = strucutreListeners.iterator(); i.hasNext();) {
            StructureListener structureListener = (StructureListener) i.next();
            structureListener.noEventOnTransition(sm, transition);
        }
    }
}
