/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.validation;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.Transition;

/**
 * Adds some functionality to <code>State</code>
 * @author Maxim Mazin
 * @version Revision: 1
 * @see com.evelopers.unimod.core.stateworks.State
 */
class StateTransitionsDecorator {
    private State state;

    public StateTransitionsDecorator(State state) {
        this.state = state;
    }

    public State getState() {
        return state;
    }

    /**
     * @return All outgoing transitions including superstate transitions.
     */
    public List getAllTransitions() {
        List transisions = new ArrayList();
        for (State current = state; current != null; current = current.getSuperstate()) {
            transisions.addAll(current.getOutgoingTransitions());
        }
        return transisions;
    }

    /**
     * @return All outgoing transitions for <code>event</code> including
     * superstate transition.
     */
    public List getAllNotElseTransitions(Event event) {
        List transitions = new ArrayList();
        for (State current = state; current != null; current = current.getSuperstate()) {
            transitions.addAll(current.getFilteredOutgoingTransitions(event, false));
        }
//        for (State current = state; current != null; current = current.getSuperstate()) {
//            transitions.addAll(current.getFilteredOutgoingTransitions(Event.ANY, false));
//        }
        return transitions;
    }

    /**
     * @return first found Else transition that is the to the state nearest. Or <code>null</code>
     * if no else transition defined in hierarchy.
     */
    public Transition getElseTransition(Event event) {
        for (State current = state; current != null; current = current.getSuperstate()) {
            List t = current.getFilteredOutgoingTransitions(event, true);
            
            if (t != null && t.size() > 0) {
                return (Transition)t.get(0);
            }
        }
        return null;
    }

    /**
     * Returns all events that are handled in the state without {@link Event#ANY}.
     * 
     * @return All events that are handled in the state without {@link Event#ANY}
     */
    public Set getHandledEvents() {
        Set events = new HashSet();
        for (Iterator i = getAllTransitions().iterator(); i.hasNext();) {
            Transition transition = (Transition) i.next();

            Event e = transition.getEvent();
            if (!e.equals(Event.ANY)) {
                events.add(e);
            }
        }
        
        return events;
    }
}