/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.validation;

import java.util.Iterator;
import java.util.List;

import antlr.RecognitionException;

import com.evelopers.unimod.analysis.FormulaResolver;
import com.evelopers.unimod.core.ModelElement;
import com.evelopers.unimod.core.ModelVisitor;
import com.evelopers.unimod.core.VisitorException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Guard;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.StateType;
import com.evelopers.unimod.core.stateworks.Transition;

/**
 * Visits every state and checks consistency of guard conditions set.
 *
 * @author Maxim Mazin
 * @version Revision: 1
 */
abstract class ConsistencyCheckVisitor implements ModelVisitor {
    private StateMachine sm;

    public void setSm(StateMachine sm) {
        this.sm = sm;
    }
    
    /**
     * Checks consistency of <code>e</code> and its descendants. <code>e</code>
     * should be instance of {@link State}.
     * @param e state to visit.
     * @return Whether <code>e</code> is composite state and thus whether its
     * descendants should be checked.
     * @throws VisitorException if <code>e</code> or its descendants aren't
     * instances of type {@link State}.
     */
    public boolean visit(ModelElement e) throws VisitorException {
        if (e instanceof State) {
            State state = (State) e;
            checkState(state);
            return state.isComposite();
        }
        throw new VisitorException(e + " is not a state!");
    }

    /**
     * Checks consistency of {@link State} s.
     * @param s state, consistency of what should be checked.
     * @throws VisitorException if any error occurs.
     */
    protected void checkState(State s) throws VisitorException {
        if (StateType.NORMAL.equals(s.getType())) {
            StateTransitionsDecorator state = new StateTransitionsDecorator(s);
            for (Iterator i = state.getHandledEvents().iterator(); i.hasNext();) {
                Event event = (Event) i.next();

                int elseCount = s.getFilteredOutgoingTransitions(event, true).size();
                if (elseCount > 1) {
                    fireRedundantElseTransitions(sm, s, event, elseCount);
                }

                checkConsistency(state.getAllNotElseTransitions(event), state.getState());
            }
        } else if (StateType.INITIAL.equals(s.getType())) {
            checkConsistency(s.getOutgoingTransitions(), s);
        }
    }

    /**
     * @param allTransitions
     * @param state
     * @throws VisitorException
     */
    private void checkConsistency(List allTransitions, State state) throws VisitorException {
        try {
        	/* Remove from check list transitions with noncompiled guards */
        	for (Iterator i = allTransitions.iterator(); i.hasNext();) {
				Transition transition = (Transition) i.next();
				if (transition.getGuard().getAST() == null) {
					i.remove();
				}
			}
        	
        	/* Check consitency only for transitions with compiled guards */
            Guard intersection;
            for (int i = 0; i < allTransitions.size(); i++) {
                Transition t1 = (Transition) allTransitions.get(i);
                if (t1.getSourceState().equals(state)) {
                    for (int j = i + 1; j < allTransitions.size(); j++) {
                        Transition t2 = (Transition) allTransitions.get(j);
                        intersection = getIntersection(t1, t2);
                        if (intersection != null) {
                            fireInconsistentGuardSet(sm, t1, t2, intersection);
                        }
                    }
                }
            }
        } catch (RecognitionException e) {
            throw new VisitorException(e);
        }
    }

    private Guard getIntersection(Transition t1, Transition t2) throws RecognitionException  {
        FormulaResolver conjunction = FormulaResolver.createConjunction(
                t1.getGuard().getAST(), t2.getGuard().getAST());        
        if (conjunction.isUnsatisfiable()) {
            return null;
        } else {
            return sm.createGuard(conjunction.getSimplifiedDNF(true));
        }
    }

    abstract protected void fireInconsistentGuardSet(StateMachine sm, Transition t1, Transition t2, Guard guard);

    abstract protected void fireRedundantElseTransitions(StateMachine sm, State state, Event e, int elseTransitionsNumber);
}
