/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.transform.xml;

import com.evelopers.common.util.helper.StringHelper;
import com.evelopers.unimod.core.stateworks.*;
import com.evelopers.unimod.resources.Messages;
import com.evelopers.unimod.transform.TransformException;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.OutputStream;
import java.util.Iterator;

/**
 * Transforms in-memory state machine model into XML description.
 *
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
public class ModelToXML implements XML {

    private Document document = null;

	private static ModelToXML instance = null; 

	private ModelToXML() {
    }

	/**
	 * Returns transformer instance
	 * 
	 * @return singleton instance 
	 */
	public static synchronized ModelToXML create() {
		if (instance == null) {
			instance = new ModelToXML();		 
		}
		
		return instance; 
	}

    /**
     * <p>
     * Transforms given {@link StateMachine} to {@link Document} and
     * serializes it to given {@link OutputStream}.
     * </p>
     * <p>
     *  <strong>Note:</strong> this method doesn't close output stream
     * </p>
     * <p>
     *  <i>Usage:</i>
     * <code><pre>
     * StateMachine sm = StateMachine.create("A1");
     * // ... fill state machine code here ...
     * FileOutputStream out = new FileOutputStream("A1.xml");
     * StateMachineToXML.write(sm, out);
     * out.close();
     * </pre></code>
     * </p>
     * @param out output stream to serialize machine to
     * @throws TransformException if some transformation error occurs
     */
    public static void write(Model model, OutputStream out)
            throws TransformException {
        create();
        Document document = instance._transform(model);

        XMLHelper.save(document, out, PUBLIC_ID, SYSTEM_ID);
    }

    public Document transform(Model source) throws TransformException {
        return _transform(source);
    }

    private Document _transform(Model model) throws TransformException {

        // create empty XML document
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            DOMImplementation domImpl = builder.newDocument().getImplementation();
            DocumentType doctype = domImpl.createDocumentType(Tag.MODEL, PUBLIC_ID, SYSTEM_ID);
            
            document = domImpl.createDocument("", Tag.MODEL, doctype);
        } catch (Exception e) {
            throw new TransformException(e.getMessage());
        }

        try {
            createModelNode(document.getDocumentElement(), model);
        } catch (Exception e) {
            throw new TransformException(Messages.getMessages().getMessage(Messages.UNEXPECTED_EXCEPTION,
                    new Object[]{ e.getMessage(),
                                  StringHelper.stackTraceToString(e) }));
        }

        return document;
    }

    private void createModelNode(Element e, Model model) throws Exception {
        // Name attribute
        e.setAttribute(Attribute.NAME, model.getName());

        // Controlled objects
        for (Iterator iter = model.getControlledObjectHandlers().iterator(); iter.hasNext(); ) {
            e.appendChild(createControlledObjectNode((ControlledObjectHandler)iter.next()));
        }

		// Event provides
        for (Iterator i = model.getEventProviderHandlers().iterator(); i.hasNext();) {
            EventProviderHandler ep = (EventProviderHandler) i.next();
            e.appendChild(createEventProviderNode(ep));
        }
        
        // Root state machine
        e.appendChild(createRootStateMachineNode(model.getRootStateMachine()));

		// State machines
		for (Iterator iter = model.getStateMachines().iterator(); iter.hasNext(); ) {
			e.appendChild(createStateMachineNode((StateMachine)iter.next()));
		}
    }

    private Element createControlledObjectNode(ControlledObjectHandler co) throws Exception {
        Element e = document.createElement(Tag.CONTROLLED_OBJECT);

        // Name attribute
        e.setAttribute(Attribute.NAME, co.getName());

        // Class attribute
        e.setAttribute(Attribute.CLASS, co.getImplName());

        return e;
    }

    private Element createEventProviderNode(EventProviderHandler ep) throws Exception {
        Element e = document.createElement(Tag.EVENT_PROVIDER);

        // Name attribute
        if (ep.getName() != null) {
            e.setAttribute(Attribute.NAME, ep.getName());
        }

        // Class attribute
        e.setAttribute(Attribute.CLASS, ep.getImplName());
        
        // Associations
        for (Iterator i = ep.getOutgoingAssociations().iterator(); i.hasNext();) {
            Association association = (Association) i.next();
            e.appendChild(createAssociationNode(association));
        }

        return e;
    }
    
    private Element createRootStateMachineNode(StateMachine stateMachine) throws Exception {
        Element e = document.createElement(Tag.ROOT_STATE_MACHINE);

        // Reference to state machine
        e.appendChild(createStateMachineRefNode(stateMachine));

        return e;
    }

    private Element createStateMachineRefNode(StateMachine sm) {
        Element e = document.createElement(Tag.STATE_MACHINE_REF);

        // Name attribute
        e.setAttribute(Attribute.NAME, sm.getName());

        return e;
    }

    private Element createStateMachineNode(StateMachine sm) throws Exception {
        Element e = document.createElement(Tag.STATE_MACHINE);

        // Name attribute
        e.setAttribute(Attribute.NAME, sm.getName());

		// Config manager
		if (sm.getConfigManagerClassName() != null) {
			e.appendChild(createConfigManagerNode(sm.getConfigManagerClassName()));
		}

        // Associations
        for (Iterator i = sm.getOutgoingAssociations().iterator(); i.hasNext();) {
            Association association = (Association) i.next();
            e.appendChild(createAssociationNode(association));
        }

        // Top state
        e.appendChild(createStateNode(sm.getTop()));

        // Transitions
        for (Iterator i = sm.getAllTransition().iterator(); i.hasNext(); ) {
            e.appendChild(createTransitionNode((Transition)i.next()));
        }

        return e;
    }

	private Element createConfigManagerNode(String cs) throws Exception {
		Element e = document.createElement(Tag.CONFIG_STORE);

        // Class attribute
        e.setAttribute(Attribute.CLASS, cs);

		return e;
	}

    private Element createAssociationNode(Association a) throws Exception {
        Element e = document.createElement(Tag.ASSOCIATION);

        // Controlled object reference attribute
        e.setAttribute(Attribute.TARGET_REF, a.getTarget().getName());
        
        if (a.getClientRole() != null) {
            e.setAttribute(Attribute.CLIENT_ROLE, a.getClientRole());
        }

        if (a.getSupplierRole() != null) {
            e.setAttribute(Attribute.SUPPLIER_ROLE, a.getSupplierRole());
        }

        return e;
    }

    private Element createStateNode(State s) throws Exception {
        Element e = document.createElement(Tag.STATE);

        // Name attribute
        e.setAttribute(Attribute.NAME, s.getName());

        // Type attribute
        e.setAttribute(Attribute.TYPE, String.valueOf(s.getType()));

        // Substates
        for (Iterator i = s.getSubstates().iterator(); i.hasNext();) {
            State substate = (State)i.next();
            e.appendChild(createStateNode(substate));
        }

        // Included state machines
        for (Iterator i = s.getSubmachines().iterator(); i.hasNext();) {
            StateMachine subMachine = (StateMachine)i.next();
            e.appendChild(createStateMachineRefNode(subMachine));
        }

        // Output actions
        for (Iterator i = s.getOnEnterActions().iterator(); i.hasNext();) {
            Action action = (Action) i.next();
            e.appendChild(createOutputActionNode(action));
        }

        return e;
    }

    private Element createOutputActionNode(Action o) {
        Element e = document.createElement(Tag.OUTPUT_ACTION);

        // Ident attribute
        e.setAttribute(Attribute.IDENT, o.getIdentifier());

        return e;
    }

    private Element createTransitionNode(Transition t) throws Exception {
        Element e = document.createElement(Tag.TRANSITION);

        // Name attribute
        if (t.getName() != null) {
            e.setAttribute(Attribute.NAME, t.getName());
        }

        // Source attribute
        e.setAttribute(Attribute.SOURCE_REF, t.getSourceState().getName());

        // Target attribute
        e.setAttribute(Attribute.TARGET_REF, t.getTargetState().getName());

        // Event attribute
        if (t.getEvent() != null && !t.getEvent().equals(Event.NO_EVENT)) {
            e.setAttribute(Attribute.EVENT, t.getEvent().getName());
        }

        // Guard attribute
        // do not save "TRUE" guard conditions, because empty guard condition will be treated as "TRUE"
        boolean tautology = Guard.TRUE_EXPR.equalsIgnoreCase(t.getGuard().getExpr());
        if (t.getGuard() != null && !tautology) {
            e.setAttribute(Attribute.GUARD, t.getGuard().getExpr());
        }

        // Output actions
        for (Iterator i = t.getOutputActions().iterator(); i.hasNext(); ) {
            Action action = (Action) i.next();
            e.appendChild(createOutputActionNode(action));
        }

        return e;
    }
}
