/*
 * Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 * This is open source software; you can use, redistribute and/or modify 
 * it under the terms of the Open Software Licence v 2.1 as published by the Open 
 * Source Initiative.
 *
 * You should have received a copy of the Open Software Licence along with this
 * application; if not, contact the Open Source Initiative (http://opensource.org).
 */
package com.evelopers.unimod.transform.source;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;

import com.evelopers.unimod.core.stateworks.ClassElement;
import com.evelopers.unimod.core.stateworks.ControlledObjectHandler;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.EventProviderHandler;
import com.evelopers.unimod.core.stateworks.Model;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.Transition;
import com.evelopers.unimod.parser.ActionCollector;
import com.evelopers.unimod.parser.IdentNode;
import com.evelopers.unimod.parser.Infix;
import com.evelopers.unimod.parser.InterpreterException;

/**
 * Base class for tool class that may be used in velocity template. 
 * Contains several helper methods. Should be overriden by generator to
 * take into account language specific features.
 *  
 * @author vgurov, mmazin
 */
public class Tool {

    private Tool(){}
    
    public static Tool create() {
        return new Tool();
    }

    private final char[] REPLACE_SYMBOLS = {' ', '.', '/', '\\', '\'', '"'};
    
    public String toJavaIdent(String s) {
    	for (int i = 0; i < REPLACE_SYMBOLS.length; i++) {
    		s = s.replace(REPLACE_SYMBOLS[i], '_');
    	}
        
        if (s.charAt(0) >= '0' && s.charAt(0) <= '9') {
            s = "_" + s;
        }
        
        return s;
    }

    public String actionIdentToLocalVar(String s) {
        s = s.replace('.', '_');
        
        return s;
    }
    
    public Set getAllEvents(StateMachine sm) {
        Set e = new HashSet();
        
        for (Iterator i = sm.getAllStates().iterator(); i.hasNext(); ) {
            State s = (State)i.next();
            
            e.addAll(s.getEvents(false));
        }
        
        e.remove(Event.NO_EVENT);
        
        return e;
    }

    public boolean handlesEvent(State state, Event event) {
        for (State cur = state; cur != null; cur = cur.getSuperstate()) {
            List transitions = cur.getFilteredOutgoingTransitions(event,
                    false);
            transitions.addAll(state
                    .getFilteredOutgoingTransitions(event, true));
            if (!transitions.isEmpty()) {
                return true;
            }
        }
        return false;
    }

    public Set getHandledEventsWithSuperstates(State state) {
        Set e = new HashSet();
        
        for (State cur = state; cur != null; cur = cur.getSuperstate()) {
            e.addAll(cur.getEvents(false));
        }

        return e;
    }

    public Set getInputActions(StateMachine sm, Class returnType) throws InterpreterException {
        Set in = new HashSet();
        
        for (Iterator i = sm.getAllTransition().iterator(); i.hasNext(); ) {
            Transition t = (Transition)i.next();
            String g = Infix.infix(t.getGuard().getAST());
            
            if (!(g.equals("false") || g.equals("true") || g.equals("else"))) {
                for (Iterator j = ActionCollector.collectIdentNodes(t.getGuard().getAST()).iterator(); j.hasNext(); ) {
                    IdentNode id = (IdentNode)j.next();
                    
                    if (id.getReturnType().equals(returnType)) {
                        in.add(id.getAction());
                    }
                }
            }
        }
        
        return in;
    }

    /**
     * Returns list of named non state 
     * machine class elements. Checks that 
     * every class element object enters to the list only once.  
     * @param model class element container
     * @return list of non state machine class elements
     */
    public List getNonSMClassElements(Model model) {
    	Set addedClassElements = new HashSet();
    	List nonSMClassElements = new ArrayList(); 
    	List coHandlers = model.getControlledObjectHandlers();
    	for (Iterator i = coHandlers.iterator(); i.hasNext();) {
			ControlledObjectHandler coHandler = (ControlledObjectHandler) i.next();
			if (StringUtils.isNotBlank(coHandler.getName()) && 
					! addedClassElements.contains(coHandler.getName())) {
				nonSMClassElements.add(coHandler);
				addedClassElements.add(coHandler.getName());
			}
		}
    	List epHandlers = model.getEventProviderHandlers();
    	for (Iterator i = epHandlers.iterator(); i.hasNext();) {
			EventProviderHandler epHandler = (EventProviderHandler) i.next();
			if (StringUtils.isNotBlank(epHandler.getName()) && 
					! addedClassElements.contains(epHandler.getName())) {
				nonSMClassElements.add(epHandler);
				addedClassElements.add(epHandler.getName());
			}
		}    	
    	return nonSMClassElements;
    }

    /**
     * Tests if given class name has name 
     * @param classElement class element to test
     * @return true if given {@link ClassElement} is not null and has
     * non blank name. 
     */
    public boolean hasName(ClassElement classElement) {
    	return classElement != null && StringUtils.isNotBlank(classElement.getName());
    }
}