/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime.interpretation;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.evelopers.common.exception.CommonException;
import com.evelopers.unimod.core.stateworks.ControlledObjectHandler;
import com.evelopers.unimod.core.stateworks.EventProviderHandler;
import com.evelopers.unimod.core.stateworks.Model;
import com.evelopers.unimod.runtime.ControlledObject;
import com.evelopers.unimod.runtime.ControlledObjectsManager;
import com.evelopers.unimod.runtime.EventProcessorException;
import com.evelopers.unimod.runtime.EventProvider;
import com.evelopers.unimod.runtime.EventProvidersManager;
import com.evelopers.unimod.runtime.ModelEngine;

/**
 * @author Maxim Mazin
 */
public class InterpretationObjectsManager  {
    private Model model;
    private Map objects;
    private ControlledObjectsManager controlledObjectsManager = new ControlledObjectsManagerImpl();
    private EventProvidersManager eventProvidersManager = new EventProvidersManagerImpl();

    public InterpretationObjectsManager(Model model) {
        this.model = model;
        this.objects = new HashMap();
    }

    public ControlledObjectsManager getControlledObjectsManager() {
        return controlledObjectsManager;
    }

    public EventProvidersManager getEventProvidersManager() {
        return eventProvidersManager;
    }

    private class ControlledObjectsManagerImpl implements ControlledObjectsManager {
        public void init(ModelEngine engine) throws CommonException {}

        public void dispose() {}

        public ControlledObject getControlledObject(String coName) {
            ControlledObjectHandler handler = model.getControlledObjectHandler(coName);
            if (handler == null) {
                throw new IllegalArgumentException("Controlled object with name [" + coName + "] wasn't found");
            }
            ControlledObject controlledObject = (ControlledObject) objects.get(handler.getName());
            if (controlledObject == null) {
                try {
                    controlledObject = InstantiateHelper.getControlledObject(handler.getImplName());
                    objects.put(handler.getName(), controlledObject);
                } catch (EventProcessorException e) {
                    e.printStackTrace();
                }
            }
            return controlledObject;
        }
    }

    private class EventProvidersManagerImpl implements EventProvidersManager {
        private Map nonameEventProviders = new HashMap();

        public void init(ModelEngine engine) throws CommonException {
            for (Iterator i = model.getEventProviderHandlers().iterator(); i.hasNext();) {
                EventProviderHandler handler = (EventProviderHandler) i.next();
                String name = handler.getName();
                EventProvider eventProvider;
                if (name != null) {
                    eventProvider = getEventProvider(name);
                } else {
                    eventProvider = InstantiateHelper.getEventProvider(handler.getImplName());
                    nonameEventProviders.put(handler.getImplName(), eventProvider);
                }
                eventProvider.init(engine);
            }
        }

        public void dispose() {
            for (Iterator i = model.getEventProviderHandlers().iterator(); i.hasNext();) {
                EventProviderHandler handler = (EventProviderHandler) i.next();
                String name = handler.getName();
                EventProvider eventProvider;
                if (name != null) {
                    eventProvider = getEventProvider(name);
                } else {
                    eventProvider = (EventProvider) nonameEventProviders.get(handler.getImplName());
                }
                eventProvider.dispose();
            }
        }

        public EventProvider getEventProvider(String epName) {
            EventProviderHandler handler = model.getEventProviderHandler(epName);
            if (handler == null) {
                throw new IllegalArgumentException("Event provider with name [" + epName + "] wasn't found");
            }
            EventProvider eventProvider = (EventProvider) objects.get(handler.getName());
            if (eventProvider == null) {
                try {
                    eventProvider = InstantiateHelper.getEventProvider(handler.getImplName());
                    objects.put(handler.getName(), eventProvider);
                } catch (EventProcessorException e) {
                    e.printStackTrace();
                }
            }
            return eventProvider;
        }
    }

}