/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime.interpretation;

import java.util.Iterator;
import java.util.List;

import com.evelopers.common.exception.SystemException;
import com.evelopers.unimod.contract.CoreContract;
import com.evelopers.unimod.core.stateworks.Action;
import com.evelopers.unimod.core.stateworks.ControlledObjectHandler;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Guard;
import com.evelopers.unimod.core.stateworks.Model;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.Transition;
import com.evelopers.unimod.parser.InterpreterException;
import com.evelopers.unimod.runtime.AbstractEventProcessor;
import com.evelopers.unimod.runtime.EventProcessorException;
import com.evelopers.unimod.runtime.ModelStructure;
import com.evelopers.unimod.runtime.StateMachineConfig;
import com.evelopers.unimod.runtime.StateMachinePath;
import com.evelopers.unimod.runtime.context.StateMachineContext;

/**
 * <p/>Abstract low-level event processor implementation.
 * </p>
 * <p/>Concrete implementation responsible for realization of state machine
 * runtime sematic such as selection of transition for event, processing input
 * and output actions and so on.
 * </p>
 * <p/>Pattern used: <b>Template Method </b>
 * </p>
 * 
 * @author Vadim Gurov
 * @version $Revision: 3$
 */
public class InterpretationEventProcessor extends AbstractEventProcessor {
    /**
     * In-memory model
     */
    private InterpretationModelStructure modelStructure;

    /**
     * Lazy initialized field. Use {@link #getActionExecutor()}to get instance.
     */
    private transient ActionExecutor actionExecutor;

    /** @link dependency */
    /* # ActionExecutor lnkActionExecutor; */

    /**
     * Creates processor
     * 
     * @param model
     *            automata model
     */
    InterpretationEventProcessor(Model model) {
        modelStructure = new InterpretationModelStructure(model);
    }

    public ModelStructure getModelStructure() {
        return modelStructure;
    }
    
    /**
     * Do not call this method during event processing.
     * 
     * @param model
     */
    public void updateModelStructure(Model model) {
    	modelStructure.setModel(model);
    }

    protected StateMachineConfig transiteToStableState(
            StateMachineContext context, StateMachinePath path,
            StateMachineConfig config) throws SystemException {
        // Get source state
        StateMachine stateMachine = modelStructure.getStateMachine(path.getStateMachine());
        State state = config2state(stateMachine, config);

        /* While the state is composite */
        while (state.isComposite()) {
            /* Composite Target State */
            fireCompositeTargetState(context, path, config.toString());

            /* Get initial state from composite state */
            state = state.getInitialSubstate();
            config = state2config(state);

            /* Come To State */
            fireComeToState(context, path, config.toString());

            /* Get transition from current initial state */
            Transition transition = getTransitionFromInitial(state);
            
            /* Transition Found */
            fireTransitionFound(context, path, config.toString(), Event.NO_EVENT, transition2string(transition));

            /* Transite */
            state = transite(context, path, state, transition);
            config = state2config(state);
        }
        
        return config;
    }

    protected StateMachineConfig process(Event event,
            StateMachineContext context, StateMachinePath path,
            StateMachineConfig config) throws SystemException {

        // Prepare input actions pool
        InterpretationInputActions inputActions = new InterpretationInputActions(
                getActionExecutor(), path);

        // Find source state
        StateMachine stateMachine = modelStructure.getStateMachine(path
                .getStateMachine());
        State state = config2state(stateMachine, config);

        /* Look for transition */
        Transition transition = lookForTransition(event, state, path, context,
                inputActions);

        /* If transition found */
        if (transitionFound(transition)) {
            /* Transite */
            state = transite(context, path, state, transition);
            config = state2config(state);

            /* Transite to stable state */
            config = transiteToStableState(context, path, config);
            state = config2state(stateMachine, config);
        }

        /* Submachines execution */
        executeSubmachines(event, state, path, context);

        return config;
    }

    private Transition lookForTransition(Event event, State state,
            StateMachinePath path, StateMachineContext context,
            InterpretationInputActions inputActions)
            throws EventProcessorException, InterpreterException {

        /* If event isn't handled in the state or its superstate */
        if (!handlesEvent(state, event)) {
            /* Set any event as current event */
            event = Event.ANY;
            /* If ANY event is handled in the state ir its superstate */
            if (handlesEvent(state, Event.ANY)) {

            } else {
                /* Event skipped */
                fireEventSkipped(context, path, state2string(state), event);
                return null;
            }
        }

        /* Set given state as current state */
        State currentState = state;

        while (true) {
            // Get outgoing transitions from the current state
            List outgoingTransitions = currentState
                    .getFilteredOutgoingTransitions(event, false);
            outgoingTransitions.addAll(currentState
                    .getFilteredOutgoingTransitions(event, true));

            for (Iterator i = outgoingTransitions.iterator(); i.hasNext();) {
                /* Get next transition from current state on current event */
                Transition transition = (Transition) i.next();

                /* Transition Candidate */
                fireTransitionCandidate(context, path,
                        state2string(currentState), event,
                        transition2string(transition));

                /* If guard condition is met */
                if (guardConditionIsMet(transition, context, inputActions)) {
                    fireTransitionFound(context, path,
                            state2string(currentState), event,
                            transition2string(transition));
                    return transition;
                }
            }

            /* If has superstate */
            if (currentState.getSuperstate() != null) {
                /* Set superstate as current state */
                currentState = currentState.getSuperstate();

                /* Transitions Of Superstate */
                fireTransitionsOfSuperstate(context, path,
                        state2string(currentState), event);
            } else {
                /* Transition Not Found */
                fireTransitionNotFound(context, path, state2string(state),
                        event);
                return null;
            }
        }
    }

    private State transite(
            StateMachineContext context, StateMachinePath path,
            State sourceState, Transition transition)
            throws EventProcessorException, InterpreterException {

        /* Execute output actions on the transition */
        executeActions(path, transition, context);

        /* Get target state */
        State targetState = transition.getTargetState();

        /* Come To State */
        fireComeToState(context, path, state2string(targetState));

        /* Execute on-enter actions in the state */
        executeActions(path, targetState, context);

        return targetState;
    }

    private boolean guardConditionIsMet(Transition transition,
            StateMachineContext context, InterpretationInputActions inputActions)
            throws InterpreterException {
        Guard guard = transition.getGuard();
        inputActions.setTransition(transition2string(transition));
        boolean guardConditionIsMet = GuardInterpreter.interpret(guard,
                context, inputActions);
        return guardConditionIsMet;
    }

    private boolean handlesEvent(State state, Event event) {
        for (State cur = state; cur != null; cur = cur.getSuperstate()) {
            List transitions = cur.getFilteredOutgoingTransitions(event,
                    false);
            transitions.addAll(state
                    .getFilteredOutgoingTransitions(event, true));
            if (!transitions.isEmpty()) {
                return true;
            }
        }
        return false;
    }

    private boolean transitionFound(Transition transition) {
        return transition != null;
    }

    private Transition getTransitionFromInitial(State initialState) {
        List transitions = initialState.getOutgoingTransitions();
        // Initial state should have exectly one outgoing transition
        return (Transition) transitions.get(0);
    }

    /**
     * Executes all output actions on given transition
     * 
     * @param transition
     *            transition to get output actions from
     * @param context
     *            state machine context
     * @throws EventProcessorException
     */
    private void executeActions(StateMachinePath path,
            Transition transition, StateMachineContext context)
            throws EventProcessorException {

        /* While has more actions */
        for (Iterator i = transition.getOutputActions().iterator(); i.hasNext();) {
            /* Get next output action on the transition */
            Action action = (Action) i.next();

            /* Before Output Action Execution */
            fireBeforeOutputActionExecution(context, path, transition2string(transition), action2string(action));

            /* Execute output action */
            executeOutputAction(action, context);

            /* After Output Action Execution */
            fireAfterOutputActionExecution(context, path, transition2string(transition), action2string(action));
        }
    }

    /**
     * Executes on-enter output actions on given state
     * 
     * @param state
     *            state to execute actions from
     * @param context
     *            state machine context
     * @throws EventProcessorException
     *             if some action throws exception
     */
    private void executeActions(StateMachinePath path,
            State state, StateMachineContext context)
            throws EventProcessorException {

        /* While has more actions */
        for (Iterator i = state.getOnEnterActions().iterator(); i.hasNext();) {
            /* Get next on-enter action */
            Action action = (Action) i.next();

            /* Before On Enter Action Execution */
            fireBeforeOnEnterActionExecution(context, path, state2string(state), action2string(action));

            /* Execute on-enter action */
            executeOutputAction(action, context);

            /* After On Enter Action Execution */
            fireAfterOnEnterActionExecution(context, path, state2string(state), action2string(action));
        }
    }

    /**
     * Executes state machines included to the given state and its ancestors
     * 
     * @param event
     *            event to process by submachines
     * @param context
     *            state machine context of the event
     * @param path
     *            path to the state machine instance
     * @param config
     *            state machine config determining current active state
     * @param state
     *            state whose submachines are to be executed
     */
    private void executeSubmachines(Event event, State state,
            StateMachinePath path, StateMachineContext context) {
        /* Set target state as current state */
        State currentState = state;
        
        while (true) {
            /* While current state has more submachines */
            for (Iterator i = currentState.getSubmachines().iterator(); i.hasNext();) {
                /* Get next submachine */
                StateMachine subMachine = (StateMachine) i.next();
                
                // Create subpath
                StateMachinePath subPath = new StateMachinePath(path,
                        CoreContract.encode(currentState), CoreContract.encode(subMachine));

                /* Before Submachine Execution */
                fireBeforeSubmachineExecution(context, event, path, state2string(currentState), CoreContract.encode(subMachine));
                
                /* Process event with state machine */
                process(event, context, subPath);
                
                /* After Submachine Execution */
                fireAfterSubmachineExecution(context, event, path, state2string(currentState), CoreContract.encode(subMachine));
            }
            
            /* If has superstate */
            if (currentState.getSuperstate() != null) {
                /* Set superstate as current state */
                currentState = currentState.getSuperstate();                
            } else {
                return;
            }
        }
    }

    private void executeOutputAction(Action action, StateMachineContext context) throws EventProcessorException {
        if (action.getObject() instanceof ControlledObjectHandler) {
            getActionExecutor().execute(action, context);
        } else if (action.getObject() instanceof StateMachine) {
            // send event to state machine
            // create path as root path
            StateMachinePath subPath = new StateMachinePath(CoreContract
                    .encode((StateMachine) action.getObject()));
            process(new Event(action.getActionName()), context, subPath);
        }
    }

    /**
     * Lazy initialization
     * 
     * @return action executor instance intialized with
     *         {@link #getControlledObjectsMap()}
     */
    private ActionExecutor getActionExecutor() {
        if (actionExecutor == null) {
            actionExecutor = new ActionExecutor(getControlledObjectsMap());
        }
        return actionExecutor;
    }

    private static StateMachineConfig state2config(State activeState) {
        return new StateMachineConfig(state2string(activeState));
    }

    private static State config2state(StateMachine machine,
            StateMachineConfig config) {
        return CoreContract.decodeState(machine, config.getActiveState());
    }

    private static String state2string(State state) {
        return CoreContract.encode(state);
    }

    private static String transition2string(Transition transition) {
        return CoreContract.encode(transition);
    }

    private static String action2string(Action action) {
        return CoreContract.encode(action);
    }

    private class InterpretationInputActions extends CalculatedInputActions {
        private StateMachinePath path;

        private String transition;

        public InterpretationInputActions(ActionExecutor actionExecutor,
                StateMachinePath path) {
            super(actionExecutor);
            this.path = path;
        }

        public String getTransition() {
            return transition;
        }

        public void setTransition(String transition) {
            this.transition = transition;
        }

        protected void fireBefore(StateMachineContext context, Action action) {
            fireBeforeInputActionExecution(context, path, transition,
                    CoreContract.encode(action));
        }

        protected void fireAfter(StateMachineContext context, Action action,
                Object result) {
            fireAfterInputActionExecution(context, path, transition,
                    CoreContract.encode(action), result);
        }
    }
}