/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime.interpretation;

import com.evelopers.common.exception.SystemException;
import com.evelopers.common.util.helper.ClassHelper;
import com.evelopers.unimod.core.stateworks.Action;
import com.evelopers.unimod.parser.InterpreterException;
import com.evelopers.unimod.resources.Messages;
import com.evelopers.unimod.runtime.ControlledObject;
import com.evelopers.unimod.runtime.ControlledObjectsMap;
import com.evelopers.unimod.runtime.EventProcessorException;
import com.evelopers.unimod.runtime.context.StateMachineContext;

/**
 * Knows how to excecute given Action against given context.
 * Supports two types of Action return type - int and boolean.
 * 
 * @author Vadim Gurov
 * @version $Revision: 2$
 */
class ActionExecutor {
    private ControlledObjectsMap controlledObjectsMap;

    public ActionExecutor(ControlledObjectsMap controlledObjectsMap) {
        this.controlledObjectsMap = controlledObjectsMap;
    }

    /**
     * This executes given action.
     * <p/>
     * {@link com.evelopers.unimod.core.stateworks.Action#getControlledObjectHandler()
     * Controlled object handler} of the action is used only to find controlled object's
     * name. To get real controlled object instance this method uses {@link ControlledObjectsMap}
     * passed to constructor.
     * @param action action to execute
     * @param context action context
     * @return return-value of the action
     * @throws SystemException if action couldn't be executed or have thrown
     * exception
     */
	protected Object _execute(Action action, StateMachineContext context) throws SystemException {
        String coName = action.getObject().getName();
        ControlledObject controlledObject = controlledObjectsMap.getControlledObject(coName);
        return
				ClassHelper.invoke(
					controlledObject,
					action.getActionName(),
					new Class[] { StateMachineContext.class },
					new Object[] { context });
	}

	/**
	 * Executes action against given context, ignoring return type.
	 *
	 * @param action action to execute
	 * @param context state machine runtime context
	 * @throws com.evelopers.unimod.runtime.EventProcessorException if error while executing occurs
	 */
	public void execute(Action action, StateMachineContext context)
		throws EventProcessorException {
		
		try {
			_execute(action, context);
		} catch (SystemException e) {
			throw new EventProcessorException(e, "Error while executing " + action);
		}
	}

	/**
	 * Executes action against given context, treating return type as int.
	 * 
	 * @param action action to execute
	 * @param context state machine runtime context
	 * @return usually 0 as false and 1 as true. Returned values of
	 * input actions only will be taken into account by runtime interpretter.
	 * @throws InterpreterException if error while executing occurs or return type
	 * is not int.
	 */
	public int executeNum(Action action, StateMachineContext context)
		throws InterpreterException {
		
		Object result;
		
		try {
			result = _execute(action, context);
		} catch (SystemException e) {
			throw new InterpreterException(e, "Error while executing " + action);
		}

		if (!(result instanceof Integer)) {
			throw new InterpreterException(
				Messages.getMessages().getMessage(
					Messages.RETURN_MUSTBE_INT,
					new Object[] {
						action
							.getObject()
							.getImplName(),
						action.getActionName()}));
		}

		return ((Integer) result).intValue();
	}

	/**
	 * Executes action against given context, treating return type as boolean.
	 * 
	 * @param action action to execute
	 * @param context state machine runtime context
	 * @return boolean
	 * @throws InterpreterException if error while executing occurs or return type
	 * is not boolean.
	 */
	public boolean executeBool(Action action, StateMachineContext context) throws InterpreterException {
		Object result;
		
		try {
			result = _execute(action, context);
		} catch (SystemException e) {
			throw new InterpreterException(e, "Error while executing " + action);
		}


		if (!(result instanceof Boolean)) {
			throw new InterpreterException(
				Messages.getMessages().getMessage(
					Messages.RETURN_MUSTBE_BOOL,
					new Object[] {
						action
							.getObject()
							.getImplName(),
						action.getActionName()}));
		}

		return ((Boolean) result).booleanValue();
	}

    /** @link dependency */
    /*# Action lnkAction; */
}