/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime;

import java.io.Serializable;

/**
 * Path to state machine. Used to identify {@link StateMachineConfig} between
 * {@link AbstractEventProcessor} and {@link StateMachineConfigManager}.
 * @author Maxim Mazin
 */
public class StateMachinePath implements Serializable {
    /**
     * Path to super state machine
     */
    private final StateMachinePath superConfigPath;
    /**
     * State of super state machine that contains pointed state machine
     */
    private final String includingState;
    /**
     * Name of pointed state machine
     */
    private final String stateMachine;

    /**
     * Creates path leading to root state machine.
     * @param stateMachineName name of state machine pointed by this path
     */
    public StateMachinePath(String stateMachine) {
        this.superConfigPath = null;
        this.includingState = null;
        this.stateMachine = stateMachine;
    }

    /**
     * Creates path leading to superMachine's submachine in
     * superMachineActiveState.
     * @param superConfigPath path to superMachine
     * @param includingState state of super machine containing pointed state machine
     * @param stateMachineName name of state machine pointed by this path
     */
    public StateMachinePath(StateMachinePath superConfigPath, String includingState, String stateMachine) {
        this.superConfigPath = superConfigPath;
        this.includingState = includingState;
        this.stateMachine = stateMachine;
    }

    /**
     * Returns path to super state machine
     * @return path to super state machine
     */
    public StateMachinePath getSuperStateMachinePath() {
        return superConfigPath;
    }

    /**
     * Returns state of super state machine that includes pointed state machine
     * @return state of super state machine that includes pointed state machine
     */
    public String getIncludingState() {
        return includingState;
    }

    /**
     * Returns name of pointed state machine
     * @return name of pointed state machine
     */
    public String getStateMachine() {
        return stateMachine;
    }

    /**
     * Returns whether this path leads to root state machine.
     * @return whether this path leads to root state machine
     */
    public boolean isRoot() {
        return superConfigPath == null;
    }

    /**
     * Symmetric to {@link #parse(String)} method
     */
    public String toString() {
        StringBuffer buffer = new StringBuffer();

        if (superConfigPath != null) {
            buffer.append(superConfigPath).
                    append(':').
                    append(includingState);
        }

        buffer.append('/').append(stateMachine);

        return buffer.toString();
    }

   public boolean equals(Object obj) {
        if (!(obj instanceof StateMachinePath)) {
            return false;
        }

        StateMachinePath thatConfigPath = (StateMachinePath) obj;
        if (!isRoot() && 
                !(this.getSuperStateMachinePath().equals(thatConfigPath.getSuperStateMachinePath()) &&
                  this.getIncludingState().equals(thatConfigPath.getIncludingState()))) {
            return false;
        }

        return this.stateMachine.equals(thatConfigPath.stateMachine);
    }

    public int hashCode() {
        int hash = stateMachine.hashCode();

        if (! isRoot()) {
            hash ^= includingState.hashCode();
            hash ^= superConfigPath.hashCode();
        }

        return hash;
    }
}
