/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime;

import com.evelopers.unimod.runtime.context.StateMachineContext;

/**
 * Manages life cycle of {@link StateMachineConfig}. Every state machine is
 * associated with exactly one state machine config manager. So different
 * instances of state machine in model (e.g. one submachine included in
 * different states) are managed by one state machine config manager. To
 * identify different instances of the same machine {@link StateMachinePath} is
 * used.
 *
 * @author Maxim Mazin
 */
public interface StateMachineConfigManager {

    /**
     * Initializes config store.
     * @param path path to state machine instance
     * @param initialConfig initial config of the state machine
     */
    void init(StateMachinePath path, StateMachineContext context, StateMachineConfig initialConfig);

    /**
     * Loads config for given path from some store.
     * The type of store is defined by StateMachineConfigManager
     * implementation.
     * <p/>
     * If there is no stored config for given
     * path then this method must return <code>null</code>. It will
     * lead to {@link #init} method invocation.
     * @param path path to state machine config to load
     * @param context state machine context that has come with event
     * @return loaded state machine config
     */
    StateMachineConfig load(StateMachinePath path, StateMachineContext context);

    /**
     * Stores given config against given path.
     *
     * @param path path to state machine config to store
     * @param context state machine context that has come with event
     * @param config state machine config to store
     */
    void store(StateMachinePath path, StateMachineContext context, StateMachineConfig config);

    /**
     * Disposes config store.
     * <p/>
     * {@link AbstractEventProcessor} will call this method when state machine
     * comes to final state.
     * @param path path to state machine instance
     * @param finalConfig current final config of the state machine
     */
    void dispose(StateMachinePath path, StateMachineContext context, StateMachineConfig finalConfig);

    /**
     * @directed
     * @supplierCardinality *
     * @supplierQualifier path:ConfigPath 
     */
    /*# StateMachineConfig lnkStateMachineConfig; */

    /** @link dependency */
    /*# ConfigPath lnkConfigPath; */

    /** @link dependency */
    /*# StateMachineContext lnkStateMachineContext; */
}
