/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime;

import com.evelopers.common.concurrent.AbstractQueue;
import com.evelopers.common.concurrent.QueueMessageProcessor;
import com.evelopers.common.util.helper.StringHelper;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.runtime.context.StateMachineContext;

/**
 * Event handler with queue of event support.
 * Uses additional daemon thread.
 * @author Vadim Gurov
 * @author Maxim Mazin
 */
public class QueuedHandler implements EventManager {
    private AbstractQueue queue;

    public void init(ModelEngine engine) {
        queue = new AbstractQueue(new QueueMessageProcessorImpl(engine));
    }

    public void handle(Event e, StateMachineContext context) {
        if (queue == null) {
            throw new IllegalStateException("QueuedHandler.handle method called before QueuedHandler.init method or after QueuedHandler.dispose method");
        }

        if (context != null && e != null) {
            e.addParametersToContext(context.getEventContext());
        }

        queue.push(new QueuedEvent(e, context));
    }
    
    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.EventManager#handleAndWait(com.evelopers.unimod.core.stateworks.Event, com.evelopers.unimod.runtime.context.StateMachineContext)
     */
    public void handleAndWait(Event event, StateMachineContext context) {
        synchronized(event) {
            handle(event, context);
            try {
                event.wait();
            } catch (InterruptedException e) {
                // should never be thrown
                e.printStackTrace();
                throw new RuntimeException("Unexpected InterruptedException inside QueuedHandler.handleAndWait(): " + StringHelper.stackTraceToString(e));
            }
        }
    }

    public void dispose() {
        queue.stop();
        queue = null;
    }

    private static class QueuedEvent {
        private final Event e;
        private final StateMachineContext context;

        QueuedEvent(Event e, StateMachineContext context) {
            this.e = e;
            this.context = context;
        }

        Event getEvent() {
            return e;
        }

        StateMachineContext getContext() {
            return context;
        }
    }

    private static class QueueMessageProcessorImpl implements QueueMessageProcessor {
        private final ModelEngine engine;

        public QueueMessageProcessorImpl(ModelEngine engine) {
            this.engine = engine;
        }

        public void processMessage(Object msg) {
            QueuedEvent qe = (QueuedEvent) msg;
            try {
                engine.process(qe.getEvent(), qe.getContext());
            } finally {
                synchronized(qe.getEvent()) {
                    qe.getEvent().notifyAll();
                }
            }
        }
    }
}
