/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.debug.debugger;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;

import com.evelopers.common.exception.CommonException;
import com.evelopers.common.exception.SystemException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Model;
import com.evelopers.unimod.debug.Params;
import com.evelopers.unimod.debug.protocol.CommandMessage;
import com.evelopers.unimod.debug.protocol.EventMessage;
import com.evelopers.unimod.debug.protocol.Message;
import com.evelopers.unimod.debug.protocol.MessageCoder;
import com.evelopers.unimod.debug.protocol.MessageCoderException;
import com.evelopers.unimod.debug.protocol.ThreadInfo;
import com.evelopers.unimod.debug.protocol.position.Position;
import com.evelopers.unimod.runtime.ControlledObject;
import com.evelopers.unimod.runtime.EventManager;
import com.evelopers.unimod.runtime.EventProvider;
import com.evelopers.unimod.runtime.ModelEngine;
import com.evelopers.unimod.runtime.context.Parameter;
import com.evelopers.unimod.runtime.context.StateMachineContext;
import com.evelopers.unimod.runtime.context.StateMachineContextImpl;

/**
 * @author vgurov
 */
public class DebuggerConnector implements EventProvider, ControlledObject {

    /**
     * @unimod.event.descr suspended on breakpoint
     */
    public static final String E1 = "e1";

    /**
     * @unimod.event.descr suspended on step
     */
    public static final String E2 = "e2";

    /**
     * @unimod.event.descr resumed
     */
    public static final String E3 = "e3";

    /**
     * @unimod.event.descr app has closed connection
     */
    public static final String E4 = "e4";

    /**
     * @unimod.event.descr message format exception
     */
    public static final String E5 = "e5";

    /**
     * @unimod.event.descr app has received unknown command
     */
    public static final String E6 = "e6";

	/**
	* @unimod.event.descr new thread created
	*/
	public static final String E7 = "e7";

	/**
	* @unimod.event.descr can't update model in debugged engine
	*/
	public static final String E8 = "e8";
	
    private String hostName;

    private int port;

    private MessageCoder coder;

    private Socket socket;

    private OutputStream socketWriter;

    private InputStream socketReader;

    private EventManager handler;

    public DebuggerConnector(String hostName, int port, MessageCoder coder) {
        this.hostName = hostName;
        this.port = port;
        this.coder = coder;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.evelopers.unimod.runtime.EventProvider#init(com.evelopers.unimod.runtime.ModelEngine)
     */
    public void init(ModelEngine engine) throws CommonException {
        this.handler = engine.getEventManager();

        try {
            socket = new Socket(hostName, port);
            socketWriter = socket.getOutputStream();
            socketReader = socket.getInputStream();
        } catch (Exception e) {
            if (socket != null) {
                try {
                    socket.close();
                } catch (Exception e1) {
                };
            }
            throw new SystemException(e);
        }

        new SocketListenerThread().start();
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.evelopers.unimod.runtime.EventProvider#dispose()
     */
    public void dispose() {
        try {
            socketReader = null;
            socket.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * @unimod.action.descr send set breakpoints
     */
    public void z1(StateMachineContext context) throws MessageCoderException,
            IOException {
        Position[] b = (Position[]) context.getEventContext().getParameter(
                Params.Event.BREAKPOINTS);

        sendMessage(CommandMessage.createSetBreakpoints(b));
    }

    /**
     * @unimod.action.descr send remove breakpoints
     */
    public void z2(StateMachineContext context) throws MessageCoderException,
            IOException {
        Position[] b = (Position[]) context.getEventContext().getParameter(
                Params.Event.BREAKPOINTS);

        sendMessage(CommandMessage.createRemoveBreakpoints(b));
    }

    /**
     * @unimod.action.descr send make step
     */
    public void z3(StateMachineContext context) throws MessageCoderException,
            IOException {
        ThreadInfo ti = (ThreadInfo) context.getEventContext().getParameter(
                Params.Event.THREAD_INFO);

        sendMessage(CommandMessage.createStep(ti));
    }

    /**
     * @unimod.action.descr send resume
     */
    public void z4(StateMachineContext context) throws MessageCoderException,
            IOException {
        ThreadInfo ti = (ThreadInfo) context.getEventContext().getParameter(
                Params.Event.THREAD_INFO);

        sendMessage(CommandMessage.createResume(ti));
    }

    /**
     * @unimod.action.descr send update model message
     */
    public void z5(StateMachineContext context) throws MessageCoderException, IOException {
        Model newModel = (Model) context.getEventContext().getParameter(Params.Event.NEW_MODEL);

        sendMessage(CommandMessage.createUploadNewModel(newModel));
    }
    
    private void sendMessage(Message m) throws MessageCoderException,
            IOException {
        coder.encode(m, socketWriter);
        socketWriter.flush();
    }

    private class SocketListenerThread extends Thread {

        SocketListenerThread() {
            super("Debugger.SocketListenerThread");
        }

        public void run() {
            // while connected to target engine
            while (true) {
                try {
                    EventMessage m = (EventMessage) coder.decode(socketReader);

                    switch (m.getType()) {
                    case EventMessage.SUSPENDED_ON_BREAKPOINT:
                        handler.handle(new Event(E1, new Parameter[] {
                                new Parameter(Params.Event.THREAD_INFO, m
                                        .getThreadInfo()),
                                new Parameter(Params.Event.POSITION, m
                                        .getPosition()) }),
                                StateMachineContextImpl.create());
                        break;

                    case EventMessage.SUSPENDED_ON_STEP:
                        handler.handle(new Event(E2, new Parameter[] {
                                new Parameter(Params.Event.THREAD_INFO, m
                                        .getThreadInfo()),
                                new Parameter(Params.Event.POSITION, m
                                        .getPosition()) }),
                                StateMachineContextImpl.create());
                        break;

                    case EventMessage.RESUMED:
                        handler.handle(new Event(E3, new Parameter(
                                Params.Event.THREAD_INFO, m.getThreadInfo())),
                                StateMachineContextImpl.create());
                        break;

                    case EventMessage.THREAD_CREATED:
                        handler.handle(new Event(E7, new Parameter(
                                Params.Event.THREAD_INFO, m.getThreadInfo())),
                                StateMachineContextImpl.create());
                        break;
                        
                    case EventMessage.UNKNOWN_COMMAND:
                        handler.handle(
                                new Event(E6, new Parameter(
                                        Params.Event.UNKNOWN_COMMAND_INFO, m
                                                .getInfo())),
                                StateMachineContextImpl.create());
                        break;
                        
                    case EventMessage.CANT_UPDATE_MODEL:
                        handler.handle(
                                new Event(E8, 
                                	new Parameter[]{
                                		new Parameter(Params.Event.INFO, m.getInfo()),
                                		new Parameter(Params.Event.SUSPENED_THREADS, m.getSuspendedThreads())
                                }),
                                StateMachineContextImpl.create());
                    	break;
                    }
                } catch (MessageCoderException e) {
                    if (e.getParentException() instanceof IOException) {
                        // communication problem, or client closes connection or
                        // server shutdown

                        // check, if socket is still opened and close it
                        if (socket != null) {
                            try {
                                socket.close();
                            } catch (Exception e1) {
                            }
                            ;
                        }

                        // socketReader is used as flag, if it's null - server
                        // shutdown, if not null - client has closed connection
                        if (socketReader != null) {
                            // client has closed connection
                            handler.handle(new Event(E4),
                                    StateMachineContextImpl.create());
                        }

                        return;
                    } else {
                        // not a communication problem
                        handler.handle(new Event(E5, new Parameter(
                                Params.Event.MESSAGE_CODER_EXCEPTION, e)),
                                StateMachineContextImpl.create());
                    }

                }
            }
        }
    }

}