/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */

package com.evelopers.unimod.debug;

import com.evelopers.unimod.contract.CoreContract;
import com.evelopers.unimod.core.stateworks.Action;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.Transition;
import com.evelopers.unimod.debug.protocol.position.ComeToStatePosition;
import com.evelopers.unimod.debug.protocol.position.EventProcessingPosition;
import com.evelopers.unimod.debug.protocol.position.InputActionExecutionPosition;
import com.evelopers.unimod.debug.protocol.position.OnEnterActionExecutionPosition;
import com.evelopers.unimod.debug.protocol.position.OutputActionExecutionPosition;
import com.evelopers.unimod.debug.protocol.position.Position;
import com.evelopers.unimod.debug.protocol.position.SubmachinesExecutionPosition;
import com.evelopers.unimod.debug.protocol.position.TransitionCandidatePosition;
import com.evelopers.unimod.runtime.StateMachineConfig;
import com.evelopers.unimod.runtime.StateMachinePath;

/**
 * For any model element returns array of positions, which may be used
 * as breakpoint. It's guaranteed that debugger stops only in these positions.
 * 
 * @author vgurov
 */
public class BreakpointPositions {

    private BreakpointPositions() {}
    
    /**
     * Returns positions that must be used as breakpoints to stop in given state, before
     * execution of on-enter action
     * 
     * @param s
     * @return
     */
    public static Position[] getPositions(StateMachine sm, State s) {
        return 
        	new Position[] {
                new EventProcessingPosition(null, getSMP(sm), null, new StateMachineConfig(CoreContract.encode(s)), true),
                new ComeToStatePosition(null, getSMP(sm), CoreContract.encode(s))
            };
    }

    /**
     * Returns positions that must be used as breakpoints to stop on given transition, before
     * calculating guard condition
     * 
     * @param t
     * @return
     */
    public static Position[] getPositions(StateMachine sm, Transition t) {
        return new Position[] {
                new TransitionCandidatePosition(null, getSMP(sm), null, CoreContract.encode(t.getSourceState()), CoreContract.encode(t))
            };
    }
    
    /**
     * Returns positions that must be used as breakpoints to stop before execution of 
     * given action in given state
     * 
     * @param s
     * @param a
     * @return
     */
    public static Position[] getPositions(StateMachine sm, State s, Action a) {
        return new Position[] {
                new OnEnterActionExecutionPosition(null, getSMP(sm), CoreContract.encode(s), CoreContract.encode(a), true)
        	};
    }

    public static Position[] getPositions(StateMachine sm, State s, StateMachine subMachine) {
        return new Position[] {
                new SubmachinesExecutionPosition(null, getSMP(sm), null, CoreContract.encode(s), CoreContract.encode(subMachine), true)
        	};
    }
    
    /**
     * Returns positions that must be used as breakpoints to stop before execution
     * of input or output action on given transition
     * 
     * @param t
     * @param a
     * @param output
     * @return
     */
    public static Position[] getPositions(StateMachine sm, Transition t, Action a, boolean output) {
        return 
        	output ? 
            new Position[] {
                new OutputActionExecutionPosition(null, getSMP(sm), CoreContract.encode(t), CoreContract.encode(a), true)
        	} :
            new Position[] {
                new InputActionExecutionPosition(null, getSMP(sm), CoreContract.encode(t), CoreContract.encode(a), true, null)
            };
    }

    private static StateMachinePath getSMP(StateMachine sm) {
        return new StateMachinePath(CoreContract.encode(sm));
    }
}
