/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;

import java.io.Serializable;

/**
 * <p>
 * Represents possible state types.
 * </p>
 * <p>
 * Pattern used: <b>Enumeration</b>
 * </p>
 * 
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
public final class StateType implements Serializable {

    public static final String INITIAL_NAME = "INITIAL";
	public static final String FINAL_NAME = "FINAL";
	public static final String NORMAL_NAME = "NORMAL";

    /** Initial state type */
    public static final StateType INITIAL = new StateType(1, INITIAL_NAME);

    /** Final state type */
    public static final StateType FINAL = new StateType(2, FINAL_NAME);

    /** Normal state type (simple or composite) */
    public static final StateType NORMAL = new StateType(0, NORMAL_NAME);

    private String typeName;
    private int type;

	/**
	 * Provate constructor doesn't allow to outters to create this class instances 
	 * @param type
	 * @param typeName
	 */
    private StateType(int type, String typeName) {
        this.type = type;
        this.typeName = typeName;
    }

    /**
     * Compares two state types
     *
     * @param o another StateType
     * @return true if state types are equal
     */
    public boolean equals(Object o) {
        if (!(o  instanceof StateType)) {
            return false;
        }

        StateType that = (StateType)o;

        return this.type == that.type;
    }

    /**
     * Returns StateType hash code
     *
     * @return hash code
     */
    public int hashCode() {
        return type;
    }

    /**
     * Returns string representation
     *
     * @return string representation
     */
    public String toString() {
      return typeName;
    }

    /**
     * Returns name of state type
     *
     * @return state type name
     */
    public String getName() {
        return typeName;
    }

    /**
     * Parses StateType object from given string
     *
     * @param s string representation of StateType
     * @return StateType
     * @throws StateTypeFormatException if given string can't be parsed into StateType object
     */
    public static StateType parseStateType(String s) {
      if (s.equalsIgnoreCase(INITIAL_NAME)) {
          return INITIAL;
      } else if (s.equalsIgnoreCase(FINAL_NAME)) {
          return FINAL;
      } else if (s.equalsIgnoreCase(NORMAL_NAME)) {
         return NORMAL;
      }

      throw new StateTypeFormatException("Can't parse StateType from string [" + s + "]");
    }
}