/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.evelopers.unimod.runtime.context.Parameter;

/**
 * <p>
 * Class <code>StateMachine</code> represents StateMachine abstraction.
 * State machine "knows" directly only about it's top state and about all
 * controlled objects and event providers.
 * State machine must have top state.
 * </p>
 * <p>
 * Implements factory for {@link State}, {@link Transition}, {@link Guard},
 * {@link Event}, {@link Action}, {@link Association}, {@link Parameter}.
 * </p>
 *
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
public class StateMachine extends ClassElement {
    public static final String STATEMACHINE_STEREOTYPE = "statemachine";
	public static final String TOP_STATE_PROPERTY = "TOP_STATE_PROPERTY";

    /**
     * Top state.
     * @link aggregation
     * @undirected
     * @clientCardinality 1
     * @supplierCardinality 1
     * @label top
     */
    protected State topState = null;

    /** @link dependency 
     * @label creates in runtime*/
    /*# ConfigStore lnkConfigStore; */

    /**
     * Creates empty state machine.
     */
	protected StateMachine(String name) {
        super(name, STATEMACHINE_STEREOTYPE);
        setImplName(null);
    }

	/**
	 * Creates empty state machine with given config store.
	 */
	protected StateMachine(String name, String configStoreClassName) {
		this(name);
        setImplName(configStoreClassName);
	}

    /**
     * Return top state.
     *
     * @return top state
     */
    public State getTop() {
        return topState;
    }

    /**
     * Set top state.
     *
     * @param top new top state
     */
    public void setTop(State top) {
    	State oldTop = this.topState;
        this.topState = top;
        firePropertyChange(TOP_STATE_PROPERTY, oldTop, top);        
    }

    public String getConfigManagerClassName() {
        return getImplName();
    }

    public void setConfigManagerClassName(String configManagerClassName) {
        setImplName(configManagerClassName);
    }

    /**
     * <p>
     * Adds an association to state machine.
     * </p>
     * <p>
     * <strong>Note:</strong> this method should be
     * invoked only from method {@link Association#reconnect} or
     * its extensions and implementations.
     * </p>
     * @param association association to add
     * @throws IllegalArgumentException if given association
     * is <code>null</code> or its supplier is not this state
     * machine
     */
    protected void addOutgoingAssociation(Association association) {
        if (association == null || association.getSource() != this || 
            !(association.getTarget() instanceof ControlledObjectHandler || (association.getTarget() instanceof StateMachine && association.getTarget() != this))) {
            throw new IllegalArgumentException("Incorrect outgoing association");
        }

        super.addOutgoingAssociation(association);
    }
    
    protected void addIncomingAssociation(Association association) {
        if (association == null || association.getTarget() != this || 
            !(association.getSource() instanceof EventProviderHandler || (association.getSource() instanceof StateMachine && association.getSource() != this))) {
            throw new IllegalArgumentException("Incorrect incoming association");
        }
        
        super.addIncomingAssociation(association);
    }
    
    /**
     * Returns hash code based on the name of this state machine.
     * @return hash code based on the name of this state machine
     */
	public int hashCode() {
		return name == null ? 0 : name.hashCode();
	}

    /**
     * Checks if this state machine equals to the given object. Two
     * state machines are equal if they have same names.
     * @param o an object to compare this machine to
     * @return whether this state machine is equal to the given object
     */
	public boolean equals(Object o) {
		if (!(o instanceof StateMachine)) {
			return false;
		}

		return name != null && name.equals(((StateMachine)o).getName());
	}

	public String toString() {
	    return name;
	}
	
    /* Convenience methods */

    /**
     * Creates new association with the given controlled object handler
     * as a supplier and this state machine as a client. As a result
     * in the association's constructor
     * the association is added to the association list of
     * this state machine and to the association list of given
     * controlled object handler.
     *
     * @param controlledObjectHandler controlled object handler to add
     */
    public void addContolledObjectHandler(ControlledObjectHandler controlledObjectHandler) {
        createOutgoingAssociation(controlledObjectHandler);
    }
    
    
    public void addStateMachine(StateMachine sm) {
        createOutgoingAssociation(sm);
    }
    

    /**
     * Removes given controlled object handler from state machine. It means
     * that all association having given controlled object handler
     * as supplier will be removed from this state machine.
     *
     * @param controlledObjectHandler controlled object handler to remove
     */
    public void removeControlledObjectHandler(ControlledObjectHandler controlledObjectHandler) {
        List associationsToRemove = new LinkedList();
        
        for (Iterator i = outgoingAssociations.iterator(); i.hasNext();) {
            Association association = (Association) i.next();
            if (association.getTarget().equals(controlledObjectHandler)) {
                associationsToRemove.add(association);
            }
        }

        for (Iterator i = associationsToRemove.iterator(); i.hasNext();) {
            Association association = (Association) i.next();
            // This is save because this association won't be used later
            association.reconnect(null, null);
        }
    }
    
    /**
     * Returns all controlled object handlers
     *
     * @return all controlled object handlers
     */
    public List getControlledObjectHandlers() {
        List controlledObjects = new LinkedList();
        for (Iterator i = outgoingAssociations.iterator(); i.hasNext();) {
            Association association = (Association) i.next();
            
            if (association.getTarget() instanceof ControlledObjectHandler) {
                controlledObjects.add(association.getTarget());
            }
        }
        return controlledObjects;
    }

    /**
     * Returns all associated state machines
     *
     * @return all controlled object handlers
     */
    public List getStateMachines() {
        List sms = new LinkedList();
        for (Iterator i = outgoingAssociations.iterator(); i.hasNext();) {
            Association association = (Association) i.next();
            
            if (association.getTarget() instanceof StateMachine) {
                sms.add(association.getTarget());
            }
        }
        return sms;
    }
    
    public List getEventProviderHandlers() {
        List eventProviders = new LinkedList();
        for (Iterator i = incomingAssociations.iterator(); i.hasNext();) {
            Association association = (Association) i.next();
            
            if (association.getSource() instanceof EventProviderHandler) {
                eventProviders.add(association.getSource());
            }
        }
        return eventProviders;
    }
    
    /**
     * Returns controlled object handler with given name. To achieve this
     * method iterates through associations and returns supplier
     * of first association that has supplier role equal to given
     * <code>objectName</code>.
     *
     * @param objectName controlled object handler name
     * @return controlled object handler
     */
    public ControlledObjectHandler getControlledObjectHandler(String objectName) {
        if (objectName != null) {
            for (Iterator i = outgoingAssociations.iterator(); i.hasNext();) {
                Association association = (Association) i.next();
                
                if (association.getTarget() instanceof ControlledObjectHandler && 
                        objectName.equals(association.getSupplierRole())) {
                    return (ControlledObjectHandler)association.getTarget();
                }
            }
        }
        return null;
    }

    /**
     * Returns associated state machine with given supplier role name
     * 
     * @param smName
     * @return
     */
    public StateMachine getStateMachine(String smName) {
        if (smName != null) {
            for (Iterator i = outgoingAssociations.iterator(); i.hasNext();) {
                Association association = (Association) i.next();
                
                if (association.getTarget() instanceof StateMachine && 
                        smName.equals(association.getSupplierRole())) {
                    return (StateMachine)association.getTarget();
                }
            }
        }
        
        return null;
    }
    
    /**
     * Returns associated class element with given supplier role name 
     * 
     * @param name
     * @return
     */
    public ClassElement getClassElement(String name) {
        if (name != null) {
            for (Iterator i = outgoingAssociations.iterator(); i.hasNext();) {
                Association association = (Association) i.next();
                
                if (name.equals(association.getSupplierRole())) {
                    return (ClassElement)association.getTarget();
                }
            }
        }
        
        return null;
    }
    
	/**
	 * Return all state machines included by all states
	 * of this state machine as submachines
	 *
	 * @return included state machines
	 */
	public Set getSubmachines() {
		Set subMachines = new HashSet();

		for (Iterator i = getAllStates().iterator(); i.hasNext(); ) {
			State s = (State)i.next();

			subMachines.addAll(s.getSubmachines());
		}

		return subMachines;
	}

    /**
     * Return new set that contains all states
     *
     * @return all states
     */
    public List getAllStates() {
        return getAllStates(getTop());
    }

    /**
     * Return new set that are contained by the
     * given root directly (as substates) or indirectly
     * (as substates of substates etc.)
     *
     * @param root root state
     * @return all states contained by root
     */
    protected List getAllStates(State root) {

        List content = new ArrayList();

        content.add(root);

        if (root.isComposite()) {
            Iterator iSubstates = root.getSubstates().iterator();

            while (iSubstates.hasNext()) {
                content.addAll(getAllStates((State) iSubstates.next()));
            }
        }
        return content;
    }

    /**
     * Returns all transitions
     *
     * @return all transitions
     */
    public List getAllTransition() {

        List allTransitions = new ArrayList();
        List allStates = getAllStates();

        Iterator iSates = allStates.iterator();

        while (iSates.hasNext()) {
            Iterator iTrans = ((State) iSates.next()).getOutgoingTransitions().iterator();
            while (iTrans.hasNext()) {
                allTransitions.add(iTrans.next());
            }
        }
        return allTransitions;
    }

    public State findState(String name) {

        List l = getAllStates();

        for (Iterator i = l.iterator(); i.hasNext();) {
            State state = (State) i.next();

            if (state.getName().equals(name)) {
                return state;
            }
        }

        return null;
    }

    public StateMachine findSubmachine(String name) {
        if (getName().equals(name)) {
            return this;
        }

        List l = getAllStates();
        for (Iterator i = l.iterator(); i.hasNext();) {
            State state = (State) i.next();

            for (Iterator j = state.getSubmachines().iterator(); j.hasNext();) {
                StateMachine sm = (StateMachine) j.next();

                if (sm.getName().equals(name)) {
                    return sm;
                }
            }
        }

        return null;
    }

    /* Factory methods */

    /**
     * Creates state
     *
     * @param name state name
     * @param type state type
     * @return newly created state
     */
    public State createState(String name, StateType type) {
        return new State(name, type);
    }

    public State createTopState(String name) {
        return createState(name, StateType.NORMAL);
    }

    /**
     * Creates transition
     *
     * @param sourceState source state
     * @param targetState target state
     * @param guard guard condition
     * @param event trigger event
     * @return newly created transition
     */
    public Transition createTransition(State sourceState, State targetState,
                                       Guard guard, Event event) {
        return new Transition(sourceState, targetState, guard, event);
    }

    /**
     * Creates guard condition
     *
     * @param expr guard condition logical expression
     * parser will check return types and throw ParserException if it wrong.
     * @return newly created guard condition
     */
    public Guard createGuard(String expr) {
        return new Guard(expr);
    }

    public Association createOutgoingAssociation(ClassElement supplier, String supplierRole, String clientRole) {
        return new Association(this, supplierRole, supplier, clientRole);
    }

    public final Association createOutgoingAssociation(ClassElement supplier, String supplierRole) {
        return createOutgoingAssociation(supplier, supplierRole, null);
    }

    public final Association createOutgoingAssociation(ClassElement supplier) {
        return createOutgoingAssociation(supplier, null);
    }

    public Association createIncomingAssociation(ClassElement client, String supplierRole, String clientRole) {
        return new Association(client, supplierRole, this, clientRole);
    }

    public final Association createIncomingAssociation(ClassElement client, String clientRole) {
        return createIncomingAssociation(client, null, clientRole);
    }

    public final Association createIncomingAssociation(ClassElement client) {
        return createIncomingAssociation(client, null);
    }
    
    /**
     * Pool of already created actions.
     * If creation of new action is requested and it's already in pool - 
     * it will be returned.
     */
	private Map actionPool = new HashMap();
    
    /**
     * Creates Action (Input or Output).  
     *
     * @param identifier action ident in form o1.z1 or a1.e1
     * @return newly created Action
     */
    public synchronized Action createAction(String identifier) {
        Action a = (Action)actionPool.get(identifier);
        
        if (a == null) {
            a = new Action(identifier);
            actionPool.put(identifier, a);
        }
        
        return a;
    }
    
    private static final long serialVersionUID = 2280358399729680579L;

}