/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;

import com.evelopers.unimod.core.ModelElement;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Collection;

/**
 * <p>
 * Represents behaviour model. Model contains set of state machines,
 * set of controlled object handers and set of event provider handlers. Model
 * knows about root state machine. Do not put instances of this class into
 * hashCode aware containters like HashSet.
 * </p>
 * <p>
 * Implements factory for Model, {@link EventProviderHandler}, {@link ControlledObjectHandler}, {@link StateMachine}.
 * </p>
 *  
 * @author Maxim Mazin
 * @version Revision: 1
 */
public class Model extends ModelElement {
	public static final String CHILDREN_PROPERTY = "CHILDREN_PROPERTY"; 
	public static final String ROOT_STATE_MACHINE = "ROOT_STATE_MACHINE_PROPERTY"; 

	/**
	 * @associates <{StateMachine}>
	 * @link aggregation
	 * @supplierCardinality 0..* 
	 */
	protected List stateMachines = new ArrayList();

	/**
	 * @associates <{ControlledObjectHandler}>
	 * @link aggregation
	 * @supplierCardinality 0..* 
	 */
	protected List controlledObjectHandlers = new ArrayList();

	/**
	 * @associates <{EventProviderHandler}>
	 * @link aggregation
	 * @supplierCardinality 0..* 
	 */
	protected List eventProviderHandlers = new ArrayList();

    /**
     * @label root 
     */
	protected StateMachine root;

    protected Model(String name) {
        super(name);
    }
    
    protected void addStateMachine(StateMachine machine) {
        stateMachines.add(machine);
        
        firePropertyChange(CHILDREN_PROPERTY, null, machine);
    }
    
    protected void addControlledObjectHandler(ControlledObjectHandler object) {
        controlledObjectHandlers.add(object);

        firePropertyChange(CHILDREN_PROPERTY, null, object);
    }

    protected void addEventProviderHandler(EventProviderHandler object) {
        eventProviderHandlers.add(object);

        firePropertyChange(CHILDREN_PROPERTY, null, object);
    }

    public void setRootStateMachine(StateMachine root) {
        StateMachine oldRoot = this.root;
        this.root = root;
        
        firePropertyChange(ROOT_STATE_MACHINE, oldRoot, root);
    }
    
    protected ModelElement find(Collection c, String name) {
	    if (name == null) {
	        throw new IllegalArgumentException("null name");
	    }
	    
		for (Iterator i = c.iterator(); i.hasNext(); ) {
		    ModelElement me = (ModelElement)i.next();
		    
		    if (name.equals(me.getName())) {
		        return me;
		    }
		}
		
		return null;
    }

    /**
     * Returns first found controlled object handler with given name
     * 
     * @param name
     * @return
     */
	public ControlledObjectHandler getControlledObjectHandler(String name) {
	    return (ControlledObjectHandler)find(controlledObjectHandlers, name);
	}

    /**
     * Returns first found state machine with given name
     * 
     * @param name
     * @return
     */
	public StateMachine getStateMachine(String name) {
	    return (StateMachine)find(stateMachines, name);
	}

    /**
     * Returns first found event provider handler with given name
     * 
     * @param name
     * @return
     */
	public EventProviderHandler getEventProviderHandler(String name) {
	    return (EventProviderHandler)find(eventProviderHandlers, name);
	}
	
    public List getControlledObjectHandlers() {
        return controlledObjectHandlers;
    }
    
    public List getEventProviderHandlers() {
        return eventProviderHandlers;
    }

    public StateMachine getRootStateMachine() {
        return root;
    }

    public List getStateMachines() {
        return stateMachines;
    }
    
    protected void removeControlledObjectHandler(ControlledObjectHandler co) {
        controlledObjectHandlers.remove(co);

        firePropertyChange(CHILDREN_PROPERTY, co, null);
    }

    protected void removeStateMachine(StateMachine co) {
        stateMachines.remove(co);

        firePropertyChange(CHILDREN_PROPERTY, co, null);
    }

    protected void removeEventProviderHandler(EventProviderHandler ep) {
        eventProviderHandlers.remove(ep);

        firePropertyChange(CHILDREN_PROPERTY, ep, null);
    }

	public boolean equals(Object o) {
		return o instanceof Model;
	}

	public int hashCode() {
		return 0;
	}

    // factory methods
    
    /**
     * Creates new model
     * @param name model name
     */
    public static Model createModel(String name) {
        return new Model(name);
    }    
    
    /**
     * Creates state machine with given name and adds it to model.
     *
     * @param name state machine name
     * @return newly created state machine
     */
    public StateMachine createStateMachine(String name) {
        StateMachine sm = new StateMachine(name);
        this.addStateMachine(sm);
        
        return sm;
    }
    
    /**
     * Creates controlled object handler and adds it to model. Doesn't try to
     * instanciate implementation class. So may be called even if there's no
     * impl class yet (during design phase, for example).
     *
     * @param name controlled object handler name
     * @param implClassName controlled object implementation class name
     * @return newly created object
     */
    public ControlledObjectHandler createControlledObjectHandler(String name, String implClassName) {
        ControlledObjectHandler co = new ControlledObjectHandler(name, implClassName);
        this.addControlledObjectHandler(co);
        
        return co;
    }
    
    /**
     * Creates event provider handler and adds it to model.
     * 
     * @param name provider name
     * @param implClassName fully quialified implementation class name
     * @return
     */
    public EventProviderHandler createEventProviderHandler(String name, String implClassName) {
        EventProviderHandler ep = new EventProviderHandler(name, implClassName);
        this.addEventProviderHandler(ep);
        
        return ep;
    }

}
