/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;

import com.evelopers.unimod.core.ModelElement;
import com.evelopers.unimod.runtime.context.Parameter;
import com.evelopers.unimod.runtime.context.StateMachineContext;

/**
 * Event abstraction. Event has no originator or target. It has only
 * parameters. Runtime environment is responsible for event queueing and
 * dispatching.
 *
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
public class Event extends ModelElement {
	
    public static final String ANY_EVENT_NAME = "*";
    
    public static final Event ANY = new Event(ANY_EVENT_NAME);
    
    public static final String NO_EVENT_NAME = "";
    
    public static final Event NO_EVENT = new Event(NO_EVENT_NAME);
    
	/** Array of event parameters 
	 * @link aggregation
     */
	protected Parameter[] parameters = null;

	/**
	 * Creates event with given name.
	 *
	 * @param name event name
	 */
	public Event(String name) {
		super(name);
	}

	/**
	 * Creates event with given name and array of parameters. this constructor
	 * must be used during runtime only.
	 *
	 * @param name event name
	 * @param parameters event parameters
	 */
	public Event(String name, Parameter[] parameters) {
		this(name);

		this.parameters = parameters;
	}

	public Event(String name, Parameter parameter) {
		this(name);

		this.parameters = new Parameter[] {parameter};
	}

	public Event(String name, Parameter p1, Parameter p2) {
		this(name);

		this.parameters = new Parameter[] {p1, p2};
	}

	public Event(String name, Parameter p1, Parameter p2, Parameter p3) {
		this(name);

		this.parameters = new Parameter[] {p1, p2, p3};
	}
	
	public Parameter[] getParameters() {
		return parameters;
	}
	
	public void setParameters(Parameter[] p) {
	    parameters = p;
	}

	/**
	 * Adds all parameters to given context.
	 * Usually, target context is EventContext.
	 * 
	 * @param context context to add parameters to
	 */
	public void addParametersToContext(StateMachineContext.Context context) {
		if (parameters == null) {
			return;
		}
		
		for (int i = 0; i < parameters.length; i++) {
			parameters[i].addToContext(context);
		}
	}

	/**
	 * Sets name
	 * @param name - new name
	 */
	public void setName(String name) {
		throw new UnsupportedOperationException("You can't change name for event");
	}

	/**
	 * Compares events.
	 *
	 * @param o another event
	 *
	 * @return boolean
	 */
	public boolean equals(Object o) {
		if (!(o instanceof Event)) {
			return false;
		} else {
			return getName().equals(((Event)o).getName());
		}
	}

	/**
	 * Returns hashcode of event name.
	 *
	 * @return hashcode
	 */
	public int hashCode() {
		return getName().hashCode();
	}
	
	public String toString() {
	    return getName();
	}
}
